#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import division
import sys
import os
import argparse
import datetime
from nile.api.v1 import clusters
from yql.api.v1.client import YqlClient
from videolog_common import (
    apply_replacements,
    date_range,
    get_cluster,
    get_driver,
    get_stat_headers,
    get_dates_from_stat,
    get_date,
    YqlRunner,
)

TITLE = "[CPMADV-9] Phrases Stat | YQL"

STAB_PATH = {
    "path": "analytics/videolog/strm-stats/strm_cube_2/stability/stability_common.sql"
}


def get_max_date(yt):
    res = get_date(
        yt.get_attribute(
            "//home/yabs/stat/DirectPhraseStatV2", "last_sync_time_bs-chevent-log"
        ).split("T")[0]
    ) - datetime.timedelta(days=1)
    return res


def process_date(date, cluster, args):
    proxy = os.environ["YT_PROXY"]
    yc = YqlClient(db=proxy.lower(), token=os.environ["YQL_TOKEN"])
    yr = YqlRunner(client=yc, title=TITLE)
    mon1 = date.replace(day=1)
    with open("phrase_stat_stub.sql", "rb") as f:
        query = f.read().decode("utf8")
    query = apply_replacements(
        query, {"@[pool]": args["pool"], "@[date]": str(date), "@[mon1]": str(mon1)}
    )
    yr.run(query, wait=True, attachments=[STAB_PATH])


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("--pool", default="robot-mma-nirvana")
    parser.add_argument("--title", default=TITLE)
    parser.add_argument("--from")
    parser.add_argument("--redo", action="store_true")
    parser.add_argument("--to")
    args = vars(parser.parse_args())

    from_ = args.get("from")
    to_ = args.get("to")

    cluster = get_cluster(clusters, args)
    yt = get_driver(cluster).client

    if from_ and to_:
        dates = date_range(from_, to_)
    else:
        last_date = max(
            get_date(x)
            for x in yt.list("//home/videoquality/vh_analytics/phrases_stat")
        )
        print("last date from phrases stat: {}".format(last_date))
        if last_date >= (datetime.date.today() - datetime.timedelta(days=1)):
            print("no dates to process")
            sys.exit(0)
        last_available_date_phrases = get_max_date(yt)
        last_available_date_campaigns = max(
            get_date(x)
            for x in yt.list("//home/videoquality/vh_analytics/underimpressions_tmp")
        )
        last_available_date = min(
            last_available_date_phrases, last_available_date_campaigns
        )
        if last_available_date > last_date:
            dates = sorted(
                date_range(last_date + datetime.timedelta(days=1), last_available_date),
                reverse=True
            )
        else:
            dates = []

    if dates:
        for date in dates:
            process_date(date, cluster, args)
    else:
        print("no dates to process")


if __name__ == "__main__":
    main()
