#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import division
import sys
import os
import argparse
import datetime
from nile.api.v1 import clusters
from yql.api.v1.client import YqlClient
from videolog_common import (
    apply_replacements,
    date_range,
    get_cluster,
    get_driver,
    get_stat_headers,
    get_dates_from_stat,
    get_date,
    YqlRunner,
)

TITLE = "[CPMADV-9] Targeting | YQL"
AUX_TITLE = "[CPMADV-9] Targeting Max Date | YQL"


def process_date(date, cluster, args):
    proxy = os.environ["YT_PROXY"]
    yc = YqlClient(db=proxy.lower(), token=os.environ["YQL_TOKEN"])
    yr = YqlRunner(client=yc, title=TITLE)
    with open("targeting_stub.sql", "rb") as f:
        query = f.read().decode("utf8")
    query = apply_replacements(query, {"@[pool]": args["pool"], "@[date]": str(date)})
    yr.run(query, wait=True)


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("--pool", default="robot-mma-nirvana")
    parser.add_argument("--title", default=TITLE)
    parser.add_argument("--from")
    parser.add_argument("--redo", action="store_true")
    parser.add_argument("--to")
    args = vars(parser.parse_args())

    from_ = args.get("from")
    to_ = args.get("to")

    cluster = get_cluster(clusters, args)
    yt = get_driver(cluster).client

    if from_ and to_:
        dates = date_range(from_, to_)
    else:
        last_date = max(
            get_date(x) for x in yt.list("//home/videoquality/vh_analytics/targeting") if get_date(x)
        )
        yesterday = datetime.date.today() - datetime.timedelta(days=1)
        if last_date >= yesterday:
            print("no dates to process")
            sys.exit(0)
        last_available_date = min(
            max(get_date(x) for x in yt.list("//home/direct/db-archive") if get_date(x)), yesterday
        )
        if last_available_date > last_date:
            dates = sorted(
                date_range(last_date + datetime.timedelta(days=1), last_available_date),
                reverse=True
            )
        else:
            dates = []

    if dates:
        print("dates to process: {}".format(dates))
        for date in dates:
            print("processing {}".format(date))
            process_date(date, cluster, args)
    else:
        print("no dates to process")


if __name__ == "__main__":
    main()
