use hahn;
pragma AnsiInForEmptyOrNullableItemsCollections;

$input_table = "logs/redir-log/1d/" || $date;

$getPlatform = ($ua) -> {
    $parsed = UserAgent::Parse($ua);
    return case
    when $parsed.isTV then "tv"
    when $parsed.OSFamily in ("Windows", "iOS", "Android") then $parsed.OSFamily
    when $parsed.isMobile then "other_mobile"
    else "other_desktop"
    end;
};

$map = (
    select
        `dict`["yandexuid"] as yandexuid,
        `dict`["source_url"] as source_url,
        cast(`dict`["client_timestamp"] as Uint64) as client_timestamp,
        `dict`["path"] as path,
        `dict`["quality"] as quality,
        cast(`dict`["mute"] as Uint8) as muted,
        $getPlatform(`dict`["user_agent"]) as platform
    from $input_table
    where `dict`["source_url"] like "%youtube%"
);

insert into `home/videoquality/vh_analytics/mma-4921/map` WITH TRUNCATE 
select * from $map;

$input_type = Struct<
    yandexuid: String?,
    platform: String?,
    source_url: String?,
    client_timestamp: Uint64?,
    muted: Uint8?,
    path: String?,
    quality: String?
>;

$output_type = Struct<
    source_url: String,
    platform: String,
    muted_cat: String,
    quality: String,
    view_time: Double,
    stalled_time: Double,
    stalled_count: Double
>;

$reducer = Python::reducer(
    Callable<(Tuple<String?, String?>, Stream<$input_type>)->Stream<$output_type>>, FileContent("reducer.py")
);

$reduce = (
    reduce $map
    presort client_timestamp
    on yandexuid, source_url
    using $reducer(TableRow())
);

$nwrap = ($x) -> (IF(Math::IsNaN($x) or Math::IsInf($x), null, $x));

$add_permins = (
    select r.*,
        $nwrap(stalled_count / (view_time / 60.0)) as stalled_count_per_min,
        $nwrap(stalled_time / (view_time / 60.0)) as stalled_time_per_min,
    from $reduce as r
);

insert into `home/videoquality/vh_analytics/mma-4921/reduce` WITH TRUNCATE 
select * from $add_permins;

$totalize = ($row) -> {
    $result = [$row];
    $add = ListMap($result, ($x)->(
        AddMember(RemoveMember($x, "platform"), "platform", "_total_")
    ));
    $result = ListUnionAll($result, $add);
    $add = ListMap($result, ($x)->(
        AddMember(RemoveMember($x, "muted_cat"), "muted_cat", "_total_")
    ));
    $result = ListUnionAll($result, $add);
    $add = ListMap($result, ($x)->(
        AddMember(RemoveMember($x, "quality"), "quality", "_total_")
    ));
    $result = ListUnionAll($result, $add);
    return $result
};

$totalized = process $add_permins using $totalize(TableRow());

$grouped = (
    select
        $date as fielddate,
        unwrap(platform) as platform,
        unwrap(muted_cat) as muted_cat,
        unwrap(quality) as quality,
        count(*) as `count`,
        sum(view_time) / 60.0 as watched_time_mins,
        sum(stalled_time) / (sum(view_time) / 60.0) as stalled_time_per_min,
        sum(stalled_count) / (sum(view_time) / 60.0) as stalled_count_per_min,
        percentile(stalled_time_per_min, 0.25) as stalled_time_per_min_p25,
        percentile(stalled_time_per_min, 0.50) as stalled_time_per_min_p50,
        percentile(stalled_time_per_min, 0.75) as stalled_time_per_min_p75,
        percentile(stalled_time_per_min, 0.95) as stalled_time_per_min_p95,
        percentile(stalled_count_per_min, 0.25) as stalled_count_per_min_p25,
        percentile(stalled_count_per_min, 0.50) as stalled_count_per_min_p50,
        percentile(stalled_count_per_min, 0.75) as stalled_count_per_min_p75,
        percentile(stalled_count_per_min, 0.95) as stalled_count_per_min_p95
    from $totalized
    group by platform,
        muted_cat,
        quality
);

insert into `home/videoquality/vh_analytics/mma-4921/report` WITH TRUNCATE 
select * from $grouped;

upsert into stat.`Video/Others/Strm/Stability/MMA-4921-youtube-quality/daily`
select * from $grouped;
