#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import division
import sys
import os
import argparse
import datetime
from nile.api.v1 import clusters
from yql.api.v1.client import YqlClient
from videolog_common import (
    apply_replacements,
    date_range,
    get_cluster,
    get_driver,
    get_stat_headers,
    get_dates_from_stat,
    get_date,
    StatPusher,
    YqlRunner,
)

log_roots = [
    "//home/direct/logs/ppclog_api",
    "//cooked_logs/bs-chevent-cooked-log/1d"
]
TITLE = "[MMA-5132] K50 Report | YQL"


def process_dates(dates, cluster, args):
    proxy = os.environ["YT_PROXY"]
    output_table = "{}/stat/{}_{}".format(
        args["root"], dates[0], dates[-1]
    )
    yc = YqlClient(db=proxy.lower(), token=os.environ["YQL_TOKEN"])
    yr = YqlRunner(client=yc, title=TITLE)
    with open("mma_5132_k50_stub.sql", "rb") as f:
        query = f.read().decode("utf8")
    query = apply_replacements(
        query,
        {
            "@[pool]": args["pool"],
            "@[output_table]": output_table,
            # "@[tmp_table]": tmp_table,
            "@[date_from]": str(dates[0]),
            "@[date_to]": str(dates[-1]),
        },
    )
    yr.run(
        query,
        wait=True,
        attachments=[
            {"path": "analytics/videolog/strm-stats/strm_cube_2/stability/stability_common.sql"}
        ]
    )
    if not args.get("debug"):
        sp = StatPusher(cluster, report=args["report"])
        print("pushing to {}".format(args["report"]))
        kwargs = {"scale": "weekly"}
        if args.get("redo"):
            kwargs["replace_mask"] = "fielddate"
        sp.push(output_table, **kwargs)


def get_available_dates(yt):
    result = None
    for log in log_roots:
        set_ = {
            get_date(x) for x in yt.search(log) if get_date(x)
        }
        if result is not None:
            result &= set_
        else:
            result = set_
    return result


def next_wd(date, wd=0):
    wd_c = date.weekday()
    if wd_c == wd:
        return date
    if wd_c < wd:
        return date + datetime.timedelta(days=wd - wd_c)
    return date + datetime.timedelta(days=7 - wd_c + wd)


def make_date_chunks(dates):
    min_date = min(dates)
    max_date = max(dates)
    result = []
    mon = next_wd(min_date, 0)
    while mon <= max_date:
        sun = next_wd(mon, 6)
        if sun <= max_date:
            result.append((mon, sun))
        mon += datetime.timedelta(days=7)
    return result[::-1]


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("--pool", default="robot-mma-nirvana")
    parser.add_argument(
        "--report", default="Video/Others/MMA-5132-K50"
    )
    parser.add_argument("--root", default="//home/videoquality/vh_analytics/MMA-5132-K50")
    parser.add_argument("--from")
    parser.add_argument("--redo", action="store_true")
    # parser.add_argument("--notmp", action="store_true")
    parser.add_argument("--to")
    args = vars(parser.parse_args())

    from_ = args.get("from")
    to_ = args.get("to")

    cluster = get_cluster(clusters, args)
    yt = get_driver(cluster).client

    if from_ and to_:
        date_chunks = make_date_chunks(sorted(date_range(from_, to_)))
    else:
        dates_from_stat = get_dates_from_stat(
            report=args["report"], scale="w", headers=get_stat_headers()
        )
        last_date = dates_from_stat[-1]
        print("last date: {}".format(last_date))
        available_dates = get_available_dates(yt)
        last_available_date = max(available_dates)
        print("last available date: {}".format(last_available_date))
        if last_available_date > last_date:
            dates = sorted(
                date_range(
                    last_date + datetime.timedelta(days=1), last_available_date
                )
            )
            date_chunks = make_date_chunks(dates)
        else:
            dates = []
            date_chunks = []

    if date_chunks:
        for chunk in date_chunks:
            print("processing {}".format(chunk))
            process_dates(chunk, cluster, args)
    else:
        print("no dates to process")


if __name__ == "__main__":
    main()
