#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import division
import sys
import os
import argparse
import datetime
import yt.wrapper as yt
from yql.api.v1.client import YqlClient
from videolog_common import (
    apply_replacements,
    date_range,
    get_cluster,
    get_driver,
    get_stat_headers,
    get_dates_from_stat,
    get_date,
    StatPusher,
    YqlRunner,
)

log_roots = [
    "//statbox/cube/daily/product_money/v1",
]
TITLE = "[MMA-5192] CPM Dash | YQL"


def process_dates(dates, args):
    proxy = os.environ["YT_PROXY"]
    output_table = "{}/{}_{}".format(
        args["root"], dates[0], dates[-1]
    )
    yc = YqlClient(db=proxy.lower(), token=os.environ["YQL_TOKEN"])
    yr = YqlRunner(client=yc, title=TITLE)
    with open("mma_5192.sql", "rb") as f:
        query = f.read().decode("utf8")
    query = apply_replacements(
        query,
        {
            "@[pool]": args["pool"],
            "@[output_table]": output_table,
            # "@[tmp_table]": tmp_table,
            "@[date_from]": str(dates[0]),
            "@[date_to]": str(dates[-1]),
        },
    )
    yr.run(
        query,
        wait=True,
    )


def get_available_dates(yt):
    result = None
    for log in log_roots:
        set_ = {
            get_date(x) for x in yt.search(log) if get_date(x)
        }
        if result is not None:
            result &= set_
        else:
            result = set_
    return result


def next_wd(date, wd=0):
    wd_c = date.weekday()
    if wd_c == wd:
        return date
    if wd_c < wd:
        return date + datetime.timedelta(days=wd - wd_c)
    return date + datetime.timedelta(days=7 - wd_c + wd)


def make_date_chunks(dates):
    result = []
    dates = date_range(min(dates), max(dates))[::-1]
    while dates:
        chunk, dates = dates[:7], dates[7:]
        result.append(sorted(chunk))
    return result


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("--pool", default="robot-mma-nirvana")
    parser.add_argument(
        "--report", default="Video/Others/Strm/MMA-5192-cpm-dash-dwh"
    )
    parser.add_argument("--root", default="//home/videoquality/vh_analytics/mma_5192_cpm_ad_dash")
    parser.add_argument("--from")
    parser.add_argument("--redo", action="store_true")
    # parser.add_argument("--notmp", action="store_true")
    parser.add_argument("--to")
    args = vars(parser.parse_args())

    from_ = args.get("from")
    to_ = args.get("to")

    if from_ and to_:
        date_chunks = make_date_chunks(sorted(date_range(from_, to_)))
    else:
        dates_from_stat = get_dates_from_stat(
            report=args["report"], headers=get_stat_headers()
        )
        last_date = dates_from_stat[-1]
        print("last date: {}".format(last_date))
        available_dates = get_available_dates(yt)
        last_available_date = max(available_dates)
        print("last available date: {}".format(last_available_date))
        if last_available_date > last_date:
            dates = sorted(
                date_range(
                    last_date + datetime.timedelta(days=1), last_available_date
                )
            )
            date_chunks = make_date_chunks(dates)
        else:
            dates = []
            date_chunks = []

    if date_chunks:
        for chunk in date_chunks:
            print("processing {}".format(chunk))
            process_dates(chunk, args)
    else:
        print("no dates to process")


if __name__ == "__main__":
    main()
