#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import division
import sys
import os
import argparse
import datetime
from nile.api.v1 import clusters
from yql.api.v1.client import YqlClient
from videolog_common import (
    apply_replacements,
    date_range,
    get_cluster,
    get_driver,
    get_stat_headers,
    get_dates_from_stat,
    get_date,
    StatPusher,
    YqlRunner,
)

TITLE = "[MMA-5201] Underimpressions Dash | YQL"
AUX_TITLE = "[MMA-5201] Check max date | YQL"

log_roots = [
    "//home/videoquality/vh_analytics/underimpressions_tmp",
    "//home/videoquality/vh_analytics/underimpressions_tmp_awaps",
]


def get_max_date(args):
    proxy = os.environ["YT_PROXY"]
    yc = YqlClient(db=proxy.lower(), token=os.environ["YQL_TOKEN"])
    yr = YqlRunner(client=yc, title=AUX_TITLE)
    req = yr.run(MAX_DATE_STUB)
    res = get_date(yr.get_results(req)[0]["max_all"].decode("utf8"))
    return res


def process_date(date, cluster, args):
    proxy = os.environ["YT_PROXY"]
    output_table = "{}/{}".format(args["root"], date)
    yc = YqlClient(db=proxy.lower(), token=os.environ["YQL_TOKEN"])
    yr = YqlRunner(client=yc, title=TITLE)
    with open("mma_5201.sql", "rb") as f:
        query = f.read().decode("utf8")
    query = apply_replacements(
        query,
        {
            "@[pool]": args["pool"],
            "@[output_table]": output_table,
            "@[date]": str(date),
            "@[mon1]": str(date.replace(day=1)),
            "@[nextmon1]": str(
                (date.replace(day=1) + datetime.timedelta(days=32)).replace(
                    day=1
                )
            ),
        },
    )
    yr.run(
        query,
        wait=True,
        attachments=[
            {
                "path": "analytics/videolog/strm-stats/media_ads/mma_5201_underimpression_dash/mma_5201_subqueries.sql"
            }
        ],
    )
    if not args.get("debug"):
        sp = StatPusher(cluster, report=args["report"])
        print("pushing to {}".format(args["report"]))
        kwargs = {}
        if args.get("redo"):
            kwargs["replace_mask"] = "fielddate"
        sp.push(output_table, **kwargs)


def get_available_dates(yt):
    result = None
    for log in log_roots:
        set_ = {get_date(x) for x in yt.search(log) if get_date(x)}
        if result is not None:
            result &= set_
        else:
            result = set_
    return result


def next_wd(date, wd=0):
    wd_c = date.weekday()
    if wd_c == wd:
        return date
    if wd_c < wd:
        return date + datetime.timedelta(days=wd - wd_c)
    return date + datetime.timedelta(days=7 - wd_c + wd)


def make_date_chunks(dates):
    min_date = min(dates)
    max_date = max(dates)
    result = []
    mon = next_wd(min_date, 0)
    while mon <= max_date:
        sun = next_wd(mon, 6)
        if sun <= max_date:
            result.append((mon, sun))
        mon += datetime.timedelta(days=7)
    return result[::-1]


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("--pool", default="robot-mma-nirvana")
    parser.add_argument(
        "--report", default="Video/Others/MMA-5201-underimpressions"
    )
    parser.add_argument(
        "--root",
        default="//home/videoquality/vh_analytics/MMA-5201-underimpressions",
    )
    parser.add_argument("--title", default=TITLE)
    parser.add_argument("--from")
    parser.add_argument("--redo", action="store_true")
    parser.add_argument("--to")
    args = vars(parser.parse_args())

    from_ = args.get("from")
    to_ = args.get("to")

    cluster = get_cluster(clusters, args)
    yt = get_driver(cluster).client

    if from_ and to_:
        dates = date_range(from_, to_)
    else:
        last_available_date = max(get_available_dates(yt))
        last_stat_date = get_dates_from_stat(
            get_stat_headers(), args["report"]
        )[-1]
        if last_available_date > last_stat_date:
            dates = sorted(
                date_range(
                    last_stat_date + datetime.timedelta(days=1),
                    last_available_date,
                )
            )
        else:
            dates = []

    if dates:
        for date in dates:
            process_date(date, cluster, args)
    else:
        print("no dates to process")


if __name__ == "__main__":
    main()
