#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import division
import sys
import os
import argparse
import datetime
from nile.api.v1 import clusters
from yql.api.v1.client import YqlClient
from videolog_common import (
    apply_replacements,
    date_range,
    get_cluster,
    get_driver,
    get_stat_headers,
    get_dates_from_stat,
    get_date,
    StatPusher,
    YqlRunner,
)

TITLE = "[MMA-5254] Bstags Base Updater | YQL"
AUX_TITLE = "[MMA-5254] Bstags Base Updater_Check max date | YQL"

MAX_DATE_STUB = """
use hahn;

select last_date from `//home/videoquality/vh_analytics/mma-5254/base_last_date` limit 1;
"""
LOG_ROOTS = ["//logs/bs-hit-log/1d"]


def get_max_date(args):
    proxy = os.environ["YT_PROXY"]
    yc = YqlClient(db=proxy.lower(), token=os.environ["YQL_TOKEN"])
    yr = YqlRunner(client=yc, title=AUX_TITLE)
    req = yr.run(MAX_DATE_STUB)
    res = get_date(yr.get_results(req)[0]["last_date"].decode("utf8"))
    return res


def process_dates(dates, cluster, args):
    proxy = os.environ["YT_PROXY"]
    yc = YqlClient(db=proxy.lower(), token=os.environ["YQL_TOKEN"])
    yr = YqlRunner(client=yc, title=TITLE)
    with open("mma_5254_bstags_base_query.sql", "rb") as f:
        query = f.read().decode("utf8")
    query = apply_replacements(
        query,
        {
            "@[pool]": args["pool"],
            "@[date_from]": str(min(dates)),
            "@[date_to]": str(max(dates)),
        },
    )
    yr.run(
        query,
        wait=True,
    )


def get_available_dates(yt):
    result = None
    for log in LOG_ROOTS:
        set_ = {
            get_date(x)
            for x in yt.search(log, node_type="table")
            if get_date(x.split("/")[-1])
        }
        if result is not None:
            result &= set_
        else:
            result = set_
    return result


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("--pool", default="robot-mma-nirvana")
    parser.add_argument("--title", default=TITLE)
    parser.add_argument("--from")
    parser.add_argument("--redo", action="store_true")
    parser.add_argument("--to")
    args = vars(parser.parse_args())

    from_ = args.get("from")
    to_ = args.get("to")

    cluster = get_cluster(clusters, args)
    yt = get_driver(cluster).client

    if from_ and to_:
        dates = date_range(from_, to_)
    else:
        last_base_date = get_max_date(args)
        last_available_date = max(get_available_dates(yt))
        if last_available_date > last_base_date:
            dates = sorted(
                date_range(
                    last_base_date + datetime.timedelta(days=1),
                    last_available_date,
                )
            )
        else:
            dates = []

    if dates:
        process_dates(dates, cluster, args)
    else:
        print("no dates to process")


if __name__ == "__main__":
    main()
