#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import division
import sys
import os
import argparse
import datetime
from nile.api.v1 import clusters
from yql.api.v1.client import YqlClient
from videolog_common import (
    apply_replacements,
    date_range,
    get_cluster,
    get_driver,
    get_stat_headers,
    get_dates_from_stat,
    get_date,
    StatPusher,
    YqlRunner,
)

TITLE = "[MMA-5254] Bstags Lag Report | YQL"

MAX_DATE_STUB = """
use hahn;

SELECT last_date from `//home/videoquality/vh_analytics/mma-5254/base_last_date` limit 1;
"""


def process_date(date, cluster, args):
    proxy = os.environ["YT_PROXY"]
    output_table = "{}/{}".format(
        args["root"], date
    )
    yc = YqlClient(db=proxy.lower(), token=os.environ["YQL_TOKEN"])
    yr = YqlRunner(client=yc, title=TITLE)
    week_ago = date - datetime.timedelta(days=7)
    with open("mma_5254_report.sql", "rb") as f:
        query = f.read().decode("utf8")
    query = apply_replacements(
        query,
        {
            "@[pool]": args["pool"],
            "@[output_table]": output_table,
            # "@[tmp_table]": tmp_table,
            "@[date]": str(date),
            "@[week_ago]": str(week_ago),
        },
    )
    yr.run(
        query,
        wait=True,
    )
    if not args.get("debug"):
        sp = StatPusher(cluster, report=args["report"])
        print("pushing to {}".format(args["report"]))
        kwargs = {}
        if args.get("redo"):
            kwargs["replace_mask"] = "fielddate"
        sp.push(output_table, **kwargs)


def get_last_date(yt):
    proxy = os.environ["YT_PROXY"]
    yc = YqlClient(db=proxy.lower(), token=os.environ["YQL_TOKEN"])
    yr = YqlRunner(client=yc, title="Last Date " + TITLE)
    req = yr.run(MAX_DATE_STUB)
    res = get_date(yr.get_results(req)[0]["last_date"].decode("utf8"))
    return res


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("--pool", default="robot-mma-nirvana")
    parser.add_argument(
        "--report", default="Video/Others/Strm/MMA-5254-BsTagsReachLag/"
    )
    parser.add_argument("--root", default="//home/videoquality/vh_analytics/mma_5254_report")
    parser.add_argument("--from")
    parser.add_argument("--redo", action="store_true")
    # parser.add_argument("--notmp", action="store_true")
    parser.add_argument("--to")
    args = vars(parser.parse_args())

    from_ = args.get("from")
    to_ = args.get("to")

    cluster = get_cluster(clusters, args)
    yt = get_driver(cluster).client

    if from_ and to_:
        dates = sorted(date_range(from_, to_), reverse=True)
    else:
        dates_from_stat = get_dates_from_stat(
            report=args["report"], headers=get_stat_headers()
        )
        last_date = dates_from_stat[-1]
        print("last date: {}".format(last_date))
        last_available_date = get_last_date(yt)
        print("last available date: {}".format(last_available_date))
        if last_available_date > last_date:
            dates = sorted(
                date_range(
                    last_date + datetime.timedelta(days=1), last_available_date
                ), reverse=True
            )
        else:
            dates = []

    if dates:
        for date in dates:
            print("processing {}".format(date))
            process_date(date, cluster, args)
    else:
        print("no dates to process")


if __name__ == "__main__":
    main()
