#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import division
import sys
import os
import argparse
import datetime
from nile.api.v1 import clusters
from yql.api.v1.client import YqlClient
from videolog_common import (
    apply_replacements,
    date_range,
    get_cluster,
    get_driver,
    get_stat_headers,
    get_dates_from_stat,
    get_date,
    StatPusher,
    YqlRunner,
)

log_roots = [
    "//cooked_logs/bs-dsp-cooked-log/1d",
    "//statbox/crypta-yandexuid-profiles-log"
]
TITLE = "[MMA-4767] Ad Squeeze Regular | YQL"


def process_dates(dates, args):
    proxy = os.environ["YT_PROXY"]
    yc = YqlClient(db=proxy.lower(), token=os.environ["YQL_TOKEN"])
    yr = YqlRunner(client=yc, title=args["title"])
    with open(args["query_stub"], "rb") as f:
        query = f.read().decode("utf8")
    query = apply_replacements(
        query,
        {
            "@[pool]": args["pool"],
            "@[output_root]": args["root"],
            "@[date_from]": str(dates[0]),
            "@[date_to]": str(dates[-1]),
        },
    )
    yr.run(
        query,
        wait=True,
        attachments=[
            {
                "path": "arcadia/analytics/videolog/strm-stats/strm_cube_2/stability/stability_common.sql"
            }
        ]
    )


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("--pool", default="robot-mma-nirvana")
    parser.add_argument("--root", default="//home/videoquality/vh_analytics/ad_squeeze")
    parser.add_argument("--query_stub", default="ad_squeeze_stub.sql")
    parser.add_argument("--title", default=TITLE)
    parser.add_argument("--from")
    parser.add_argument("--redo", action="store_true")
    parser.add_argument("--to")
    args = vars(parser.parse_args())

    from_ = args.get("from")
    to_ = args.get("to")

    cluster = get_cluster(clusters, args)
    yt = get_driver(cluster).client

    if from_ and to_:
        dates = sorted(date_range(from_, to_))
    else:
        dates_set = None
        for log_root in log_roots:
            log_dates = {
                get_date(x)
                for x in yt.search(
                    root=log_root,
                    node_type="table",
                    path_filter=lambda x: bool(get_date(x)),
                )
            }
            if dates_set is None:
                dates_set = log_dates
            else:
                dates_set &= log_dates
        last_date = max(
            get_date(x)
            for x in yt.search(
                root=args["root"],
                node_type="table",
                path_filter=lambda x: bool(get_date(x)),
            )
        )
        print("last date: {}".format(last_date))
        last_available_date = max(dates_set)
        print("last available date: {}".format(last_available_date))
        if last_available_date > last_date:
            dates = sorted(
                date_range(
                    last_date + datetime.timedelta(days=1), last_available_date
                )
            )
        else:
            dates = []

    if dates:
        print("processing {}".format(dates))
        process_dates(dates, args)
    else:
        print("no dates to process")


if __name__ == "__main__":
    main()
