#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import unicode_literals
from __future__ import division
import sys
import os
import codecs
import argparse
from nile.api.v1 import clusters
from yql.api.v1.client import YqlClient
from videolog_common import (
    get_cluster,
    get_driver,
    YqlRunner,
    apply_replacements,
)


ROOT = "//home/videoquality/vh_analytics/mma-3568-tele2"
ORGVISITS_ROOT = (
    "//home/user_identification/orgvisits/prod/state/geologs/current"
)
DEFAULT_CONDITION = "where city == 'Екатеринбург' and isp == 'tele2 russia'"
DEFAULT_HAVING = "having count(*) >= 100 and count(distinct yandexuid) >= 50"


def run_query(yr, query_file, attachments=[], wait=True, replacements={}):
    with open(query_file, "r") as f:
        query = f.read()
    query = apply_replacements(query, replacements)
    return yr.run(query, wait=wait, attachments=attachments)


def wrap_utf8(s):
    if isinstance(s, bytes):
        return s.decode("utf8")
    return s


def make_location_data_processed(args, yt):
    with open("location_data.sql") as f:
        loc_data = f.read()
    if yt.exists("{}/{}".format(ORGVISITS_ROOT, args["from"])) and yt.exists(
        "{}/{}".format(ORGVISITS_ROOT, args["to"])
    ):
        args["orgvisits"] = True
    else:
        args["mobmetrika"] = True
    if args.get("orgvisits"):
        loc_data = apply_replacements(
            loc_data, {"/*--[orgvisits]": "", "--[orgvisits]*/": ""}
        )
    if args.get("mobmetrika"):
        loc_data = apply_replacements(
            loc_data, {"/*--[mobmetrika]": "", "--[mobmetrika]*/": ""}
        )
    with open("location_data_processed.sql", "w") as f:
        f.write(loc_data)


def process_dates(args):
    cluster = get_cluster(clusters, {})
    yt = get_driver(cluster).client
    yc = YqlClient(token=os.environ["YQL_TOKEN"])
    yr = YqlRunner(yc, title="Operators Data | YQL")
    root = "{}/{}_{}".format(args.get("root") or ROOT, args["from"], args["to"])
    tmp_table = "{}/joined_sessions".format(root)
    if args["mode"] == "pre":
        make_location_data_processed(args, yt)
        if not yt.exists(tmp_table):
            print("saving temp data into {}".format(root))
            run_query(
                yr,
                "main.sql",
                attachments=[
                    "sphere.py",
                    "perf_reducer.py",
                    "location_data_processed.sql",
                    "location_join.sql",
                    "perf_data.sql",
                    "perf_join.sql",
                ],
                wait=True,
                replacements={
                    "@[date_from]": args["from"],
                    "@[date_to]": args["to"],
                    "@[pool]": args["pool"],
                    "@[output_table]": tmp_table,
                },
            )
        else:
            print("{} already exists".format(tmp_table))
    else:
        for level in ["s2_p13", "s2_p14", "s2_p15"]:
            output_table = "{}/{}_{}".format(root, args["id"], level)
            print("reducing into {}".format(output_table))
            run_query(
                yr,
                "grouping.sql",
                wait=False,
                replacements={
                    "@[output_table]": output_table,
                    "@[input_table]": tmp_table,
                    "@[grouping_field]": level,
                    "@[pool]": args["pool"],
                    "@[custom_condition]": wrap_utf8(args.get("custom_condition"))
                    or DEFAULT_CONDITION,
                    "@[having]": wrap_utf8(args.get("having"))
                    or DEFAULT_HAVING,
                },
            )


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("--pool", default="cubes-video-strm")
    parser.add_argument("--from")
    parser.add_argument("--root")
    parser.add_argument("--to")
    parser.add_argument("--id", default="tele2")
    parser.add_argument("--custom_condition")
    parser.add_argument("--having")
    parser.add_argument("--mode", default="pre")
    args = parser.parse_args()

    args = vars(args)

    process_dates(args)


if __name__ == "__main__":
    main()
