#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import unicode_literals
from __future__ import division
import os
import argparse
import toml
import tempfile
import subprocess
import shutil
import re
import datetime
import tqdm
from collections import defaultdict
import numpy as np
import yt.wrapper as yt
from yt import yson
from de2 import CsvExporter, KmzExporter, BeelineCsvExporter, KmlExporter
from de2.kml import get_coordinates_raw

# from simplekml import Kml, Color


def read_yson(filepath):
    with open(filepath, "rb") as f:
        for obj in yson.load(f, yson_type="list_fragment", encoding=None):
            yield obj


def ensure_str(obj):
    if isinstance(obj, bytes):
        return obj.decode("utf8", errors="surrogateescape")
    if isinstance(obj, list):
        return [ensure_str(x) for x in list]
    if isinstance(obj, dict):
        return {ensure_str(k): ensure_str(v) for k, v in obj.items()}
    return obj


masking_dict = {
    "mts": "operator2",
    "tele2": "operator1",
    "megafon": "operator3",
    "beeline": "operator4",
    "yota": "operator5",
}

exporters_dict = {
    "csv": CsvExporter,
    "kmz": KmzExporter,
    "kml": KmlExporter,
    "beelinecsv": BeelineCsvExporter,
}


def coords_for_csv(cellid):
    coords_raw = get_coordinates_raw(cellid)
    return ";".join(
        "{},{}".format(l.lat().degrees, l.lng().degrees) for l in coords_raw
    )


def wrap_region(row):
    if "Москва" in row["region"] or "Санкт-Петербург" in row["region"]:
        row["region"] = "{}, {}, {}".format(
            row["region"], row["city"], row["town"]
        )
    else:
        row["region"] = "{}, {}".format(row["region"], row["city"])


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("--config", "-c")
    parser.add_argument("--redownload", action="store_true")
    args = vars(parser.parse_args())

    cwd = os.path.abspath(os.getcwd())
    config_abspath = os.path.abspath(args["config"])
    with open(args["config"]) as f:
        config = toml.load(f)
    wd = os.path.dirname(args["config"])
    os.chdir(wd)

    exporters = []
    for exp_format in config["export_formats"]:
        exporters.append(
            exporters_dict[exp_format](config, config_file=config_abspath)
        )

    if not os.path.isfile(config["input_filepath"]) or args["redownload"]:
        # subprocess.call(["yt", "read", "--format"])
        pass

    if "filter" in config:
        filter_func = lambda x: eval(config["filter"])
    else:
        filter_func = None

    add_coords = config.get("add_coords")

    for row_ in tqdm.tqdm(read_yson(config["input_filepath"])):
        row = ensure_str(row_)
        if (
            "operator" in config
            and row.get("operator")
            and row["operator"] != config["operator"]
        ):
            row["operator"] = masking_dict[row["operator"]]
        if add_coords:
            row["lat_lon"] = coords_for_csv(row["s2_value"])
        if "region" in row:
            wrap_region(row)
        if filter_func and not filter_func(row):
            continue
        for exporter in exporters:
            exporter.process(row)

    for exporter in exporters:
        exporter.save_data()


if __name__ == "__main__":
    main()
