#!/usr/bin/env python
# -*- coding: utf-8 -*-
import sys
import os
import argparse
import random
import string
import datetime
import boto3
import threading
from boto3.s3.transfer import TransferConfig


class Callback(object):
    def __init__(self, file_path, interactive=False):
        self.counter = 0
        self.interactive = interactive
        self._file_path = file_path
        self._size = float(os.path.getsize(file_path))
        self._seen_so_far = 0
        self._lock = threading.Lock()

    def __call__(self, bytes_amount):
        with self._lock:
            self._seen_so_far += bytes_amount
            percentage = (self._seen_so_far / self._size) * 100
            now = datetime.datetime.now()
            line = f"[{now}] {self._file_path}  {self._seen_so_far} / {self._size}  ({round(percentage, 1)}%)"
            if self.interactive:
                sys.stdout.write("\r" + line)
                sys.stdout.flush()
            elif int(percentage) % 5 == 0:
                sys.stdout.write(line + "\n\n")


def upload_file(s3, bucket, file_path, key_path, interactive=False, config=None):
    cb = Callback(interactive)
    s3.upload_file(
        file_path,
        bucket,
        key_path,
        ExtraArgs={"ACL": "public-read"},
        Config=config,
        Callback=Callback(file_path, interactive),
    )


def get_rand(n=10):
    rnd = random.SystemRandom()
    return "".join(
        random.SystemRandom().choice(
            string.ascii_uppercase + string.ascii_lowercase + string.digits
        )
        for _ in range(n)
    )


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("--bucket", "-b")
    parser.add_argument("--key-path", "-k")
    parser.add_argument("--file-path", "-f")
    parser.add_argument("--interactive", "-i", action="store_true")
    parser.add_argument("--s3-host", "-s", default="s3.mdst.yandex.net")
    parser.add_argument("--output", "-o", default="output.txt")
    parser.add_argument("--add-rand", "-a", action="store_true")
    args = parser.parse_args()

    if args.add_rand:
        sp = args.key_path.split("/")
        key_path = "/".join(sp[:-1] + [get_rand()] + [sp[-1]])
    else:
        key_path = args.key_path

    session = boto3.session.Session(
        aws_access_key_id=os.environ["ACCESS_KEY_ID"],
        aws_secret_access_key=os.environ["ACCESS_KEY"],
    )
    s3 = session.client(service_name="s3", endpoint_url=f"https://{args.s3_host}")
    config = TransferConfig(
        multipart_threshold=8388608,
        max_concurrency=10,
        multipart_chunksize=8388608,
        use_threads=True,
        max_io_queue=100,
        io_chunksize=262144,
    )

    sys.stdout.write(
        "[{}] started uploading {}...\n".format(datetime.datetime.now(), args.key_path)
    )
    upload_file(
        s3=s3,
        bucket=args.bucket,
        file_path=args.file_path,
        key_path=key_path,
        interactive=args.interactive,
        config=config,
    )
    filepath = "https://{host}/{bucket}/{key_path}\n".format(
        host=args.s3_host, bucket=args.bucket, key_path=key_path
    )
    public_filepath = f"https://{args.bucket}.s3.yandex.net/{key_path}"
    sys.stderr.write(
        f"boto uploaded file at: {filepath} / {public_filepath}\n".format(filepath)
    )
    with open(args.output, mode="a") as f:
        f.write(public_filepath + "\n")


if __name__ == "__main__":
    main()
