#!/usr/bin/env python
# -*- coding: utf-8 -*-
import sys
import os
import argparse
import random
import string
import datetime
from boto.s3.connection import S3Connection
from boto.s3.key import Key


class Callback(object):
    def __init__(self, interactive=False):
        self.counter = 0
        self.interactive = interactive

    def __call__(self, *args):
        if self.interactive:
            sys.stdout.write("{:02}%\r".format(self.counter))
            sys.stdout.flush()
        elif self.counter % 5 == 0:
            sys.stdout.write(
                "[{}] {}% uploaded\n".format(datetime.datetime.now(), self.counter)
            )
        self.counter += 1


def upload_file(bucket, file_path, key_path, interactive=False):
    cb = Callback(interactive)
    key = Key(bucket)
    key.key = key_path
    key.set_contents_from_filename(file_path, cb=cb, num_cb=99)
    key.make_public()


def get_rand(n=10):
    rnd = random.SystemRandom()
    return "".join(
        random.SystemRandom().choice(
            string.ascii_uppercase + string.ascii_lowercase + string.digits
        )
        for _ in range(n)
    )


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("--bucket", "-b")
    parser.add_argument("--key-path", "-k")
    parser.add_argument("--file-path", "-f")
    parser.add_argument("--interactive", "-i", action="store_true")
    parser.add_argument("--s3-host", "-s", default="s3.mdst.yandex.net")
    parser.add_argument("--output", "-o", default="output.txt")
    parser.add_argument("--add-rand", "-a", action="store_true")
    args = parser.parse_args()

    if args.add_rand:
        sp = args.key_path.split("/")
        key_path = "/".join(sp[:-1] + [get_rand()] + [sp[-1]])
    else:
        key_path = args.key_path

    conn = S3Connection(
        host=args.s3_host,
        aws_access_key_id=os.environ["ACCESS_KEY_ID"],
        aws_secret_access_key=os.environ["ACCESS_KEY"],
    )
    bucket = [x for x in conn.get_all_buckets() if x.name == args.bucket][0]
    sys.stdout.write(
        "[{}] started uploading {}...\n".format(datetime.datetime.now(), args.key_path)
    )
    upload_file(
        bucket=bucket,
        file_path=args.file_path,
        key_path=key_path,
        interactive=args.interactive,
    )
    filepath = "https://{host}/{bucket}/{key_path}\n".format(
        host=args.s3_host, bucket=args.bucket, key_path=key_path
    )
    sys.stderr.write("boto uploaded file at: {}\n".format(filepath))
    with open(args.output, mode="a") as f:
        f.write(filepath)


if __name__ == "__main__":
    main()
