from collections import defaultdict, Counter
import math
import datetime
import json
import hashlib
from yt import yson
from yt.yson.yson_types import YsonEntity


WHITELIST = ["times_on_seekbar"]


def get_counter():
    return defaultdict(Counter)


def new_microsession():
    return {
        "heartbeats": get_counter(),
        "tcpinfo_total_retrans": 0,
        "bytes_sent": 0,
        "errors": [],
        "player_events": [],
        "browser_name_counter": get_counter(),
        "browser_version_counter": get_counter(),
        "ip_counter": get_counter(),
        "user_agent_counter": get_counter(),
        "device_type_counter": get_counter(),
        "a_station_counter": get_counter(),
        "provider_counter": get_counter(),
        "os_family_counter": get_counter(),
        "yandexuid_counter": get_counter(),
        "yu_hash_counter": get_counter(),
        "user_id_counter": get_counter(),
        "reqid_counter": get_counter(),
        "device_id_counter": get_counter(),
        "device_uuid_counter": get_counter(),
        "gogol_service_counter": get_counter(),
        "category_id_counter": get_counter(),
        "channel_id_counter": get_counter(),
        "page_id_counter": get_counter(),
        "puid_counter": get_counter(),
        "region_counter": get_counter(),
        "country_counter": get_counter(),
        "ref_from_counter": get_counter(),
        "ref_from_block_counter": get_counter(),
        "stream_block_counter": get_counter(),
        "player_version_counter": get_counter(),
        "channel_old_counter": get_counter(),
        "view_type_player_counter": get_counter(),
        "icookie_counter": get_counter(),
        "redir_licence_counter": get_counter(),
        "strongest_id_counter": get_counter(),
        "item_id_counter": get_counter(),
        "ppi_counter": get_counter(),
        "rid_counter": get_counter(),
        "_gogol_test_buckets": {},
        "sources_aggr": "",
        "zen_data": None,
        "ad_events": [],
        "ad_tracking_events": [],
        "lp": ListProcessor(),
        "times_on_seekbar": set(),
        "add_info": {
            "durations": {"chunks": 0},
            "traffic": defaultdict(lambda: {"count": 0, "bytes_sent": 0}),
            "chunks": 0,
            "chunks_types": Counter(),
            "playlists_types": Counter(),
            "playlist_is_kal": Counter(),
            "playlist_is_dhd": Counter(),
            "resolutions": defaultdict(lambda: {"count": 0, "bytes_sent": 0}),
            "errors_additional": defaultdict(lambda: {"times": [], "resolutions": []}),
            "sources": get_counter(),
        },
    }


def process_value(value):
    if isinstance(value, bytes):
        return value.decode("utf8", errors="replace")
    if isinstance(value, dict) or isinstance(value, list):
        return yson.dumps(value)
    return value


def process_dict(dct):
    return {k: (v if k in WHITELIST else process_value(v)) for k, v in dct.items()}


def normalify(dct):
    items = list(dct.items())
    for k, v in items:
        if not isinstance(k, str):
            k_ = str(k)
            dct.pop(k)
        else:
            k_ = k
        if isinstance(v, dict):
            dct[k_] = normalify(v)
        else:
            dct[k_] = v
    return dct


def get_counter_value(counter_collection, val=None):
    sources = ["rtb-dsp", "js_tracer", "redir", "strm"]
    last_hope = None
    for source in sources:
        cntr = counter_collection[source]
        if val == "icookie" and len(cntr) > 2:  # monkey-patching weird icookies
            continue
        mc = cntr.most_common(1)
        if mc:
            val = mc[0][0]
            if val.lower() in {
                "unknown",
                "",
                "null",
                "none",
                "undefined",
                "0.0.0.0",
                "unk",
            }:
                continue
            if getattr(cntr, "toint", False):
                val = int(val)
            return val
    return last_hope


def try_up_ms(field, ms, msid, rec, source, counter_field_name=None):
    if not counter_field_name:
        counter_field_name = "{}_counter".format(field)
    value = getattr(rec, field, None)
    if isinstance(value, bytes):
        value = value.decode("utf8", errors="replace")
    if value:
        if not isinstance(value, str):
            ms[msid][counter_field_name][source].toint = True
        val = str(value)
        if val is not None and len(ms[msid][counter_field_name][source]) <= 10:
            ms[msid][counter_field_name][source][val] += 1


def common_error_chunk(ms, msid, rec, source):
    try_up_ms("yandexuid", ms, msid, rec, source)
    try_up_ms("yu_hash", ms, msid, rec, source)
    try_up_ms("user_id", ms, msid, rec, source)
    try_up_ms("page_id", ms, msid, rec, source)
    try_up_ms("ref_from", ms, msid, rec, source)
    try_up_ms("ref_from_block", ms, msid, rec, source)
    try_up_ms("stream_block", ms, msid, rec, source)
    try_up_ms("reqid", ms, msid, rec, source)
    try_up_ms("category_id", ms, msid, rec, source)
    try_up_ms("channel_old", ms, msid, rec, source)
    try_up_ms("channel_id", ms, msid, rec, source)
    try_up_ms("player_version", ms, msid, rec, source)
    try_up_ms("browser_name", ms, msid, rec, source)
    try_up_ms("browser_version", ms, msid, rec, source)
    try_up_ms("os_family", ms, msid, rec, source)
    try_up_ms("ip", ms, msid, rec, source)
    try_up_ms("region", ms, msid, rec, source)
    try_up_ms("country", ms, msid, rec, source)
    try_up_ms("device_type", ms, msid, rec, source)
    try_up_ms("user_agent", ms, msid, rec, source)
    try_up_ms("a_station", ms, msid, rec, source)
    try_up_ms("provider", ms, msid, rec, source)
    try_up_ms("device_id", ms, msid, rec, source)
    try_up_ms("device_uuid", ms, msid, rec, source)
    try_up_ms("gogol_service", ms, msid, rec, source)
    try_up_ms("puid", ms, msid, rec, source)
    try_up_ms("view_type_player", ms, msid, rec, source)
    try_up_ms("icookie", ms, msid, rec, source)
    try_up_ms("strongest_id", ms, msid, rec, source)
    try_up_ms("item_id", ms, msid, rec, source)
    try_up_ms("ppi", ms, msid, rec, source)
    try_up_ms("rid", ms, msid, rec, source)
    if rec.time_on_seekbar and (0 < rec.time_on_seekbar < 86400):
        ms[msid]["times_on_seekbar"].add(round(rec.time_on_seekbar))
    bytes_sent = getattr(rec, "bytes_sent", 0) or 0
    if bytes_sent:
        event = process_value(rec.event)
        ms[msid]["bytes_sent"] += bytes_sent
        ms[msid]["add_info"]["traffic"][event]["count"] += 1
        ms[msid]["add_info"]["traffic"][event]["bytes_sent"] += bytes_sent
    if rec.tvandroid_data:
        try:
            ms[msid]["add_info"]["tvandroid_data"] = json.loads(rec.tvandroid_data)
        except:
            pass


def get_sources(sources):
    set_ = set()
    for k in sources:
        for kk in sources[k]:
            set_.add("{}_{}".format(k, kk))
    return ",".join(sorted(set_))


def ms_count_hits(ms_, ad_events):
    by_bidreqid = defaultdict(list)
    for event in ad_events:
        if event["countertype"] == 0 and event["win"] == 1:
            by_bidreqid[event["BidReqid"]].append(event["maxadscount"])
    ms_["hits_good"] = sum([by_bidreqid[br][0] for br in by_bidreqid])
    ms_["hits_block_good"] = len(by_bidreqid)


def ms_count_money(ms_, ad_events):
    ad_events = [x for x in ad_events if "countertype" in x]
    ms_count_hits(ms_, ad_events)
    ms_["winhits_good"] = sum(
        [
            (
                1
                if (
                    x["countertype"] == 0
                    and x["win"] == 1
                    and x["DspID"] not in {5, 10}
                )
                else 0
            )
            for x in ad_events
        ]
    )
    ms_["winhits_block_good"] = sum(
        [
            (
                1
                if (
                    x["countertype"] == 0
                    and x["win"] == 1
                    and x["DspID"] not in {5, 10}
                    and x["position"] == 0
                )
                else 0
            )
            for x in ad_events
        ]
    )
    ms_["shows_good"] = sum(
        [
            (
                1
                if (
                    x["countertype"] == 1
                    and x["win"] == 1
                    and x["DspID"] not in {5, 10}
                )
                else 0
            )
            for x in ad_events
        ]
    )
    ms_["shows_block_good"] = sum(
        [
            (
                1
                if (
                    x["countertype"] == 1
                    and x["win"] == 1
                    and x["DspID"] not in {5, 10}
                    and x["position"] == 0
                )
                else 0
            )
            for x in ad_events
        ]
    )
    ms_["price"] = sum(
        [
            (x["price"] if (x["countertype"] == 1 and x["DspID"] not in {5, 10}) else 0)
            for x in ad_events
        ]
    )
    ms_["partner_price"] = sum(
        [
            (
                x["partnerprice"]
                if (x["countertype"] == 1 and x["DspID"] not in {5, 10})
                else 0
            )
            for x in ad_events
        ]
    )


def get_extension(request):
    request = request.split("?")[0]
    return request.split(".")[-1]


def wrap_yson_dict(yson_object):
    if isinstance(yson_object, dict):
        return {process_value(k): wrap_yson_dict(v) for k, v in yson_object.items()}
    return process_value(yson_object)


# begin list processor code


def get_resolution_cat(h):
    if h is None or h < 0:
        return "unknown"
    return str(int(h))


def tryint(x):
    try:
        return int(x)
    except:
        return


def get_resolution(k, v):
    v.update({"resolution": k})
    return v


def is_(x):
    return x is not None and not isinstance(x, YsonEntity)


def round_ts(ts):
    to_secs = int(ts / 1000.0)
    rounded = to_secs - to_secs % 3600
    return rounded


def wrap_time(time_):
    if not time_ or time_ < 0 or time_ > 86400:
        return 0.0
    return time_


def get_dd():
    return defaultdict(
        lambda: {
            "watchedTime": 0,
            "maxWT": 0,
            "watchedTimeVisible": 0,
            "watchedTimeNonMuted": 0,
            "watchedTimeVisibleNonMuted": 0,
            "lastWatchedVNMTimestamp": 0,
            "lastWatchedTimestamp": 0,
            "stalledCount": 0,
            "stalledTime": 0,
        }
    )


def try_get_resolution(rec, cap=False):
    height_attr = "capHeight" if cap else "height"
    width_attr = "capWidth" if cap else "width"
    height = getattr(rec, height_attr)
    width = getattr(rec, width_attr)
    if is_(height) and is_(width):
        return min(height, width)
    elif is_(height):
        return height
    elif is_(width):
        return width
    else:
        return


class ListProcessor:
    def __init__(self):
        self.dd = get_dd()
        self.avglogs = []
        self.prev = None
        self.processed_recs = 0

    def __call__(self, rec):
        if not rec or not rec.timestamp:
            return
        if rec.is_ad:
            self.prev = rec
            return
        if (
            rec.watchedTime is None
            or rec.stalledCount is None
            or rec.stalledTime is None
        ):
            return
        if self.prev is None:
            wt = wrap_time(rec.watchedTime)
            sc = rec.stalledCount
            st = wrap_time(rec.stalledTime)
        else:
            wt = wrap_time(rec.watchedTime) - wrap_time(self.prev.watchedTime)
            sc = max(0, (rec.stalledCount or 0) - (self.prev.stalledCount or 0))
            st = wrap_time(rec.stalledTime) - wrap_time(self.prev.stalledTime)
        resolution = try_get_resolution(rec)
        if resolution:
            res_cat = get_resolution_cat(resolution)
            if wt >= 0:
                self.dd[res_cat]["watchedTime"] += wt
                self.dd[res_cat]["lastWatchedTimestamp"] = rec.timestamp
                if rec.isVisible:
                    self.dd[res_cat]["watchedTimeVisible"] += wt
                if rec.isMuted == False:
                    self.dd[res_cat]["watchedTimeNonMuted"] += wt
                if rec.isVisible and rec.isMuted == False:
                    self.dd[res_cat]["watchedTimeVisibleNonMuted"] += wt
                    self.dd[res_cat]["lastWatchedVNMTimestamp"] = rec.timestamp
            if wt > self.dd[res_cat]["maxWT"]:
                self.dd[res_cat]["maxWT"] = wt
            if st >= 0:
                self.dd[res_cat]["stalledTime"] += st
            if sc >= 0:
                self.dd[res_cat]["stalledCount"] += sc
        cap_resolution = try_get_resolution(rec, cap=True)
        if (
            resolution
            and cap_resolution
            and len(self.avglogs) <= 2000
        ):
            if resolution <= cap_resolution:
                avglogs = 0
            else:
                avglogs = math.log(resolution + 1) - math.log(cap_resolution + 1)
            self.avglogs.append(avglogs)
        self.prev = rec
        self.processed_recs += 1


# end list processor code


def add_pa_data(dct, pa_data):
    dct["player_alive_data"] = dict(pa_data)
    pa_wt = 0
    pa_wtv = 0
    pa_wtnm = 0
    pa_wtvnm = 0
    pa_st = 0
    pa_sc = 0
    for k in pa_data:
        pa_wt += pa_data[k].get("watchedTime") or 0
        pa_wtv += pa_data[k].get("watchedTimeVisible") or 0
        pa_wtnm += pa_data[k].get("watchedTimeNonMuted") or 0
        pa_wtvnm += pa_data[k].get("watchedTimeVisibleNonMuted") or 0
        pa_st += pa_data[k].get("stalledTime") or 0
        pa_sc += pa_data[k].get("stalledCount") or 0
    dct["view_time_player_alive"] = pa_wt
    dct["view_time_player_alive_non_muted"] = pa_wtnm
    dct["view_time_player_alive_visible_non_muted"] = pa_wtvnm
    dct["player_alive_data"]["total_view_time"] = pa_wt
    dct["player_alive_data"]["total_view_time_non_muted"] = pa_wtnm
    dct["player_alive_data"]["total_view_time_visible"] = pa_wtv
    dct["player_alive_data"]["total_view_time_visible_non_muted"] = pa_wtvnm
    dct["player_alive_data"]["total_stalled_time"] = pa_st
    dct["player_alive_data"]["total_stalled_count"] = pa_sc
    tryint_keys = [k for k in dct if tryint(k)]
    if len(tryint_keys) > 10:
        top10 = sorted(
            tryint_keys,
            key=lambda k: pa_data[k]["watchedTime"] + pa_data[k]["stalledTime"],
            reverse=True,
        )[:10]
        dct["player_alive_data"] = {
            k: v
            for k, v in dct["player_alive_data"].items()
            if not tryint(k) or k in top10
        }


def prev_hb(vt):
    if vt > 30:
        return vt - ((vt % 30) or 30)
    if vt > 20:
        return 20
    if vt > 10:
        return 10
    return 0


def next_hb(vt):
    if vt >= 30:
        return vt + 30 - (vt % 30)
    if vt >= 20:
        return 30
    if vt >= 10:
        return 20
    return 10


def apply_pa_correction(gd, vtpa):
    if not gd and not vtpa:
        return
    elif not gd:
        return int(round(vtpa)) if vtpa <= 10 else 0
    elif not vtpa:
        return min(gd)
    else:
        for vt in gd:
            if prev_hb(vt) <= vtpa <= next_hb(vt):
                return int(round(vtpa))
    return min(gd)


def wrap_bytes(s):
    if isinstance(s, bytes):
        return s.decode("utf8", errors="replace")
    return s


def has_bucket(tb):
    sp = tb.split(",")
    return len(sp) == 3 and tryint(sp[2]) and (0 <= tryint(sp[2]) <= 100)


def process_testids(rec, dct):
    try:
        test_buckets = process_value(rec.testIds.decode("utf8")).split(";")
    except:
        return
    for tb in test_buckets:
        testid = tb.split(",")[0]
        if testid not in dct or (has_bucket(tb) and not has_bucket(dct[testid])):
            dct[testid] = tb


def get_fake_bucket(id_):
    id_ = str(id_) + "salt"
    return int(hashlib.sha256(id_.encode("utf8")).hexdigest(), 16) % 100


def make_test_buckets(dct):
    tb_dict = dct["_gogol_test_buckets"]
    fake_bucket = get_fake_bucket(
        dct.get("puid")
        or dct.get("device_id")
        or dct.get("yandexuid")
        or dct.get("icookie")
        or dct.get("vsid")
    )
    for testid in tb_dict:
        if not has_bucket(testid):
            tb_dict[testid] = "{},0,{}".format(testid, fake_bucket)
    return ";".join(sorted(tb_dict.values()))


def microsessions_reducer(key, recs):
    ms = defaultdict(new_microsession)  # microsessions
    msid = None
    vcid = None
    last_chunk_ts = 0
    last_ts = 0
    last_resolution = "start"
    bad_msid = False
    prev_msid = None
    rec_count = 0
    vcids = set()
    for rec in recs:
        if rec.add_info:
            add_info = wrap_yson_dict(yson.loads(rec.add_info))
        else:
            add_info = {}
        vcids.add(rec.video_content_id)
        if len(vcids) == 20:
            break
        event = process_value(rec.event)
        source = process_value(rec.source)
        video_content_id = process_value(rec.video_content_id)
        if not msid or (
            video_content_id != vcid
            and video_content_id is not None
            and getattr(rec, "error_id_raw", "") != b"PlayerAlive"
        ):
            vcid = video_content_id
            if msid and not bad_msid:
                ms[msid]["add_info"]["last_timestamp"] = last_ts
            prev_msid = msid
            msid = (vcid, rec.timestamp)
            rec_count = 0
            if prev_msid and not prev_msid[0]:
                try:
                    ms[msid] = ms.pop(prev_msid)
                except KeyError:  # if too many rtb-dsp ad_events
                    ms[msid] = new_microsession()
            last_chunk_ts = 0
            last_resolution = "start"
            bad_msid = False
        if not msid:
            continue
        if bad_msid:
            continue
        if rec_count >= 1000000:
            continue
        ms[msid]["add_info"]["sources"][source][event] += 1
        rec_count += 1
        if (
            ms[msid]["add_info"]["sources"]["rtb-dsp"]["rtb-dsp"]
            + ms[msid]["add_info"]["sources"]["awaps"]["awaps"]
            + ms[msid]["add_info"]["sources"]["chtracking"]["chtracking"]
            >= 5000
        ):
            ms.pop(msid)
            bad_msid = True
            continue
        if source == "js_tracer" and rec.testIds:
            process_testids(rec, ms[msid]["_gogol_test_buckets"])
        if rec.redir_licence is not None:
            ms[msid]["redir_licence_counter"][source][rec.redir_licence] += 1
        if event == "heartbeat":
            common_error_chunk(ms, msid, rec, source)
            if source == "redir":
                hb = "redir_heartbeat"
            else:
                hb = process_value(rec.error_id)
            ms[msid]["heartbeats"][source][hb] += 1
            if getattr(rec, "non_muted", None):
                ms[msid]["heartbeats"][source]["{}_non_muted".format(hb)] += 1
        elif event in {"start", "playlist", "create_player"}:
            common_error_chunk(ms, msid, rec, source)
            ms[msid]["add_info"]["playlists_types"][
                process_value(rec.extension)
            ] += 1
            ms[msid]["add_info"]["playlist_is_kal"][
                wrap_bytes(rec.is_kal) or "not_kal"
            ] += 1
            ms[msid]["add_info"]["playlist_is_dhd"][rec.is_dhd] += 1
            if rec.utm_data:
                try:
                    ms[msid]["add_info"]["utm_data"] = wrap_yson_dict(
                        yson.loads(rec.utm_data)
                    )
                except ValueError:
                    pass
        elif event in {"error", "player_event"}:
            common_error_chunk(ms, msid, rec, source)
            column_name = "errors" if event == "error" else "player_events"
            try:
                rec.error_id
            except AttributeError:
                raise Exception("vsid: {}: attributes: {}".format(rec.vsid, dir(rec)))
            if rec.error_id_raw == b"PlayerAlive" and rec.parsed_player_state:
                ms[msid]["lp"](rec.parsed_player_state)
            if len(ms[msid][column_name]) < 5000 and rec.error_id_raw != b"PlayerAlive":
                event = {
                    "id": process_value(rec.error_id),
                    "id_raw": process_value(rec.error_id_raw) or "",
                    "resolution": last_resolution,
                    "rel_time": rec.timestamp - msid[1],
                    "details": add_info.get("error_details") or {},
                    "source": process_value(rec.source),
                }
                ms[msid][column_name].append(event)
        elif event == "chunk":
            common_error_chunk(ms, msid, rec, source)
            if rec.timestamp and last_chunk_ts:
                ms[msid]["add_info"]["durations"]["chunks"] += (
                    rec.timestamp - last_chunk_ts
                )
            ms[msid]["add_info"]["chunks"] += 1
            ms[msid]["tcpinfo_total_retrans"] += add_info.get(
                "tcpinfo_total_retrans", 0
            )
            last_chunk_ts = rec.timestamp
            if getattr(rec, "resolution", 0):
                ms[msid]["add_info"]["resolutions"][process_value(rec.resolution)]["count"] += 1
                ms[msid]["add_info"]["resolutions"][process_value(rec.resolution)]["bytes_sent"] += getattr(rec, "bytes_sent", 0) or 0
                last_resolution = rec.resolution
            ms[msid]["add_info"]["chunks_types"][process_value(rec.extension)] += 1
        elif event == "rtb-dsp":
            common_error_chunk(ms, msid, rec, source)
            ms[msid]["ad_events"].append(add_info)
        elif event in ("awaps", "chtracking"):
            ms[msid]["ad_tracking_events"].append(add_info)
        last_ts = rec.timestamp
        if msid and not bad_msid:
            ms[msid]["add_info"]["prev_event"] = ms[msid]["add_info"].get("last_event")
            ms[msid]["add_info"]["last_event"] = rec.event
    if msid and not bad_msid:
        ms[msid]["add_info"]["last_timestamp"] = last_ts
    for msid in sorted(ms, key=lambda x: x[1]):
        ms[msid]["video_content_id"] = msid[0]
        ms[msid]["timestamp"] = msid[1]
        ms[msid]["vsid"] = key
        ms[msid]["add_info"]["resolutions"] = dict(ms[msid]["add_info"]["resolutions"])
        ms[msid]["add_info"]["chunks_types"] = dict(
            ms[msid]["add_info"]["chunks_types"]
        )
        ms[msid]["add_info"]["sources"] = {
            k: dict(v) for k, v in ms[msid]["add_info"]["sources"].items()
        }
        rl_val = get_counter_value(ms[msid]["redir_licence_counter"])
        if rl_val:
            ms[msid]["add_info"]["redir_licence"] = rl_val
        ms[msid].pop("redir_licence_counter")
        counters = [x for x in ms[msid] if x.endswith("_counter")]
        zen_data = {}
        for c in counters:
            c_sp = c.split("_counter")[0]
            val = get_counter_value(ms[msid][c], val=c_sp)
            if c_sp == "region" and isinstance(val, str):
                val = 0
            if c_sp in ("strongest_id", "item_id", "ppi", "rid"):
                if val:
                    zen_data[c_sp] = val
            else:
                ms[msid][c_sp] = val
            ms[msid]["add_info"][c] = dict(ms[msid][c])
            ms[msid].pop(c)
        if zen_data:
            ms[msid]["zen_data"] = zen_data
        if ms[msid]["_gogol_test_buckets"]:
            ms[msid]["gogol_test_buckets"] = make_test_buckets(ms[msid])
        ms[msid].pop("_gogol_test_buckets")
        new_hbdict = {}
        for source in ms[msid]["heartbeats"]:
            hbdict = dict(ms[msid]["heartbeats"][source])
            new_hbdict[source] = hbdict
            if source in {"js_tracer", "strm"}:
                jst_duration = 0
                jst_duration_non_muted = 0
                if "30SecHeartbeat" in hbdict:
                    jst_duration = 30 * hbdict["30SecHeartbeat"]
                elif "20SecWatched" in hbdict:
                    jst_duration = 20
                elif "10SecWatched" in hbdict:
                    jst_duration = 10
                elif "4SecWatched" in hbdict:
                    jst_duration = 4
                if "30SecHeartbeat_non_muted" in hbdict:
                    jst_duration_non_muted = 30 * hbdict["30SecHeartbeat_non_muted"]
                elif "20SecWatched_non_muted" in hbdict:
                    jst_duration_non_muted = 20
                elif "10SecWatched_non_muted" in hbdict:
                    jst_duration_non_muted = 10
                elif "4SecWatched_non_muted" in hbdict:
                    jst_duration_non_muted = 4
                ms[msid]["add_info"]["durations"][source] = jst_duration
                ms[msid]["add_info"]["durations"][
                    "js_tracer_non_muted"
                ] = jst_duration_non_muted
            elif source == "redir":
                redir_duration = 30 * hbdict["redir_heartbeat"]
                redir_duration_non_muted = 30 * hbdict.get(
                    "redir_heartbeat_non_muted", 0
                )
                ms[msid]["add_info"]["durations"]["redir"] = redir_duration
                ms[msid]["add_info"]["durations"][
                    "redir_non_muted"
                ] = redir_duration_non_muted
        ms[msid]["heartbeats"] = new_hbdict
        lp = ms[msid].pop("lp")
        if lp.processed_recs:
            add_pa_data(ms[msid], lp.dd)
        else:
            for k in (
                "player_alive_data",
                "view_time_player_alive",
                "view_time_player_alive_non_muted",
                "view_time_player_alive_visible_non_muted",
            ):
                ms[msid][k] = None
        avglogs = None
        if lp.avglogs:
            avglogs = sum(lp.avglogs) / len(lp.avglogs)
        ms[msid]["avglogs"] = avglogs
        ms[msid]["times_on_seekbar"] = sorted(ms[msid]["times_on_seekbar"])
        duration = None
        durdict = ms[msid]["add_info"]["durations"]
        good_durations = [
            durdict[k]
            for k in durdict
            if k in {"js_tracer", "redir", "strm"} and durdict[k]
        ]
        duration = apply_pa_correction(
            good_durations, ms[msid]["view_time_player_alive"]
        )
        if (
            duration is None
            and "chunks" in durdict
            and (ms[msid].get("channel_old") or "").startswith("fm_")
        ):  # TODO: remove fm_ fix (MMA-4111)
            duration = durdict["chunks"]
        good_durations_non_muted = [
            durdict[k]
            for k in durdict
            if k in {"js_tracer_non_muted", "redir_non_muted"} and durdict[k]
        ]
        duration_non_muted = apply_pa_correction(
            good_durations_non_muted, ms[msid]["view_time_player_alive_non_muted"]
        )
        ms[msid]["view_time"] = duration or 0
        ms[msid]["view_time_non_muted"] = duration_non_muted or 0
        if ms[msid]["view_time_non_muted"] > ms[msid]["view_time"]:
            ms[msid]["view_time_non_muted"] = ms[msid]["view_time"]
        yu_hash = ms[msid].get("yu_hash", "")
        if yu_hash and yu_hash != "-":
            ms[msid]["user_id"] = ms[msid]["yu_hash"]
        ms[msid]["add_info"]["errors_additional"] = dict(
            ms[msid]["add_info"]["errors_additional"]
        )
        ms[msid]["add_info"] = normalify(ms[msid]["add_info"])
        ms[msid]["timestamp"] = msid[1]
        if ms[msid]["ad_events"]:
            ad_events = ms[msid]["ad_events"]
            ms_count_money(ms[msid], ad_events)
        sources_aggr = get_sources(ms[msid]["add_info"]["sources"])
        ms[msid]["is_kal"] = (
            ",".join(sorted(ms[msid]["add_info"]["playlist_is_kal"].keys())) or None
        )
        is_dhd = bool(ms[msid]["add_info"]["playlist_is_dhd"]["True"])
        is_not_dhd = bool(ms[msid]["add_info"]["playlist_is_dhd"]["False"])
        ms[msid]["is_dhd"] = {
            (True, True): "both",
            (False, False): "unknown",
            (True, False): "dhd",
            (False, True): "not_dhd",
        }[(is_dhd, is_not_dhd)]
        ms[msid]["chunks_types"] = ",".join(
            sorted(ms[msid]["add_info"]["chunks_types"].keys())
        )
        ms[msid]["playlists_types"] = ",".join(
            sorted(ms[msid]["add_info"]["playlists_types"].keys())
        )
        ms[msid]["add_info"]["sources_aggr"] = sources_aggr
        ms[msid]["sources_aggr"] = sources_aggr
        yield process_dict(ms[msid])
        del ms[msid]
