#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import division
import os
import argparse
from nile.api.v1 import (
    clusters,
    filters as nf,
    extractors as ne,
    aggregators as na,
    statface as ns,
    with_hints,
    Record
)
import nile.files as nfi
from qb2.api.v1 import typing as qt
from yql.api.v1.client import YqlClient
import codecs
from videolog_common import (
    date_range,
    yt_get_date_from_table as get_date,
    get_dates_from_stat,
    get_stat_headers,
    get_cluster,
    get_driver,
    StatPusher
)


def apply_replacements(s, pairs):
    for p in pairs:
        s = s.replace(p[0], p[1])
    return s


def process_dates(
    dates, cluster, yql_client, report, pool, replace_mask=None
):
    if len(dates) == 1:
        date_s = str(dates[0])
    else:
        date_s = '{}_{}'.format(dates[0], dates[-1])
    root = '//home/videolog/kpi_money_graph/{}'.format(date_s)
    money_table = '{}/money'.format(root)
    report_table = '{}/report'.format(root)
    date_start = str(dates[0])
    date_finish = str(dates[-1])

    with codecs.open('myshell_money.sql', 'r', 'utf8') as f:
        money_query = f.read()
    base_replacements = [
        ('@date_start', date_start),
        ('@date_finish', date_finish),
        ('@pool', pool),
    ]
    money_query = apply_replacements(
        money_query,
        base_replacements + [('@output_table', money_table)]
    )

    req = yql_client.query(
        money_query, title='KPI Money Graph | YQL', syntax_version=1
    )
    req.run()
    req.wait_progress()

    with codecs.open('money_kpi_graph.sql', 'r', 'utf8') as f:
        report_query = f.read()
    report_query = apply_replacements(
        report_query,
        base_replacements + [
            ('@money_table', money_table),
            ('@output_table', report_table)
        ]
    )

    req = yql_client.query(
        report_query, title='KPI Money Graph | YQL', syntax_version=1
    )
    req.run()
    req.wait_progress()

    stat_pusher = StatPusher(
        cluster,
        report=report,
        replace_mask=replace_mask
    )

    stat_pusher.push(report_table)


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument(
        '--report', default='Video/Others/Strm/money_per_hour_ci'
    )
    parser.add_argument('--from', default=None)
    parser.add_argument('--pool', default=None)
    parser.add_argument('--replace_mask', default=None)
    parser.add_argument('--job_root', default='//home/videolog/tmp')
    parser.add_argument('--to', default=None)
    parser.add_argument('--title', default='KPI Money Graph | YQL')
    args = parser.parse_args()

    cluster = get_cluster(clusters, args)
    yql_client = YqlClient(
        db=os.environ['YT_PROXY'],
        token=os.environ['YQL_TOKEN']
    )

    from_ = getattr(args, 'from')
    to_ = getattr(args, 'to')

    if from_ and to_:
        dates = date_range(from_, to_)
    else:
        stat_headers = get_stat_headers()

        last_date_from_stat = get_dates_from_stat(
            headers=stat_headers,
            report=args.report,
            dimensions=[]
        )[-1]
        available_dates_1 = set(
            get_date(x) for x in get_driver(cluster).client.search(
                root='//home/videoquality/vh_analytics/strm_cube_2',
                path_filter=lambda x: get_date(x) and x.endswith('/sessions')
            )
        )
        available_dates_2 = set(
            get_date(x) for x in get_driver(cluster).client.search(
                root='//statbox/cooked_logs/bs-dsp-cooked-log/v1/1d',
                path_filter=lambda x: get_date(x)
            )
        )

        available_dates = sorted(available_dates_1 & available_dates_2)

        dates = [x for x in available_dates if x > last_date_from_stat]

    print('processing {}'.format(dates))
    if dates:
        process_dates(
            dates, cluster, yql_client, args.report, args.pool,
            replace_mask=args.replace_mask
        )


if __name__ == "__main__":
    main()
