pragma library('stability_common.sql');
import stability_common symbols $getPlatform, $getRefFrom, $refFromWrapperSafe;

use hahn;
pragma yt.PoolTrees = "physical";
pragma yt.UseDefaultTentativePoolTrees;
pragma yson.DisableStrict;

$date_from = "@[date_from]";
$date = "@[date]";

-- @[graph_info]

$getString = ($json, $fieldPath) -> {
    return Yson::ConvertToString(Yson::YPath(Yson::ParseJson($json), $fieldPath));
};

$getBool = ($json, $fieldPath) -> {
    return Yson::ConvertToBool(Yson::YPath(Yson::ParseJson($json), $fieldPath));
};

$isMPDDrmError = ($EventName) -> {
    return $EventName like 'shaka.%_6%';
};

$isFairplayError = ($EventName) -> {
    return $EventName like 'FAIRPLAY_%';
};

$isDRMProxyError = ($EventName) -> {
    return $EventName == 'DRM_PROXY_ERROR' or
        $EventName == 'DRM_PROXY_ERROR_SIGNATURE_EXPIRED';
};

$isFatalError = ($eventType, $data) -> {
    return $eventType == 'fatal' or 
        ($eventType == 'error' and $getBool($data, '/data/isFatal'));
};

$loads = (
    select 
        videoContentId, 
        vsid,
        TableName() as fielddate,
        $getPlatform(service, userAgent) as platform,
        $refFromWrapperSafe($getRefFrom(Yson::ParseJson(data))) as ref_from,
        version,
    from range("//logs/strm-gogol-log/1d", $date_from, $date)
    where 
        service == 'StreamPlayer' and 
        eventType == 'event' and
        eventName == 'LoadSource' and
        videoContentId is not null and
        data like '%drmConfig%'
);

$fails = (select eventName, videoContentId, vsid
    from range("//logs/strm-gogol-log/1d", $date_from, $date)
where 
    service == 'StreamPlayer' and 
    $isFatalError(eventType, data) and
    (
        eventName == 'NO_DRM' or
        $isDRMProxyError(eventName) or
        $isFairplayError(eventName) or
        $isMPDDrmError(eventName)
    ) and
    videoContentId is not null
);
    
$sessions = (
    select 
        b.eventName as eventName,
        a.fielddate as fielddate,
        a.platform as platform,
        a.ref_from as ref_from,
        a.version as player_version,
    from $loads as a
    left join $fails as b
    on a.vsid == b.vsid and a.videoContentId == b.videoContentId
);

$totalize = ($row) -> {
    $result = AsList($row);
    $result = ListUnionAll(
        $result,
        ListMap(
            $result, ($x)->(AddMember(RemoveMember($x, "ref_from"), "ref_from", "_total_"))
        )
    );
    $result = ListUnionAll(
        $result,
        ListMap(
            $result, ($x)->(AddMember(RemoveMember($x, "platform"), "platform", "_total_"))
        )
    );
    $result = ListUnionAll(
        $result,
        ListMap(
            $result, ($x)->(AddMember(RemoveMember($x, "player_version"), "player_version", "_total_"))
        )
    );
    RETURN $result
};

$totalized = process $sessions using $totalize(TableRow());

insert into `@[root]/@[date_from]-@[date]` WITH TRUNCATE
select 
    fielddate,
    platform,
    ref_from,
    player_version,
    count(*) as hits,
    count_if(eventName is null) as ok,
    count_if(eventName == 'NO_DRM') as no_drm,
    count_if($isFairplayError(eventName)) as fairplay_error,
    count_if($isMPDDrmError(eventName)) as mpd_drm_error,
    
    count_if(eventName  == 'DRM_PROXY_ERROR') as drm_proxy_error,
    count_if(eventName  == 'DRM_PROXY_ERROR_SIGNATURE_EXPIRED') as drm_proxy_error_signature_expired,
from $totalized
group by fielddate, platform, ref_from, player_version;
