#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import division
import os
import argparse
import datetime
import json
import time
import smtplib
import sys
from email.mime.text import MIMEText
from yql.api.v1.client import YqlClient
from videolog_common import YqlRunner

TITLE = "DumaTV Monitoring | YQL"
QUERY_STUB = """
use hahn;

select count_If(price > 0 or view_time > 0) as value
from `//cubes/video-strm/{date}/sessions`
where ParentUUID == '46b23efbcc4871f38087004a49a84a28';

select sum(IsView) as value
from `//home/videoquality/vh_analytics/vhds_reduced/{date}`
where ParentUUID == '46b23efbcc4871f38087004a49a84a28';
"""


class ErrorWrapper(object):
    def __init__(self):
        self.errors = []
        self.smtp = smtplib.SMTP("outbound-relay.yandex.net")

    def write(self, msg):
        if not msg.endswith("\n"):
            msg += "\n"
        sys.stderr.write(msg)
        self.errors.append(msg)

    def _send_email(self, addressees, msg):
        msg = MIMEText(msg)
        msg["Subject"] = "DumaTV monitoring"
        msg["From"] = "dumatv-watch@yandex-team.ru"
        msg["To"] = ",".join(addressees)
        print("Sending message to {}".format(addressees))
        retries = 0
        while retries < 10:
            try:
                self.smtp = smtplib.SMTP("outbound-relay.yandex.net")
                self.smtp.sendmail(msg["From"], addressees, msg.as_string())
            except Exception as e:
                sys.stderr.write("error: {} \n\n\n".format(e))
                time.sleep(60)
                retries += 1
                if retries >= 10:
                    raise

    def send_email(self, *addressees):
        text = "The following problems occurred:\n\n{}".format(
            "\n".join(self.errors)
        )
        self._send_email(addressees, text)


def process_dates(dates, args, ew):
    for date in dates:
        print("processing {}".format(date))
        yc = YqlClient(token=os.environ["YQL_TOKEN"])
        yc.config.db = None
        assert yc.config.db is None
        yr = YqlRunner(yc, title=TITLE)
        query = QUERY_STUB.format(date=date)
        req = yr.run(query)
        results = yr.get_results(req)
        value_sessions = results[0][0]["value"]
        value_ss = results[1][0]["value"]
        quotient = value_ss / float(value_sessions)
        message = "\n".join(
            [
                "date: {}".format(date),
                "sessions value: {}".format(value_sessions),
                "ss value: {}".format(value_ss),
                "quotient: {}".format(quotient),
            ]
        )
        if not (0.95 <= quotient <= 1.05):
            ew.write(message)
        else:
            print(message)


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("--dates")
    args = vars(parser.parse_args())

    with open(args["dates"], "r") as f:
        dates = json.load(f)

    ew = ErrorWrapper()

    if dates:
        process_dates(dates, args, ew)
    else:
        print("no dates to process")

    if ew.errors:
        ew.send_email("pecheny@yandex-team.ru", "ensuetina@yandex-team.ru")


if __name__ == "__main__":
    main()
