#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import division
import sys
import os
import argparse
import datetime
from nile.api.v1 import clusters
from yql.api.v1.client import YqlClient
from videolog_common import (
    apply_replacements,
    date_range,
    get_cluster,
    get_driver,
    get_stat_headers,
    get_dates_from_stat,
    get_date,
    StatPusher,
    zen_ref_from_treatment
)

jstracer_root = "//logs/jstracer-log/1d"
TITLE = "First Frame | YQL"


def process_dates(dates, cluster, args):
    root = "//home/videolog/first_frame/{}_{}".format(
        dates[0], dates[-1]
    )
    tmp_table = "{}/tmp".format(root)
    output_table = "{}/report".format(root)
    output_table_browser = "{}/report_browser".format(root)
    proxy = os.environ["YT_PROXY"]

    yc = YqlClient(db=proxy.lower(), token=os.environ["YQL_TOKEN"])
    with open("first_frame.sql", "r") as f:
        query = f.read().decode("utf8")
    query = apply_replacements(
        query,
        {
            "@pool": args["pool"],
            "@output_table": output_table,
            "@[output_table_browser]": output_table_browser,
            "@tmp_table": tmp_table,
            "@date_from": str(dates[0]),
            "@date_to": str(dates[-1]),
            "-- @ref_from_preprocess": zen_ref_from_treatment
        },
    )
    req = yc.query(query, title=TITLE, syntax_version=1)
    req.run()
    req.wait_progress()
    if req.status != "COMPLETED":
        raise Exception("Error in YQL operation: {}".format(req.share_url))
    sp = StatPusher(cluster, report=args["report"])
    print("pushing to {}".format(args["report"]))
    sp.push(output_table)
    sp = StatPusher(cluster, report=args["report_browser"])
    print("pushing to {}".format(args["report_browser"]))
    sp.push(output_table_browser)


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("--pool", default="robot-mma-nirvana")
    parser.add_argument(
        "--report", default="Video/Others/Strm/Stability/FirstFrameBeta"
    )
    parser.add_argument(
        "--report_browser",
        default="Video/Others/Strm/Stability/FirstFrameWindowsDetailed"
    )
    parser.add_argument("--from")
    parser.add_argument("--to")
    args = vars(parser.parse_args())

    from_ = args.get("from")
    to_ = args.get("to")

    cluster = get_cluster(clusters, args)
    yt = get_driver(cluster).client

    if from_ and to_:
        dates = sorted(date_range(from_, to_))
    else:
        dates_from_stat = get_dates_from_stat(
            report=args["report"], headers=get_stat_headers()
        )
        last_date = dates_from_stat[-1]
        jstracer_dates = set(
            get_date(x) for x in yt.search(
                root=jstracer_root,
                node_type="table",
                path_filter=lambda x: bool(get_date(x))
            )
        )
        sessions_dates = set(
            get_date(x) for x in yt.search(
                root="//cubes/video-strm",
                node_type="table",
                path_filter=lambda x: bool(get_date(x)) and x.endswith("/sessions")
            )
        )
        last_available_date = sorted(
            jstracer_dates & sessions_dates
        )[-1]
        if last_available_date > last_date:
            dates = sorted(
                date_range(
                    last_date + datetime.timedelta(days=1), last_available_date
                )
            )
        else:
            dates = []

    if dates:
        process_dates(dates, cluster, args)
    else:
        print("no dates to process")


if __name__ == "__main__":
    main()
