pragma library('stability_common.sql');
import stability_common symbols $getPlatform, $processRefFromDetailed, $mockService, $totalizePlatform;

use hahn;
pragma yt.UseDefaultTentativePoolTrees;
pragma yson.DisableStrict;
pragma AnsiInForEmptyOrNullableItemsCollections;
pragma yt.PoolTrees = "physical";
pragma yt.TentativePoolTrees = "cloud";
pragma SimpleColumns;
pragma yt.MaxRowWeight = "128M";

-- @[graph_info]

$from = "@[date_from]";
$to = "@[date]";
$table = '//logs/strm-gogol-log/1d';

$getString = ($json, $fieldPath) -> {
    return Yson::ConvertToString(Yson::YPath(Yson::ParseJson($json), $fieldPath));
};
$getBool = ($json, $fieldPath) -> {
    return Yson::ConvertToBool(Yson::YPath(Yson::ParseJson($json), $fieldPath));
};
$getDouble = ($json, $fieldPath) -> {
    return Yson::ConvertToDouble(Yson::YPath(Yson::ParseJson($json), $fieldPath));
};

$NO_ERROR = 'OK';

$getErrorCode = ($eventName, $data) -> {
    return if(
        $eventName == 'AdPodError',
        $getString($data, '/data/code') ?? 'unknown',
        null
    )
};

$getAdType = ($eventName, $data) -> {
    return if(
        $eventName == 'AdEnd',
        $getString($data, '/data/adType') ?? 'unknown',
        null
    )
};

$getStopAdReason = ($eventName, $data) -> {
    return if(
        $eventName == 'AdEnd',
        $getString($data, '/data/stopAdReason'),
        null
    )
};

$tmp = select
    eventName,
    lag(eventName) over w as prevEventName,
    clientTimestamp,
    vsid,
from (
    select
        eventName, vsid, clientTimestamp,
    from range($table, $from, $to)
    where
        service == 'StreamPlayer' and
        eventType == 'event' and
        eventName in ('AdStart', 'AdEnd', 'AdPodStart')
) window w as (
    partition by vsid
    order by clientTimestamp
);

-- достаем ad breaks в которых не было AdPodStart
$failed_ad_breaks = select
    vsid,
    clientTimestamp,
    prevEventName == 'AdPodStart' as was_ad_shown,
from $tmp where eventName == 'AdEnd' and prevEventName is not null;

$ad_events = select
    TableName() as fielddate,

    $processRefFromDetailed(
        substring(labels_from, 0, 1000), 
        userAgent, 
        additionalParameters_from_block, 
        additionalParameters_stream_block
    ) as ref_from,
    $getPlatform(service ?? $mockService(labels_from), userAgent) as platform,

    eventName,
    vsid,
    clientTimestamp,
    $getErrorCode(eventName, data) as error_code,
    $getAdType(eventName, data) as ad_type,
    $getBool(data, '/isVideoADB') ?? false as adblock,
    $getStopAdReason(eventName, data) as stop_ad_reason,
from range($table, $from, $to)
where
    service == 'StreamPlayer' and
    eventType == 'event' and
    eventName in ('AdStart', 'AdEnd', 'AdPodError');

$data = select
    fielddate,
    eventName,
    ad_type,
    lag(error_code) over w as error_code,
    lag(eventName) over w as prevEventName,
    ref_from,
    platform,
    clientTimestamp,
    vsid,
    adblock,
    stop_ad_reason,
from $ad_events
window w as (
    partition by vsid
    order by clientTimestamp
);

$ad_breaks_with_code = select
    fielddate,
    error_code,
    ad_type,
    platform,
    ref_from,
    clientTimestamp,
    vsid,
    adblock,
    stop_ad_reason,
from $data
where eventName == 'AdEnd' and prevEventName is not null;

$ad_breaks = select
    a.fielddate as fielddate,
    if(b.was_ad_shown, $NO_ERROR, a.error_code ?? 'unknown') as error_code,
    a.ad_type as ad_type,
    a.platform as platform,
    a.ref_from as ref_from,
    if(a.adblock, 'true', 'false') as adblock, 
    a.stop_ad_reason as stop_ad_reason,
    a.vsid as vsid,
from $ad_breaks_with_code as a
inner join $failed_ad_breaks as b
on (a.vsid == b.vsid and a.clientTimestamp == b.clientTimestamp);

$ad_breaks = select * from $ad_breaks
where error_code != 'unknown' or (error_code == 'unknown' and stop_ad_reason == 'AdCompleted');

$slow_net = select 
    if(percentile(bandwidth_estimate, 0.6) < 500000, 'true', 'false') as slow_net,
    vsid, 
from (
    select 
        $getDouble(data, '/bandwidthEstimate') ?? 0 as bandwidth_estimate,
        vsid
    from range($table, $from, $to)
    where
        service == 'StreamPlayer' and 
        eventType == 'event' and
        eventName == 'PlayerAlive'
)
where bandwidth_estimate > 0
group by vsid;

$ad_breaks = select 
    fielddate,
    error_code,
    ad_type,
    platform,
    ref_from,
    adblock, 
    stop_ad_reason,
    slow_net ?? 'false' as slow_net
from $ad_breaks as a
left join $slow_net as b
using (vsid);


$totalize = ($row) -> {
    $rf = ListExtend($row.ref_from, ["_total_"]);
    $result = ListMap($rf, ($x)->(AddMember(RemoveMember($row, "ref_from"), "ref_from", $x)));

    $add = ListFlatMap($result, $totalizePlatform);
    $result = ListUnionAll($result, $add);

    $add = ListMap($result, ($x)->(AddMember(RemoveMember($x, "ref_from"), "ref_from", "_total_")));
    $result = ListUnionAll($result, $add);

    $add = ListMap($result, ($x)->(AddMember(RemoveMember($x, "ad_type"), "ad_type", "_total_")));
    $result = ListUnionAll($result, $add);

    $add = ListMap($result, ($x)->(AddMember(RemoveMember($x, "adblock"), "adblock", "_total_")));
    $result = ListUnionAll($result, $add);

    $add = ListMap($result, ($x)->(AddMember(RemoveMember($x, "slow_net"), "slow_net", "_total_")));
    $result = ListUnionAll($result, $add);

    RETURN $result
};

$totalized = process $ad_breaks using $totalize(TableRow());


$result = select
    fielddate,
    platform,
    ref_from,
    error_code,
    ad_type,
    adblock,
    slow_net,
    count(*) as hits
from $totalized
group by
    fielddate,
    platform,
    ref_from,
    error_code,
    ad_type,
    adblock,
    slow_net;

upsert into stat.`@[report]/daily`
select
    error_code ?? 'unknown' as error_code,
    platform ?? 'unknown' as platform,
    ad_type ?? 'unknown' as ad_type,
s.* without error_code, platform, ad_type from $result as s;
