import datetime

THRESHOLD = 60 * 10


class FixedOffset(datetime.tzinfo):
    def __init__(self, hours, name):
        self.__offset = datetime.timedelta(hours=hours)
        self.__name = name

    def utcoffset(self, dt):
        return self.__offset

    def tzname(self, dt):
        return self.__name

    def dst(self, dt):
        return datetime.timedelta(0)


moscow = FixedOffset(3, "Moscow")


def timestamp_to_hour(ts):
    dt = datetime.datetime.fromtimestamp(ts, tz=moscow)
    return dt.strftime("%Y-%m-%d %H:00:00")


def reducer(key, recs):
    last_location_rec = None
    for rec in recs:
        if rec.source_log == b"cryptageo":
            last_location_rec = rec
        if not (
            last_location_rec
            and (rec.timestamp - last_location_rec.timestamp) <= THRESHOLD
        ):
            continue
        if (
            rec.source_log == b"access"
            and last_location_rec
            and ((rec.timestamp - last_location_rec.timestamp) <= THRESHOLD)
        ):
            yield {
                "yandexuid": key,
                "region_id": last_location_rec.region_id,
                "timestamp": rec.timestamp,
                "ip": rec.ip,
                "source_log": "access",
                "user_agent": rec.user_agent,
                "os_family": rec.os_family,
                "operator": rec.operator,
                "fielddate": timestamp_to_hour(rec.timestamp),
                "ect": rec.ect,
                "downlink": rec.downlink,
                "rtt": rec.rtt,
            }
            continue


class StructWrapper:
    def __init__(self, obj, dct):
        self.obj = obj
        self.dct = dct

    def __getattr__(self, name):
        try:
            return self.dct[name]
        except KeyError:
            return getattr(self.obj, name)


def reducer2(_, recs):
    last_mm_rec = None
    for rec in recs:
        if rec.source_log == b"metrika" and rec.timestamp:
            last_mm_rec = rec
            continue
        if (
            last_mm_rec
            and rec.timestamp
            and (rec.timestamp - last_mm_rec.timestamp) <= THRESHOLD
        ):
            yield StructWrapper(
                rec, {"connection_type": last_mm_rec.connection_type}
            )
        else:
            yield rec


def reducer_chunks(_, recs):
    last_location_rec = None
    for rec in recs:
        if rec.source_log == b"cryptageo" and rec.timestamp:
            last_location_rec = rec
            continue
        if not (
            last_location_rec
            and rec.timestamp
            and (rec.timestamp - last_location_rec.timestamp) <= THRESHOLD
        ):
            continue
        yield StructWrapper(
            rec,
            {
                "region_id": last_location_rec.region_id,
                "fielddate": timestamp_to_hour(rec.timestamp),
            },
        )
