#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import division
import sys
import os
import argparse
import datetime
from nile.api.v1 import clusters
from yql.api.v1.client import YqlClient
from videolog_common import (
    apply_replacements,
    date_range,
    get_cluster,
    get_driver,
    get_stat_headers,
    get_dates_from_stat,
    get_date,
    StatPusher,
    YqlRunner,
)

gogol_root = "//logs/strm-gogol-log/1d"
TITLE = "[MMA-3930] Quality Report | YQL"
TITLE_KP = "[MMA-4715] Quality Report KP | YQL"
REPORT = "Video/Others/Strm/Stability/MMA-3930-quality-v4a"
REPORT_KP = "Video/Others/Strm/Stability/MMA-4715-quality-kinopoisk"
ROOT = "//home/videoquality/vh_analytics/mma-3930-quality-v3"
ROOT_KP = "//home/videoquality/vh_analytics/mma-4715-quality-kinopoisk"


def process_dates(dates, cluster, args):
    output_table = "{}/{}_{}".format(args["root"], dates[0], dates[-1])
    tmp_table = "{}/tmp/{}_{}".format(args["root"], dates[0], dates[-1])
    proxy = os.environ["YT_PROXY"]
    yc = YqlClient(db=proxy.lower(), token=os.environ["YQL_TOKEN"])
    yr = YqlRunner(client=yc, title=args["title"])
    with open(args["query"], "rb") as f:
        query = f.read().decode("utf8")
    query = apply_replacements(
        query,
        {
            "@[pool]": args["pool"],
            "@[report]": args["report"],
            "@[output_table]": output_table,
            "@[tmp_table]": tmp_table,
            "@[date_from]": str(dates[0]),
            "@[date_to]": str(dates[-1]),
        },
    )
    attachments = [
        {
            "path": "analytics/videolog/strm-stats/strm_cube_2/stability/quality_report_avglog/quality_report_avglog_common.sql"
        },
        {
            "path": "analytics/videolog/strm-stats/strm_cube_2/stability/quality_report_avglog/{}".format(
                args["list_processor"]
            )
        },
        {
            "path": "analytics/videolog/strm-stats/strm_cube_2/stability/stability_common.sql"
        },
    ]
    if args["kinopoisk"]:
        attachments.append({
            "path": "analytics/videolog/strm-stats/strm_cube_2/stability/mma_4376_covid_operator_stats/providers.sql"
        })
    yr.run(
        query,
        attachments=attachments,
        wait=True,
    )
    if not args.get("debug"):
        sp = StatPusher(cluster, report=args["report"])
        print("pushing to {}".format(args["report"]))
        kwargs = {"replace_mask": "fielddate"} if args.get("redo") else {}
        sp.push(output_table, **kwargs)
    if not args.get("save_tmp"):
        get_driver(cluster).remove(tmp_table)


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("--pool", default="robot-mma-nirvana")
    parser.add_argument("--report", default=REPORT)
    parser.add_argument("--root", default=ROOT)
    parser.add_argument("--title", default=TITLE)
    parser.add_argument("--from")
    parser.add_argument("--to")
    parser.add_argument("--kinopoisk", action="store_true")
    parser.add_argument("--redo", action="store_true")
    parser.add_argument("--no-sort-dates", action="store_true")
    parser.add_argument("--query", default="quality_report_avglog.sql")
    parser.add_argument("--list_processor", default="list_processor.py")
    parser.add_argument("--save_tmp", action="store_true")
    args = vars(parser.parse_args())
    if args["kinopoisk"]:
        if args["root"] == ROOT:
            args["root"] = ROOT_KP
        if args["report"] == REPORT:
            args["report"] = REPORT_KP
        if args["title"] == TITLE:
            args["title"] = TITLE_KP
        args["list_processor"] = "list_processor_kp.py"
        args["query"] = "quality_report_avglog_kp.sql"

    from_ = args.get("from")
    to_ = args.get("to")

    cluster = get_cluster(clusters, args)
    yt = get_driver(cluster).client

    if from_ and to_:
        dates = sorted(date_range(from_, to_))
    else:
        dates_from_stat = get_dates_from_stat(
            report=args["report"], headers=get_stat_headers()
        )
        last_date = dates_from_stat[-1]
        print("last date: {}".format(last_date))
        last_available_date = sorted(
            get_date(x)
            for x in yt.search(
                root=gogol_root,
                node_type="table",
                path_filter=lambda x: bool(get_date(x)),
            )
        )[-1]
        print("last available date: {}".format(last_available_date))
        if last_available_date > last_date:
            dates = sorted(
                date_range(
                    last_date + datetime.timedelta(days=1), last_available_date
                )
            )
        else:
            dates = []

    if dates:
        if not args["no_sort_dates"]:
            dates = sorted(dates, reverse=True)
        for date in dates:
            print("processing {}".format(date))
            process_dates([date], cluster, args)
    else:
        print("no dates to process")


if __name__ == "__main__":
    main()
