use hahn;
pragma library("stability_common.sql");
PRAGMA AnsiInForEmptyOrNullableItemsCollections;
pragma yt.Pool = "@[pool]";
pragma yson.DisableStrict;
pragma yt.MaxRowWeight = "64M";
import stability_common symbols $getRefFrom, $refFromWrapper, $getPlatform, $getFielddate, $getVideoContentId;
pragma library("quality_report_avglog_common.sql");
import quality_report_avglog_common symbols $parsePlayerState, $avglogProcessList, $avglogParseResolutions, $getPlayerStateFromData;
pragma library("providers.sql");
import providers symbols $providers;

$date_from = "@[date_from]";
$date_to = "@[date_to]";
$tmp_table = "@[tmp_table]";
$out_table = "@[output_table]";

$getStalledInfo = ($ev, $ts, $data) -> {
    $parsed = Yson::ParseJson($data);
    return AsStruct(
        $ev as ev,
        $ts as ts,
        Yson::ConvertToInt64(Yson::YPath($parsed, "/data/stalledId")) as id,
        Yson::ConvertToDouble(Yson::YPath($parsed, "/data/stalledDuration")) as duration,
        Yson::ConvertToString(Yson::YPath($parsed, "/labels/reason")) as reason,
    )
};

$tmp = (
select
    TableName() as fielddate,
    vsid as VSID,
    version,
    xRealIp,
    $refFromWrapper(labels_from) as ref_from,
    userAgent,
    service,
    if(eventType == "fatal", 1, 0) as fatal,
    IF(
        eventName in ("Stalled", "StalledEnd"),
        $getStalledInfo(eventName, clientTimestamp, data)
    ) as stalled_info,
    IF(
        eventName in ("Start", "CreatePlayer"),
        <|ev: eventName, ts: clientTimestamp|>
    ) as event_info,
    IF(
        eventName == "PlayerAlive",
        AsList($parsePlayerState(
            $getPlayerStateFromData(data)
        )) ?? ListMap(Yson::ConvertToList(Yson::YPath(
            Yson::ParseJson(data), "/data/states"
        )), $parsePlayerState)
    ) as state,
    videoContentId ?? $getVideoContentId(Yson::ParseJson(data)) as video_content_id,
    yandexuid as Yandexuid
from range(
    `logs/strm-gogol-log/1d`, $date_from, $date_to
) with columns Struct<yandexuid:String?>
where service is not null and (eventName in ("PlayerAlive", "CreatePlayer", "Start", "Stalled", "StalledEnd") or eventType == "fatal")
);

$tmp_flattened = (
    select t.*,
    from $tmp as t
    flatten list by state
);

$regions = (
    select distinct name
    from `//home/geotargeting/public/geobase/regions`
    where country_id == 225 and type == 5
);

$getRegion = ($ip) -> {
    $region = Geo::RoundRegionByIp($ip, "region").name;
    return if(
        $region in $regions,
        $region,
        "Неизвестный регион"
    )
};

$getProvider = ($ip) -> {
    $isp = Geo::GetIspNameByIp($ip);
    return IF(
        DictContains($providers, $isp),
        unwrap($providers[$isp]),
        "Прочие"
    )
};

$ei_si_grouped = (
    select
        fielddate,
        VSID,
        video_content_id,
        aggregate_list(stalled_info) as stalled_infos,
        aggregate_list(event_info) as event_infos
    from $tmp
    where stalled_info is not null
    or event_info is not null
    group by fielddate,
        VSID,
        video_content_id
);

$group_flatten = (
    SELECT 
        VSID,
        video_content_id,
        fielddate,
        max(xRealIp) as ip,
        $getRegion(max(xRealIp)) as region,
        $getProvider(max(xRealIp)) as provider,
        Geo::GetAsset(max(xRealIp)) as a_system,
        max(version) as version,
        max(Yandexuid) as yandexuid,
        max(ref_from) as ref_from,
        max(fatal) as fatal,
        $getPlatform(max(service), max(userAgent)) as platform,
        listsort(aggregate_list(state), ($x)->($x.`timestamp`)) as unprocessed_states,
    from $tmp_flattened
    where state.`timestamp` is not null
    and state.watchedTime is not null
    and state.stalledTime is not null
    and state.stalledCount is not null
    group by VSID,
        video_content_id,
        fielddate
);

$processed = (
    select
        gf.*,
        stalled_infos,
        event_infos
    from $group_flatten as gf
    left join $ei_si_grouped as esg using (fielddate, VSID, video_content_id)
);

$unprocessed_state_type = Struct<
    'capHeight':Int64?,
    'capWidth':Int64?,
    'height':Int64?,
    'isMuted':Bool?,
    'isVisible':Bool?,
    'is_ad':Bool,
    'maxHeight':Int64?,
    'maxWidth':Int64?,
    'stalledCount':Int64?,
    'stalledTime':Double?,
    'state':String?,
    'timestamp':Uint64?,
    'watchedTime':Double?,
    'width':Int64?
>;

$input_type = Struct<
    'VSID':String?,
    'platform':String?,
    'ref_from':String?,
    'region':String?,
    'provider':String?,
    'a_system':String?,
    'fielddate':String?,
    'fatal':Int64?,
    'unprocessed_states':List<$unprocessed_state_type>,
    'event_infos':List<Struct<ev:String?,ts:Uint64?>>?,
    'stalled_infos':List<Struct<ev:String?,ts:Uint64?,id:Int64?,duration:Double?,reason:String?>>?,
    'version':String?,
    'video_content_id':String?,
    'yandexuid':String?
>;

$output_type = Struct<
    'VSID':String?,
    'color': String?,
    'len_category': String?,
    'avglogs':List<Double>?,
    'region':String?,
    'provider':String?,
    'a_system':String?,
    'start_time':Double?,
    'fatal':Bool?,
    'color_reason': String?,
    'tvt':Double?,
    'tst':Double?,
    'tsc':Int64?,
    'p_4k_stalled_count':Int64?,
    'p_4k_stalled_time':Double?,
    'p_4k_watched_time':Double?,
    'p_fhd_stalled_count':Int64?,
    'p_fhd_stalled_time':Double?,
    'p_fhd_watched_time':Double?,
    'p_hd_stalled_count':Int64?,
    'p_hd_stalled_time':Double?,
    'p_hd_watched_time':Double?,
    'p_ld_stalled_count':Int64?,
    'p_ld_stalled_time':Double?,
    'p_ld_watched_time':Double?,
    'p_sd_stalled_count':Int64?,
    'p_sd_stalled_time':Double?,
    'p_sd_watched_time':Double?,
    'platform':String?,
    'ref_from':String?,
    'muted_cat':String?,
    'timestamp':Int64?,
    'fielddate':String?,
    'unprocessed_states':List<$unprocessed_state_type>,
    'event_infos':List<Struct<ev:String?,ts:Uint64?>>?,
    'stalled_infos':List<Struct<ev:String?,ts:Uint64?,id:Int64?,duration:Double?,reason:String?>>?,
    'version':String?,
    'video_content_id':String?,
    'yandexuid':String?
>;

$mapper = Python::mapper(
    Callable<($input_type)->Stream<$output_type>>, FileContent("list_processor_kp.py")
);

$tmp_add = (
    process $processed using $mapper(TableRow())
);

$ib = (
    select
        JoinKey,
        license
    from `//home/videolog/strm_meta/iron_branch/concat`
    where license is not null
);

$tmp_g = (
    select
        VSID,
        video_content_id,
        license ?? "UNKNOWN" as license,
        len_category,
        color,
        muted_cat,
        platform,
        ref_from,
        fielddate,
        provider,
        a_system,
        region,
        avglogs,
        p_4k_watched_time,
        p_4k_stalled_time,
        p_4k_stalled_count,
        p_fhd_watched_time,
        p_fhd_stalled_time,
        p_fhd_stalled_count,
        p_hd_watched_time,
        p_hd_stalled_time,
        p_hd_stalled_count,
        p_sd_watched_time,
        p_sd_stalled_time,
        p_sd_stalled_count,
        p_ld_watched_time,
        p_ld_stalled_time,
        p_ld_stalled_count,
    from $tmp_add as t
    left join $ib as i on (t.video_content_id == i.JoinKey)
);

$totalize = ($row) -> {
    $result = AsList($row);
    $result = ListUnionAll(
        $result,
        ListMap(
            $result, ($x)->(AddMember(RemoveMember($x, "ref_from"), "ref_from", "_total_"))
        )
    );
    $result = ListUnionAll(
        $result,
        ListMap(
            $result, ($x)->(AddMember(RemoveMember($x, "platform"), "platform", "_total_"))
        )
    );
    $result = ListUnionAll(
        $result,
        ListMap(
            $result, ($x)->(AddMember(RemoveMember($x, "a_system"), "a_system", "_total_"))
        )
    );
    $result = ListUnionAll(
        $result,
        ListMap(
            $result, ($x)->(AddMember(RemoveMember($x, "provider"), "provider", "_total_"))
        )
    );
    $result = ListUnionAll(
        $result,
        ListMap(
            $result, ($x)->(AddMember(RemoveMember($x, "region"), "region", "_total_"))
        )
    );
    $result = ListUnionAll(
        $result,
        ListMap(
            $result, ($x)->(AddMember(RemoveMember($x, "license"), "license", "_total_"))
        )
    );
    $result = ListUnionAll(
        $result,
        ListMap(
            $result, ($x)->(AddMember(RemoveMember($x, "len_category"), "len_category", "_total_"))
        )
    );
    RETURN $result
};

$wrap = ($x) -> {
    RETURN IF(
        not ($x >= 0) or cast($x as String) in ("-inf", "inf", "nan"),
        0.0,
        $x
    )
};
$nullwrap = ($x) -> {
    RETURN IF(
        not ($x >= 0) or cast($x as String) in ("-inf", "inf", "nan"),
        null,
        $x
    )
};

$totalized_ = process $tmp_g using $totalize(TableRow());

$totalized = (
    select
        t.*,
        $nullwrap(p_ld_stalled_count / p_ld_watched_time * 60.0) as p_ld_stalled_count_per_min,
        $nullwrap(p_sd_stalled_count / p_sd_watched_time * 60.0) as p_sd_stalled_count_per_min,
        $nullwrap(p_hd_stalled_count / p_hd_watched_time * 60.0) as p_hd_stalled_count_per_min,
        $nullwrap(p_fhd_stalled_count / p_fhd_watched_time * 60.0) as p_fhd_stalled_count_per_min,
        $nullwrap(p_4k_stalled_count / p_4k_watched_time * 60.0) as p_4k_stalled_count_per_min,
        $nullwrap(p_ld_stalled_time / p_ld_watched_time * 60.0) as p_ld_stalled_time_per_min,
        $nullwrap(p_sd_stalled_time / p_sd_watched_time * 60.0) as p_sd_stalled_time_per_min,
        $nullwrap(p_hd_stalled_time / p_hd_watched_time * 60.0) as p_hd_stalled_time_per_min,
        $nullwrap(p_fhd_stalled_time / p_fhd_watched_time * 60.0) as p_fhd_stalled_time_per_min,
        $nullwrap(p_4k_stalled_time / p_4k_watched_time * 60.0) as p_4k_stalled_time_per_min,
        $nullwrap(
            (
                (p_fhd_stalled_time ?? 0)
                + (p_hd_stalled_time ?? 0)
                + (p_sd_stalled_time ?? 0)
                + (p_ld_stalled_time ?? 0)
                + (p_4k_stalled_time ?? 0)
            )
        ) as p_total_stalled_time,
        $nullwrap(
            (
                (p_fhd_stalled_count ?? 0)
                + (p_hd_stalled_count ?? 0)
                + (p_sd_stalled_count ?? 0)
                + (p_ld_stalled_count ?? 0)
                + (p_4k_stalled_count ?? 0)
            )
        ) as p_total_stalled_count,
        $nullwrap(
            (
                (p_fhd_watched_time ?? 0)
                + (p_hd_watched_time ?? 0)
                + (p_sd_watched_time ?? 0)
                + (p_ld_watched_time ?? 0)
                + (p_4k_watched_time ?? 0)
            )
        ) as p_total_watched_time,
        $nullwrap(
            (
                (p_fhd_stalled_time ?? 0)
                + (p_hd_stalled_time ?? 0)
                + (p_sd_stalled_time ?? 0)
                + (p_ld_stalled_time ?? 0)
                + (p_4k_stalled_time ?? 0)
            ) / 
            (
                (p_fhd_watched_time ?? 0)
                + (p_hd_watched_time ?? 0)
                + (p_sd_watched_time ?? 0)
                + (p_ld_watched_time ?? 0)
                + (p_4k_watched_time ?? 0)
            ) * 60.0
        ) as p_total_stalled_time_per_min,
        $nullwrap(
            (
                (p_fhd_stalled_count ?? 0)
                + (p_hd_stalled_count ?? 0)
                + (p_sd_stalled_count ?? 0)
                + (p_ld_stalled_count ?? 0)
                + (p_4k_stalled_count ?? 0)
            ) / 
            (
                (p_fhd_watched_time ?? 0)
                + (p_hd_watched_time ?? 0)
                + (p_sd_watched_time ?? 0)
                + (p_ld_watched_time ?? 0)
                + (p_4k_watched_time ?? 0)
            ) * 60.0
        ) as p_total_stalled_count_per_min,
    from $totalized_ as t
);

$fielddateCheck = ($fd) -> {
    RETURN $date_from <= $fd and $fd <= $date_to
};

$log_line_tmp = (
    select
        avglogs,
        platform,
        ref_from,
        muted_cat,
        a_system,
        provider,
        region,
        license,
        len_category,
        fielddate
    from $totalized
    flatten list by avglogs
    where $fielddateCheck(fielddate)
);

$log_line_grouped = (
    select
        ref_from,
        fielddate,
        platform,
        muted_cat,
        a_system,
        provider,
        region,
        license,
        len_category,
        SUM(avglogs) / CAST(COUNT(*) as Double) as avglog,
        $nullwrap(percentile(avglogs, 0.25)) as avglog_p25,
        $nullwrap(percentile(avglogs, 0.50)) as avglog_p50,
        $nullwrap(percentile(avglogs, 0.75)) as avglog_p75,
        $nullwrap(percentile(avglogs, 0.80)) as avglog_p80,
        $nullwrap(percentile(avglogs, 0.85)) as avglog_p85,
        $nullwrap(percentile(avglogs, 0.90)) as avglog_p90,
        $nullwrap(percentile(avglogs, 0.95)) as avglog_p95,
        $nullwrap(percentile(avglogs, 0.99)) as avglog_p99,
    from $log_line_tmp
    group by ref_from,
        fielddate,
        platform,
        muted_cat,
        a_system,
        provider,
        region,
        license,
        len_category
);

$rest_count = (
    select
        ref_from,
        fielddate,
        platform,
        muted_cat,
        a_system,
        provider,
        region,
        license,
        len_category,
        count(*) as `count`,
        count_if(color == "green") / cast(count(*) as double) as color_green_share,
        count_if(color == "yellow") / cast(count(*) as double) as color_yellow_share,
        count_if(color == "red") / cast(count(*) as double) as color_red_share,
        $nullwrap(SUM(p_total_watched_time) / 60.0) as tvt,
        $nullwrap(SUM(p_4k_watched_time) / 60.0) as p_4k_watched_time_mins,
        $nullwrap(SUM(p_fhd_watched_time) / 60.0) as p_fhd_watched_time_mins,
        $nullwrap(SUM(p_hd_watched_time) / 60.0) as p_hd_watched_time_mins,
        $nullwrap(SUM(p_sd_watched_time) / 60.0) as p_sd_watched_time_mins,
        $nullwrap(SUM(p_ld_watched_time) / 60.0) as p_ld_watched_time_mins,
        $nullwrap(SUM(p_4k_stalled_time) / sum(p_4k_watched_time) * 60.0) as p_4k_stalled_time_per_min,
        $nullwrap(SUM(p_4k_stalled_count) / sum(p_4k_watched_time) * 60.0) as p_4k_stalled_count_per_min,
        $nullwrap(SUM(p_fhd_stalled_time) / sum(p_fhd_watched_time) * 60.0) as p_fhd_stalled_time_per_min,
        $nullwrap(SUM(p_fhd_stalled_count) / sum(p_fhd_watched_time) * 60.0) as p_fhd_stalled_count_per_min,
        $nullwrap(SUM(p_hd_stalled_time) / sum(p_hd_watched_time) * 60.0) as p_hd_stalled_time_per_min,
        $nullwrap(SUM(p_hd_stalled_count) / sum(p_hd_watched_time) * 60.0) as p_hd_stalled_count_per_min,
        $nullwrap(SUM(p_sd_stalled_time) / sum(p_sd_watched_time) * 60.0) as p_sd_stalled_time_per_min,
        $nullwrap(SUM(p_sd_stalled_count) / sum(p_sd_watched_time) * 60.0) as p_sd_stalled_count_per_min,
        $nullwrap(SUM(p_ld_stalled_time) / sum(p_ld_watched_time) * 60.0) as p_ld_stalled_time_per_min,
        $nullwrap(SUM(p_ld_stalled_count) / sum(p_ld_watched_time) * 60.0) as p_ld_stalled_count_per_min,
        $nullwrap(
            sum(p_total_stalled_time) / sum(p_total_watched_time) * 60.0
        ) as p_total_stalled_time_per_min,
        $nullwrap(
            sum(p_total_stalled_count) / sum(p_total_watched_time) * 60.0
        ) as p_total_stalled_count_per_min,
        
        $nullwrap(percentile(p_ld_stalled_count_per_min, 0.25)) as p_ld_stalled_count_per_min_p25,
        $nullwrap(percentile(p_ld_stalled_count_per_min, 0.50)) as p_ld_stalled_count_per_min_p50,
        $nullwrap(percentile(p_ld_stalled_count_per_min, 0.75)) as p_ld_stalled_count_per_min_p75,
        $nullwrap(percentile(p_ld_stalled_count_per_min, 0.95)) as p_ld_stalled_count_per_min_p95,

        $nullwrap(percentile(p_sd_stalled_count_per_min, 0.25)) as p_sd_stalled_count_per_min_p25,
        $nullwrap(percentile(p_sd_stalled_count_per_min, 0.50)) as p_sd_stalled_count_per_min_p50,
        $nullwrap(percentile(p_sd_stalled_count_per_min, 0.75)) as p_sd_stalled_count_per_min_p75,
        $nullwrap(percentile(p_sd_stalled_count_per_min, 0.95)) as p_sd_stalled_count_per_min_p95,

        $nullwrap(percentile(p_hd_stalled_count_per_min, 0.25)) as p_hd_stalled_count_per_min_p25,
        $nullwrap(percentile(p_hd_stalled_count_per_min, 0.50)) as p_hd_stalled_count_per_min_p50,
        $nullwrap(percentile(p_hd_stalled_count_per_min, 0.75)) as p_hd_stalled_count_per_min_p75,
        $nullwrap(percentile(p_hd_stalled_count_per_min, 0.95)) as p_hd_stalled_count_per_min_p95,

        $nullwrap(percentile(p_fhd_stalled_count_per_min, 0.25)) as p_fhd_stalled_count_per_min_p25,
        $nullwrap(percentile(p_fhd_stalled_count_per_min, 0.50)) as p_fhd_stalled_count_per_min_p50,
        $nullwrap(percentile(p_fhd_stalled_count_per_min, 0.75)) as p_fhd_stalled_count_per_min_p75,
        $nullwrap(percentile(p_fhd_stalled_count_per_min, 0.95)) as p_fhd_stalled_count_per_min_p95,

        $nullwrap(percentile(p_fhd_stalled_count_per_min, 0.25)) as p_4k_stalled_count_per_min_p25,
        $nullwrap(percentile(p_fhd_stalled_count_per_min, 0.50)) as p_4k_stalled_count_per_min_p50,
        $nullwrap(percentile(p_fhd_stalled_count_per_min, 0.75)) as p_4k_stalled_count_per_min_p75,
        $nullwrap(percentile(p_fhd_stalled_count_per_min, 0.95)) as p_4k_stalled_count_per_min_p95,

        $nullwrap(percentile(p_total_stalled_count_per_min, 0.25)) as p_total_stalled_count_per_min_p25,
        $nullwrap(percentile(p_total_stalled_count_per_min, 0.50)) as p_total_stalled_count_per_min_p50,
        $nullwrap(percentile(p_total_stalled_count_per_min, 0.75)) as p_total_stalled_count_per_min_p75,
        $nullwrap(percentile(p_total_stalled_count_per_min, 0.95)) as p_total_stalled_count_per_min_p95,

        $nullwrap(percentile(p_ld_stalled_time_per_min, 0.25)) as p_ld_stalled_time_per_min_p25,
        $nullwrap(percentile(p_ld_stalled_time_per_min, 0.50)) as p_ld_stalled_time_per_min_p50,
        $nullwrap(percentile(p_ld_stalled_time_per_min, 0.75)) as p_ld_stalled_time_per_min_p75,
        $nullwrap(percentile(p_ld_stalled_time_per_min, 0.95)) as p_ld_stalled_time_per_min_p95,

        $nullwrap(percentile(p_sd_stalled_time_per_min, 0.25)) as p_sd_stalled_time_per_min_p25,
        $nullwrap(percentile(p_sd_stalled_time_per_min, 0.50)) as p_sd_stalled_time_per_min_p50,
        $nullwrap(percentile(p_sd_stalled_time_per_min, 0.75)) as p_sd_stalled_time_per_min_p75,
        $nullwrap(percentile(p_sd_stalled_time_per_min, 0.95)) as p_sd_stalled_time_per_min_p95,

        $nullwrap(percentile(p_hd_stalled_time_per_min, 0.25)) as p_hd_stalled_time_per_min_p25,
        $nullwrap(percentile(p_hd_stalled_time_per_min, 0.50)) as p_hd_stalled_time_per_min_p50,
        $nullwrap(percentile(p_hd_stalled_time_per_min, 0.75)) as p_hd_stalled_time_per_min_p75,
        $nullwrap(percentile(p_hd_stalled_time_per_min, 0.95)) as p_hd_stalled_time_per_min_p95,

        $nullwrap(percentile(p_fhd_stalled_time_per_min, 0.25)) as p_fhd_stalled_time_per_min_p25,
        $nullwrap(percentile(p_fhd_stalled_time_per_min, 0.50)) as p_fhd_stalled_time_per_min_p50,
        $nullwrap(percentile(p_fhd_stalled_time_per_min, 0.75)) as p_fhd_stalled_time_per_min_p75,
        $nullwrap(percentile(p_fhd_stalled_time_per_min, 0.95)) as p_fhd_stalled_time_per_min_p95,
        
        $nullwrap(percentile(p_4k_stalled_time_per_min, 0.25)) as p_4k_stalled_time_per_min_p25,
        $nullwrap(percentile(p_4k_stalled_time_per_min, 0.50)) as p_4k_stalled_time_per_min_p50,
        $nullwrap(percentile(p_4k_stalled_time_per_min, 0.75)) as p_4k_stalled_time_per_min_p75,
        $nullwrap(percentile(p_4k_stalled_time_per_min, 0.95)) as p_4k_stalled_time_per_min_p95,

        $nullwrap(percentile(p_total_stalled_time_per_min, 0.25)) as p_total_stalled_time_per_min_p25,
        $nullwrap(percentile(p_total_stalled_time_per_min, 0.50)) as p_total_stalled_time_per_min_p50,
        $nullwrap(percentile(p_total_stalled_time_per_min, 0.75)) as p_total_stalled_time_per_min_p75,
        $nullwrap(percentile(p_total_stalled_time_per_min, 0.95)) as p_total_stalled_time_per_min_p95,
    from $totalized
    where $fielddateCheck(fielddate) and (
        p_fhd_watched_time > 0
        or p_hd_watched_time > 0
        or p_ld_watched_time > 0
        or p_sd_watched_time > 0
        or p_fhd_stalled_count > 0
        or p_hd_stalled_count > 0
        or p_ld_stalled_count > 0
        or p_sd_stalled_count > 0
    )
    group by ref_from, fielddate, platform, muted_cat, a_system,
        provider,
        region,
        license,
        len_category
);

$joined = (
    select
        t.*,
        $nullwrap(avglog) as avglog,
        $nullwrap(avglog_p25) as avglog_p25,
        $nullwrap(avglog_p50) as avglog_p50,
        $nullwrap(avglog_p75) as avglog_p75,
        $nullwrap(avglog_p80) as avglog_p80,
        $nullwrap(avglog_p85) as avglog_p85,
        $nullwrap(avglog_p90) as avglog_p90,
        $nullwrap(avglog_p95) as avglog_p95,
        $nullwrap(avglog_p99) as avglog_p99,
    from $rest_count as t
    left join $log_line_grouped as l using (ref_from, fielddate, platform, muted_cat, a_system,
        provider,
        region,
        license,
        len_category)
);

$nwrap = ($x) -> (IF($x == "", null, $x));

insert into $out_table with truncate
select * from $joined
where $nwrap(platform) is not null
and $nwrap(muted_cat) is not null
and $nwrap(provider) is not null
and $nwrap(region) is not null
and $nwrap(license) is not null
and $nwrap(len_category) is not null
and $nwrap(a_system) is not null
and `count` >= 5000;

insert into $tmp_table WITH TRUNCATE 
select * from $tmp_add;
