#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import division
import sys
import os
import argparse
import datetime
from nile.api.v1 import clusters
from yql.api.v1.client import YqlClient
from videolog_common import (
    apply_replacements,
    date_range,
    get_cluster,
    get_driver,
    get_stat_headers,
    get_missing_dates_from_stat,
    get_date,
    StatPusher,
    YqlRunner,
)

cube_root = "//cubes/video-strm"
TITLE = "[MMA-4388] Remapped Cube | YQL"


def process_dates(dates, cluster, args):
    output_table = "{}/{}_{}".format(args["root"], dates[0], dates[-1])
    # tmp_table = "{}/tmp/{}_{}".format(args["root"], dates[0], dates[-1])
    proxy = os.environ["YT_PROXY"]
    yc = YqlClient(db=proxy.lower(), token=os.environ["YQL_TOKEN"])
    yr = YqlRunner(client=yc, title=TITLE)
    with open("remapped_cube_report.sql", "rb") as f:
        query = f.read().decode("utf8")
    query = apply_replacements(
        query,
        {
            "@[pool]": args["pool"],
            "@[output_table]": output_table,
            # "@[tmp_table]": tmp_table,
            "@[date_from]": str(dates[0]),
            "@[date_to]": str(dates[-1]),
        },
    )
    yr.run(
        query,
        wait=True,
        attachments=[
            {
                "path": "analytics/videolog/strm-stats/strm_cube_2/stability/stability_common.sql"
            },
            {
                "path": "analytics/videolog/strm-stats/strm_cube_2/stability/remapped_cube_report/ad_load_counter.py"
            },
            {
                "path": "analytics/videolog/strm-stats/strm_cube_2/vh_cube_2/vh_cube_2_common.sql"
            },
            {
                "path": "quality/mstand_metrics/users/24julia/yandex_shows.sql"
            }
        ],
    )
    # if not args.get("debug"):
    #     sp = StatPusher(cluster, report=args["report"])
    #     print("pushing to {}".format(args["report"]))
    #     kwargs = {"replace_mask": "fielddate"} if args.get("redo") else {}
    #     sp.push(output_table, **kwargs)
    # if args.get("notmp"):
    #     get_driver(cluster).remove(tmp_table)


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("--pool", default="robot-mma-nirvana")
    parser.add_argument(
        "--report", default="Video/Others/Strm/remapped_cube"
    )
    parser.add_argument("--root", default="//home/videolog/remapped_cube")
    parser.add_argument("--from")
    parser.add_argument("--redo", action="store_true")
    # parser.add_argument("--notmp", action="store_true")
    parser.add_argument("--to")
    args = vars(parser.parse_args())

    from_ = args.get("from")
    to_ = args.get("to")

    cluster = get_cluster(clusters, args)
    yt = get_driver(cluster).client

    if from_ and to_:
        dates = sorted(date_range(from_, to_))
    else:
        stat_headers = get_stat_headers()
        yesterday = datetime.date.today() - datetime.timedelta(days=1)
        weekago = yesterday - datetime.timedelta(days=7)
        missing_dates = set(
            get_missing_dates_from_stat(
                stat_headers, args["report"], weekago, yesterday
            )[0]
        )
        print(
            "missing dates: {}".format(
                ",".join(sorted(map(str, missing_dates)))
            )
        )

        available_dates = set(
            get_date(x)
            for x in yt.search(
                root=cube_root,
                path_filter=lambda x: get_date(x) and x.endswith("/sessions"),
            )
            if get_date(x) >= weekago
        )
        print(
            "available dates: {}".format(
                ",".join(sorted(map(str, available_dates)))
            )
        )
        dates = sorted(missing_dates & available_dates)

    if dates:
        chunks = []
        while dates:
            chunk, dates = dates[:3], dates[3:]
            chunks.append(chunk)
        for chunk in chunks:
            print("processing {}".format(chunk))
            process_dates(chunk, cluster, args)
    else:
        print("no dates to process")


if __name__ == "__main__":
    main()
