#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import division
import sys
import os
import json
import argparse
import datetime
from nile.api.v1 import clusters
from yql.api.v1.client import YqlClient
from videolog_common import (
    apply_replacements,
    get_cluster,
    get_driver,
    get_date,
    YqlRunner,
)

cube_root = "//cubes/video-strm"
TITLE = "[VIDEOHUB-936] ORI Stats | YQL"


rory_stub_common = """
use hahn;

$fielddateFormat = DateTime::Format("%Y-%m-%d %H:%M:%S");
$getFielddate = ($ts) -> {
    $ts = CAST($ts as Uint32);
    $tm = AddTimezone(DateTime::FromSeconds($ts), "Europe/Moscow");
    RETURN $fielddateFormat($tm)
};

$date_from = "@[date_from]";
$date_to = "@[date_to]";
"""

rory_stub_uuid = rory_stub_common + """

insert into `@[output_table]` with truncate
select `timestamp`, puid, ip
from range(
    `cubes/video-strm`, $date_from, $date_to, `sessions`
) with columns Struct<puid:String?>
where `UUID` == "@[uuid]"
group by $getFielddate(`timestamp`) as `timestamp`, puid, ip
order by `timestamp`, puid, ip

"""

rory_stub_puid = rory_stub_common + """

insert into `@[output_table]` with truncate
select `timestamp`, "https://frontend.vh.yandex.ru/player/" || `UUID` as player_url
from range(
    `cubes/video-strm`, $date_from, $date_to, `sessions`
) with columns Struct<puid:String?>
where `UUID` is not null and puid == "@[puid]"
group by $getFielddate(`timestamp`) as `timestamp`, `UUID`
order by `timestamp`, player_url

"""


def process_dates(date, args):
    proxy = os.environ["YT_PROXY"]
    yc = YqlClient(db=proxy.lower(), token=os.environ["YQL_TOKEN"])
    yr = YqlRunner(client=yc, title=TITLE)
    output_table = "{}/{}_{}".format(args["root"], args["input_id"], date)
    date_from = max(
        date.replace(year=date.year - 1), datetime.date(2020, 4, 1)
    )
    if args["export_type"] == "by_video_id":
        query = apply_replacements(
            rory_stub_uuid,
            {
                "@[pool]": args["pool"],
                "@[output_table]": output_table,
                "@[date_to]": str(date),
                "@[date_from]": str(date_from),
                "@[uuid]": args["input_id"],
            },
        )
    elif args["export_type"] == "by_puid":
        query = apply_replacements(
            rory_stub_puid,
            {
                "@[pool]": args["pool"],
                "@[output_table]": output_table,
                "@[date_to]": str(date),
                "@[date_from]": str(date_from),
                "@[puid]": args["input_id"],
            },
        )
    yr.run(query, wait=True)
    return output_table


ALLOWED_EXPORT_TYPES = ("by_video_id", "by_puid")


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("--pool", default="robot-mma-nirvana")
    parser.add_argument(
        "--root", default="//home/videoquality/vh_analytics/rory"
    )
    parser.add_argument("--output", default="output.json")
    parser.add_argument("--input_id")
    parser.add_argument("--export_type", default="by_video_id")
    args = vars(parser.parse_args())

    assert args["export_type"] in ALLOWED_EXPORT_TYPES

    cluster = get_cluster(clusters, args)
    yt = get_driver(cluster).client

    last_available_date = max(
        get_date(x)
        for x in yt.search(
            root=cube_root,
            node_type="table",
            path_filter=lambda x: x.endswith("sessions"),
        )
    )
    print("last available date: {}".format(last_available_date))
    output_table = process_dates(last_available_date, args)
    with open(args["output"], "w") as f:
        json.dump(
            {"cluster": os.environ["YT_PROXY"], "table": output_table}, f
        )


if __name__ == "__main__":
    main()
