#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import division
import sys
import os
import argparse
import datetime
from nile.api.v1 import clusters
from yql.api.v1.client import YqlClient
from videolog_common import (
    apply_replacements,
    date_range,
    get_cluster,
    get_driver,
    get_stat_headers,
    get_dates_from_stat,
    get_date,
    StatPusher,
    YqlRunner
)

strm_perf_root = "//logs/strm-perf-log/1d"
TITLE = "Strm Perf Dash | YQL"


def process_dates(dates, cluster, args):
    date_segment = (
        dates[0] if dates[0] == dates[-1] else "{}_{}".format(
            dates[0], dates[-1]
        )
    )
    root = "//home/videoquality/vh_analytics/strm_perf_dash/{}".format(
        date_segment
    )
    tmp_table = "{}/tmp".format(
        root
    )
    report_table = "{}/country_chunk_view_report".format(
        root
    )
    quality_report_table = "{}/quality_chunk_view_dt_report".format(
        root
    )
    proxy = os.environ["YT_PROXY"]

    yc = YqlClient(db=proxy.lower(), token=os.environ["YQL_TOKEN"])
    yr = YqlRunner(client=yc, title=TITLE)
    if args["redo"] or not get_driver(cluster).client.exists(tmp_table):
        with open("strm_perf_dash.sql", "r") as f:
            query = f.read().decode("utf8")
        query = apply_replacements(
            query,
            {
                "@[pool]": args["pool"],
                "@[date_from]": str(dates[0]),
                "@[date_to]": str(dates[-1]),
            },
        )
        yr.run(query)
    with open("country_chunk_view.sql", "r") as f:
        query = f.read().decode("utf8")
    query = apply_replacements(
        query,
        {
            "@[pool]": args["pool"],
            "@[date_from]": str(dates[0]),
            "@[date_to]": str(dates[-1]),
        },
    )
    yr.run(query)
    with open("chunk_view_quality_dt.sql", "r") as f:
        query = f.read().decode("utf8")
    query = apply_replacements(
        query,
        {
            "@[pool]": args["pool"],
            "@[date_from]": str(dates[0]),
            "@[date_to]": str(dates[-1]),
        },
    )
    yr.run(query)
    sp = StatPusher(cluster, report=args["report"], scale="i")
    print("pushing to {}".format(args["report"]))
    sp.push(report_table, replace_mask=("fielddate" if args["redo"] else None))
    sp = StatPusher(cluster, report=args["report_chunk_view_quality"], scale="i")
    print("pushing to {}".format(args["report_chunk_view_quality"]))
    sp.push(quality_report_table, replace_mask=("fielddate" if args["redo"] else None))


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("--pool", default="robot-mma-nirvana")
    parser.add_argument(
        "--report", default="Video/Others/Strm/Stability/Traffic/country_view_type_chunk_type"
    )
    parser.add_argument(
        "--report_chunk_view_quality", default="Video/Others/Strm/Stability/Traffic/view_type_chunk_type_quality_dt"
    )
    parser.add_argument("--from")
    parser.add_argument("--to")
    parser.add_argument("--redo", action="store_true")
    args = vars(parser.parse_args())

    from_ = args.get("from")
    to_ = args.get("to")

    cluster = get_cluster(clusters, args)
    yt = get_driver(cluster).client

    if from_ and to_:
        dates = sorted(date_range(from_, to_))
    else:
        dates_from_stat = get_dates_from_stat(
            report=args["report"], headers=get_stat_headers(), scale="i"
        )
        last_date = dates_from_stat[-1]
        last_available_date = sorted(
            get_date(x) for x in yt.search(
                root=strm_perf_root,
                node_type="table",
                path_filter=lambda x: bool(get_date(x))
            )
        )[-1]
        if last_available_date > last_date:
            dates = sorted(
                date_range(
                    last_date + datetime.timedelta(days=1), last_available_date
                )
            )
        else:
            dates = []

    if dates:
        process_dates(dates, cluster, args)
    else:
        print("no dates to process")


if __name__ == "__main__":
    main()
