#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import division
import os
import argparse
from nile.api.v1 import (
    clusters,
    aggregators as na,
    with_hints,
    extended_schema,
    Record,
)
import nile.files as nf
import itertools
from collections import Counter, defaultdict
try:
    from yql.api.v1.client import YqlClient
except ImportError:
    pass
import codecs
from videolog_common import (
    zen_ref_from_treatment,
    ref_from_treatment,
    wrap_ref_from,
    apply_replacements,
    date_range,
    yt_get_date_from_table as get_date,
    get_dates_from_stat,
    get_stat_headers,
    get_cluster,
    get_driver,
    StatPusher,
    YqlRunner
)


def process_dates(
    dates,
    cluster,
    yql_client,
    report,
    pool,
    replace_mask=None,
    root=None,
    query_file=None,
    skip_push=False,
    redo=False
):
    if len(dates) == 1:
        date_s = str(dates[0])
    else:
        date_s = "{}_{}".format(dates[0], dates[-1])
    date_from = str(min(dates))
    date_to = str(max(dates))
    root_ = "{}/{}".format(root, date_s)
    by_vsid_table = "{}/by_vsid".format(root_)
    report_table = "{}/report".format(root_)
    pairs_report_table = "{}/report_pairs".format(root_)
    unknown_events_table = "{}/unknown_events".format(root_)
    if redo:
        replace_mask = "fielddate"
    yr = YqlRunner(yql_client, title="UnhappyEnd JsTracer | YQL")

    if redo or not (
        get_driver(cluster).exists(report_table)
        and get_driver(cluster).exists(pairs_report_table)
    ):
        with codecs.open(query_file, "r", "utf8") as f:
            query = f.read()
        base_replacements = [
            ("@date_from", date_from),
            ("@date_to", date_to),
            ("@pool", pool),
            ("@unhappyend_report_table", report_table),
            ("@pairs_report_table", pairs_report_table),
            ("@unknown_events_table", unknown_events_table),
            ("@by_vsid_table", by_vsid_table),
            ("--@ref_from", zen_ref_from_treatment),
        ]
        query = apply_replacements(
            query,
            base_replacements
        )
        yr.run(query, attachments=["unhappyend_common.sql"])

    if not skip_push:
        stat_pusher = StatPusher(
            cluster,
            report=report,
            replace_mask=replace_mask,
            remote_publish=True,
        )
        stat_pusher.push(report_table)
        stat_pusher = StatPusher(
            cluster,
            report=report + "_pairs",
            replace_mask=replace_mask,
            remote_publish=True,
        )
        stat_pusher.push(pairs_report_table)


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument(
        "--report", default="Video/Others/Strm/Stability/UnhappyEndJST"
    )
    parser.add_argument("--from", default=None)
    parser.add_argument("--pool", default=None)
    parser.add_argument(
        "--root", default="//home/videoquality/vh_analytics/unhappyend_jst"
    )
    parser.add_argument("--query_file", default="unhappyend_jst.sql")
    parser.add_argument("--debug", action="store_true")
    parser.add_argument("--redo", action="store_true")
    parser.add_argument("--replace_mask", default=None)
    parser.add_argument(
        "--tmp_job_root", default="//home/videoquality/vh_analytics/tmp"
    )
    parser.add_argument("--to", default=None)
    parser.add_argument("--title", default="UnhappyEnd | YQL")
    parser.add_argument("--skip_push", action="store_true")
    parser.add_argument("--skip_report", action="store_true")
    args = parser.parse_args()
    args.templates = {"tmp_root": args.tmp_job_root}

    cluster = get_cluster(clusters, args)
    yql_client = YqlClient(
        db=os.environ["YT_PROXY"], token=os.environ["YQL_TOKEN"]
    )

    from_ = getattr(args, "from")
    to_ = getattr(args, "to")

    if from_ and to_:
        dates = date_range(from_, to_)
    else:
        stat_headers = get_stat_headers()

        last_date_from_stat = get_dates_from_stat(
            headers=stat_headers, report=args.report, dimensions=[]
        )[-1]
        available_dates_preprocessed = sorted(
            get_date(x)
            for x in get_driver(cluster).client.search(
                root="//cubes/video-strm",
                path_filter=(
                    lambda x: get_date(x) and x.endswith("/preprocessed")
                ),
            )
        )
        available_dates_sessions = sorted(
            get_date(x)
            for x in get_driver(cluster).client.search(
                root="//cubes/video-strm",
                path_filter=(
                    lambda x: get_date(x) and x.endswith("/sessions")
                ),
            )
        )
        available_dates = sorted(
            set(available_dates_sessions) & set(available_dates_preprocessed)
        )

        dates = [x for x in available_dates if x > last_date_from_stat]

    print("processing {}".format(dates))
    if dates:
        process_dates(
            dates,
            cluster,
            yql_client,
            args.report,
            args.pool,
            replace_mask=args.replace_mask,
            query_file=args.query_file,
            root=args.root,
            skip_push=args.skip_push,
            redo=args.redo
        )


if __name__ == "__main__":
    main()
