use hahn;
pragma yson.DisableStrict;
pragma AnsiInForEmptyOrNullableItemsCollections;

$channel_data_table = "//home/video-hosting/ugc_replica/ugc_channel";
$meta_data_table = "//home/video-hosting/ugc_replica/ugc_video_meta";
$file_data_table = "//home/video-hosting/ugc_replica/ugc_video_file";
$stream_data_table = "//home/video-hosting/ugc_replica/ugc_video_stream";
$youtube_import_table = "//home/videocontent/transcoder/pg_channels_copy/channels";
$publishers_table = "//home/zen/publishers/backoffice/publishers";
$documents_table = "//home/recommender/zen/dyntables/documents";
$native_ads_table = "//home/zen/nativead/flight_publications";

$_mongo_id_to_id = Python::object_id_hash(
    Callable<(String?)->Int64>, FileContent("ugc_stats.py")
);
$mongo_id_to_id = ($x) -> ($_mongo_id_to_id(String::HexDecode($x)));

$generate_vcid = Python::generate_vcid(
    Callable<(Uint64)->String>, FileContent("ugc_stats.py")
);

$generate_video_meta_id = Python::generate_video_meta_id(
    Callable<(String)->Uint64>, FileContent("ugc_stats.py")
);

$getResource = ($resources, $name) -> {
    $parsed = Yson::ConvertToList(Yson::ParseJson($resources));
    $filtered = ListFilter(
        $parsed,
        ($x)->(Yson::LookupString($x, "name") == $name)
    );
    return IF(
        ListLength($filtered) > 0,
        Yson::LookupString(unwrap($filtered[0]), "value"),
        null
    )
};

define subquery $_get_ugc_meta(
    $channel_data_table,
    $meta_data_table,
    $file_data_table
) as

$youtube_import = (
    select *
    from $youtube_import_table with schema Struct<
        service: String?,
        target: String?,
        `id`: String?,
        platform: String?,
        priority: String?
    >
    where priority != "OFF"
);

$publishers_ = (
    select
        publisherId,
        $mongo_id_to_id(publisherId) as sourceId,
        name as publisherName,
        agencyId,
    from $publishers_table
);
$without_agency = select * from $publishers_ where agencyId is null;
$with_agency = (
    select p.*, a.publisherName as agencyName
    from (select * from $publishers_ where agencyId is not null) as p
    left join any $publishers_ as a on (p.agencyId = a.publisherId)
);
$publishers_rejoin = (
    select * from $with_agency
    union all
    select * from $without_agency
);
$publishers = (
    select p.*, y.target is not null as ugc_yt_imported
    from $publishers_rejoin as p
    left join any (
        select * from $youtube_import
        where service = "ZEN"
    ) as y on (p.publisherId = y.target)
);
$ad_items = (
    select $mongo_id_to_id(publicationMongoId) as itemId
    from $native_ads_table
);

$isVideoShort = ($v) -> {
    $secs = Yson::LookupInt64($v, "durationSeconds");
    $w = Yson::LookupInt64($v, "width");
    $h = Yson::LookupInt64($v, "height");
    return (
        ($w / cast($h as Double)) < 0.6
        and $w >= 480
        and $secs <= 60
    )
};

$documents_ = (
    select
        `id`,
        sourceId,
        Yson::LookupString(videoSpecificInfo, "uuid") as `uuid`,
        $isVideoShort(videoSpecificInfo) as isVideoShort
    from $documents_table
    where Yson::LookupString(videoSpecificInfo, "uuid") is not null
);
$documents = (
    select
        d.*, publisherId, publisherName, agencyId, agencyName, ugc_yt_imported, a.itemId is not null as is_native_ad
    from $documents_ as d
    left join any $publishers as p on (d.sourceId = p.sourceId)
    left join any $ad_items as a on (d.`id` = a.itemId)
);

$file_data = (
    select
        `video_file.id` as video_file_id,
        Yson::LookupInt64(Yson::ParseJson(`video_file.transcoder_info`), "DurationMs") / 1000 as video_duration,
        `video_file.service` as service
    from $file_data_table
);

$stream_data = (
    select
        `video_stream.id` as video_stream_id,
        `video_stream.service` as service,
        cast(`video_stream.start_time` as Uint64) as live_start_time,
        cast(`video_stream.finish_time` as Uint64) as live_finish_time,
    from $stream_data_table
);

$channel_data = (
    select
        `channel.id` as channel_id,
        `channel.title` as channel_name,
        `channel.privacy` as channel_privacy,
        `channel.user_uid` as owner_id,
        `channel.service` as channel_service,
        $getResource(`extra.resources`, "page_id") as channel_page_id,
        y.target is not null as ugc_yt_imported,
    from $channel_data_table as ch
    left join any (
        select cast(target as Uint64) as target
        from $youtube_import
        where service = "UGC" and cast(target as Uint64) is not null
    ) as y on (ch.`channel.id` == y.target)
);

$meta_data = (
    select
        `video_meta.id` as video_id,
        `video_meta.video_file_id` as video_file_id,
        `video_meta.moderation_info` as moderation_info,
        `video_meta.moderation_status` as moderation_status,
        `video_meta.deleted` as deleted,
        `video_meta.title` as video_title,
        `video_meta.privacy` as privacy,
        `video_meta.release_date` as video_release_date,
        `video_meta.create_time` as video_create_time,
        `video_meta.update_time` as video_update_time,
        `video_meta.video_stream_id` as video_stream_id,
        $generate_vcid(`video_meta.id`) as video_content_id,
        `video_meta.channel_id` as channel_id,
        $getResource(`extra.resources`, "page_id") as page_id,
    from $meta_data_table
);

$joined_channel_meta = (
    select
        owner_id,
        channel_name, channel_service, channel_privacy,
        d.ugc_yt_imported ?? c.ugc_yt_imported as ugc_yt_imported,
        page_id ?? channel_page_id as page_id,
        publisherId,
        publisherName,
        agencyId,
        agencyName,
        d.`id` as itemId,
        d.isVideoShort as isVideoShort,
        sourceId,
        is_native_ad,
        m.* without m.page_id
    from $meta_data as m
    inner join any $channel_data as c on (m.channel_id == c.channel_id)
    left join any $documents as d on (d.`uuid` = m.video_content_id)
);

$video_file_id_null = select * from $joined_channel_meta where video_file_id is null;
$video_file_id_not_null = select * from $joined_channel_meta where video_file_id is not null;

$join_file_service = (
    select s.*, `service`, video_duration
    from $video_file_id_not_null as s
    left join any $file_data as f using (video_file_id)
);
$after_file_join = select * from $join_file_service union all select * from $video_file_id_null;

$video_stream_id_null = select * from $after_file_join where video_stream_id is null;
$video_stream_id_not_null = select * from $after_file_join where video_stream_id is not null;

$join_stream_service = (
    select f.service ?? s.service as service,
    live_start_time,
    live_finish_time,
    s.video_duration ?? (live_finish_time - live_start_time) as video_duration,
    s.* without s.service, s.video_duration
    from $video_stream_id_not_null as s
    left join any $stream_data as f using (video_stream_id)
);
$after_stream_join = select * from $join_stream_service union all select * from $video_stream_id_null;

select * from $after_stream_join
where service is not null
and (video_file_id is not null or video_stream_id is not null);
end define;

define subquery $get_ugc_meta() as
select * from $_get_ugc_meta(
    $channel_data_table,
    $meta_data_table,
    $file_data_table
);
end define;

export $get_ugc_meta, $_get_ugc_meta, $getResource,
    $generate_vcid, $generate_video_meta_id, $mongo_id_to_id;
