#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import division
import sys
import os
import json
import argparse
import datetime
from nile.api.v1 import clusters
from yql.api.v1.client import YqlClient
from videolog_common import (
    apply_replacements,
    date_range,
    get_cluster,
    get_driver,
    _check_non_empty,
    get_date,
    YqlRunner,
)

TITLE = "Sender Query | YQL"
INPUT_ROOT_DEFAULT = "//home/videoquality/vh_analytics/ugc_stats"
OUTPUT_ROOT_DEFAULT = "//home/videolog/ugc_stats_by_channel"
ALREADY_SENT_TABLE = "//home/videoquality/vh_analytics/ugc_monetization_already_sent/already_sent"
TO_SEND_ROOT = "//home/videoquality/vh_analytics/ugc_monetization_to_send"
DATE_QUERY_STUB = """select * from `@[table]`"""


def process_dates(date_from, date_to, args, yr):
    with open("sender_delta_query.sql", "rb") as f:
        query = f.read().decode("utf8")
    query = apply_replacements(
        query,
        {
            "@[pool]": args["pool"],
            "@[input_root]": args["input_root"],
            "@[output_root]": args["output_root"],
            "@[to_send_root]": args["to_send_root"],
            "@[already_sent_table]": args["already_sent_table"],
            "@[date_from]": str(date_from),
            "@[date_to]": str(date_to),
        },
    )
    yr.run(
        query,
        wait=True,
        attachments=[
            {
                "path": "analytics/videolog/strm-stats/strm_cube_2/ugc_stats_regular/get_ugc_meta.sql"
            }
        ],
    )


def _check_non_empty(yt, table):
    return yt.exists(table) and yt.get_attribute(table, "row_count") > 0


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("--pool", default="robot-mma-nirvana")
    parser.add_argument("--input_root", default=INPUT_ROOT_DEFAULT)
    parser.add_argument("--output_root", default=OUTPUT_ROOT_DEFAULT)
    parser.add_argument("--already_sent_table", default=ALREADY_SENT_TABLE)
    parser.add_argument("--to_send_root", default=TO_SEND_ROOT)
    parser.add_argument("--input_date")
    parser.add_argument("--output_mr_table")
    parser.add_argument("--redo", action="store_true")
    args = vars(parser.parse_args())

    if os.path.isfile(args["input_date"]):
        with open(args["input_date"]) as f:
            input_dates = json.load(f)
        date_to = max(input_dates)
    else:
        date_to = args["input_date"]
    date_to = get_date(date_to)

    proxy = os.environ["YT_PROXY"]
    yc = YqlClient(db=proxy.lower(), token=os.environ["YQL_TOKEN"])
    yr = YqlRunner(client=yc, title=TITLE)

    date_from_query = DATE_QUERY_STUB.replace(
        "@[table]", args["output_root"] + "/last_date"
    )
    date_req = yr.run(date_from_query)
    date_from = get_date(
        yr.get_results(date_req)[0]["fielddate"].decode("utf8")
    ) + datetime.timedelta(days=1)
    assert date_from <= date_to

    process_dates(date_from, date_to, args, yr)

    with open(args["output_mr_table"], "w") as f:
        f.write(
            json.dumps(
                {
                    "propertiesJson": json.dumps(
                        {
                            "table_path": "{}/{}".format(
                                args["to_send_root"], date_to
                            )
                        }
                    )
                }
            )
        )


if __name__ == "__main__":
    main()
