#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import division
import sys
import os
import codecs
import argparse
import json
from nile.api.v1 import clusters
from yql.api.v1.client import YqlClient
from videolog_common import (
    get_cluster,
    get_driver,
    YqlRunner,
    get_date,
    apply_replacements,
)

ROOT = "//home/videolog/strm_meta/user_subscriptions"
TITLE = "User Subscriptions v2 | YQL"


def get_last_date(yr, args):
    table = "{}/latest_subscriptions_last_date".format(args.root)
    query = "use hahn;\nselect last_date from `@[table]` limit 1".replace(
        "@[table]", table
    )
    req = yr.run(query)
    res = yr.get_results(req)
    return get_date(res[0]["last_date"])


def set_last_date(yr, args, date):
    table = "{}/latest_subscriptions_last_date".format(args.root)
    query = 'use hahn;\ninsert into `@[table]` with truncate select "@[date]" as last_date'
    query = apply_replacements(
        query, {
            "@[table]": table,
            "@[date]": str(date)
        }
    )
    yr.run(query)


def process_date(date, yr, args):
    with open("user_subscriptions_stub.sql", "r") as f:
        query = f.read()
    input_table = "{}/parsed_profiles/{}".format(args.root, date)
    output_table = "{}/latest_subscriptions".format(args.root)
    query = apply_replacements(
        query,
        {
            "@[input_table]": input_table,
            "@[output_table]": output_table,
            "@[pool]": args.pool,
        },
    )
    yr.run(query)
    set_last_date(yr, args, date)


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("--date")
    parser.add_argument("--pool", default="robot-mma-nirvana")
    parser.add_argument("--root", default=ROOT)
    args = parser.parse_args()

    yc = YqlClient(token=os.environ["YQL_TOKEN"])
    yr = YqlRunner(yc, title=TITLE)

    cluster = get_cluster(clusters, {})
    yt = get_driver(cluster).client

    if not args.date:
        yc.config.db = None
        last_date = get_last_date(yr, args)
        profiles_root = "{}/parsed_profiles".format(args.root)
        available_dates = sorted(
            get_date(x)
            for x in yt.search(root=profiles_root, node_type="table")
            if get_date(x) and get_date(x) > last_date
        )
        if not available_dates:
            print("No dates available, last date is {}".format(last_date))
            sys.exit(0)
        date = available_dates[-1]
    else:
        date = get_date(args.date)
    print("Processing {}...".format(date))
    process_date(date, yr, args)


if __name__ == "__main__":
    main()
