/* global Router */
function Viewer() {
    this.root = '/results/video/';
    this.rootViewer = '/results/video/viewer/';

    this.channelListArea = $('.channels-list');
    this.programsArea = $('.programs');

    this.channels = [];
    this.views = [];

    this.currDimensions = {
        from: ['_total_'],
        viewType: ['_total_'],
        metric: ['sessions', 'duration', 'sessions_durations', 'density']
    };

    this.bindProgramVisualHint();

    this.densityLimit = 0.05;

    this.viewPercentiles = [20, 50, 80, 95];
    this.currViewPercentileIndex = 1;

    this.orderColumns = ['time', 'sessions', 'duration', 'viewtime'];

    this.lastDate = null;

    this.state = {
        dimensions: {},
        limits: {}
    };
}

Viewer.prototype.setState = function(state) {
    this.lastDate = this.state.date ? this.state.date : state.date;
    this.state.date = state.date;

    this.state.channel = state.channel;

    if (this.currDimensions.from.indexOf(state.dimensions.from) > -1) {
        this.state.dimensions.from = state.dimensions.from;
    }

    if (this.currDimensions.viewType.indexOf(state.dimensions.viewType) > -1) {
        this.state.dimensions.viewType = state.dimensions.viewType;
    }

    if (this.currDimensions.metric.indexOf(state.dimensions.metric) > -1) {
        this.state.dimensions.metric = state.dimensions.metric;
    }

    if (this.viewPercentiles.indexOf(state.viewPercentile) > -1) {
        this.state.viewPercentile = state.viewPercentile;
    }

    if (this.orderColumns.indexOf(state.orderBy) > -1) {
        this.state.orderBy = state.orderBy;
    }

    this.state.topLimit = state.topLimit;

    this.state.limits.sessions = state.limits.sessions;
    this.state.limits.viewTime = state.limits.viewTime;
};

Viewer.prototype.getViewPercentileIndex = function() {
    return this.viewPercentiles.indexOf(this.state.viewPercentile);
};

Viewer.prototype.getState = function() {
    return this.state;
};

Viewer.prototype.getDefaultState = function() {
    var date = $('.datepicker').datepicker('getDate');

    date.setHours(12);
    date = new Date(date - 86400 * 1000);
    
    return {
        date: $.datepicker.formatDate('yy-mm-dd', date),
        channel: '_total_',
        dimensions: {
            from: this.currDimensions.from[0],
            viewType: this.currDimensions.viewType[0],
            metric: this.currDimensions.metric[0]
        },
        viewPercentile: this.viewPercentiles[1],
        orderBy: this.orderColumns[0],
        limits: {
            sessions: 0,
            viewTime: 60
        }
    };
};

Viewer.prototype.getData = function() {
    var self = this;

    $.when(
        this.getDataFile('programs'), this.getDataFile('views'), this.getDataFile('report')
    ).done(function(channels, views, report) {
        self.showChannelsList(channels, views, report);
    });

    this.programsArea.html('');
    this.channelListArea.html('<div class="channellist-item__empty">Список каналов...</div>');
};

Viewer.prototype.getDataFile = function(fileName) {
    var promise = $.Deferred();

    $.get(
        this.root + this.state.date + '/' + fileName + '.json', {dataType: 'json'}
    ).then(function(result) {
            promise.resolve(result);
        }, function() {
            promise.resolve(null);
        }
    );

    return promise;
};

Viewer.prototype.showChannelsList = function(channels, views, report) {
    if (!(channels && Object.keys(channels).length)) {
        this.channelListArea.html('<div class="channellist-item__empty">No channels found</div>');

        return;
    }

    this.views = views;
    this.channels = channels;

    this.addViewsIndex();

    var reportHoursChannel = report && report.by_hour_viewer ? report.by_hour_viewer.map(function(item) {
        // 2017-02-01|00
        // 2017-02-01|00|1tv
        // 2017-02-01|00|1tv|morda
        
        var parts = item.path_tree.split('|'),
            hour = parts[1],
            channel,
            from;

        if (parts.length === 2) {
            channel = '_total_';
            from = '_total_';
        } else if (parts.length === 3) {
            channel = parts[2];
            from = '_total_';
        } else {
            channel = parts[2];
            from = parts[3];
        }

        item['_hour'] = hour;
        item['_channel'] = channel;
        item['_ref_from'] = from;
        item['_view_type'] = '_total_';

        return item;
    }) : [];

    var reportHoursFrom = report && report.by_hour_ref_from_viewer ? report.by_hour_ref_from_viewer.map(function(item) {
        // 2017-02-01|00|morda
        
        var parts = item.path_tree.split('|'),
            hour = parts[1],
            from = parts[2];

        item['_hour'] = hour;
        item['_channel'] = '_total_';
        item['_ref_from'] = from;
        item['_view_type'] = '_total_';

        return item;
    }) : [];

    this.reportHours = [];

    if (reportHoursFrom.length && reportHoursChannel.length) {
        this.reportHours = reportHoursFrom.concat(reportHoursChannel);
    }

    var reportChannelsFrom = report && report.by_channel_ref_from_viewer ? report.by_channel_ref_from_viewer.map(function(item) {
        // 2017-02-04
        // 2017-02-04|1tv
        // 2017-02-04|1tv|morda
        // 2017-02-04|1tv|morda|live

        var parts = item.path_tree.split('|'),
            channel = '_total_',
            from = '_total_',
            viewType = '_total_';

        if (parts.length === 2) {
            channel = parts[1];
        } else if (parts.length === 3) {
            channel = parts[1];
            from = parts[2];
        } else {
            channel = parts[1];
            from = parts[2];
            viewType = parts[3];
        }

        if (viewType === 'other' && channel === '1tv') {
            viewType = 'live';
        }

        item['_channel'] = channel;
        item['_ref_from'] = from;
        item['_view_type'] = viewType;

        return item;
    }) : [];

    var reportChannelsViewType = report && report.by_channel_view_type_viewer ? report.by_channel_view_type_viewer.map(function(item) {
        // 2017-02-04|1tv|live

        var parts = item.path_tree.split('|'),
            channel = parts[1],
            from = '_total_',
            viewType = parts[2];

        if (parts.length === 2) {
            channel = parts[1];
        }

        if (viewType === 'other' && channel === '1tv') {
            viewType = 'live';
        }

        item['_channel'] = channel;
        item['_ref_from'] = from;
        item['_view_type'] = viewType;

        return item;
    }) : [];

    this.reportChannels = [];

    if (reportChannelsFrom.length && reportChannelsViewType.length) {
        this.reportChannels = reportChannelsFrom.concat(reportChannelsViewType);
    }

    var html = '',
        channelTimes = this.getAllViewsAndChannels().channels;

    for (var channel in this.channels) {
        if (this.channels.hasOwnProperty(channel) && channelTimes.hasOwnProperty(channel)) {
            html += this.drawChannelListItem(channel, this.channels[channel]);
        }
    }

    this.channelListArea.html(html);

    this.attachListEvents();

    this.getCurrDimensions();

    this.getProgramViews();
};

Viewer.prototype.getProgramViews = function() {
    $('.channellist-item.active').removeClass('active');
    $('.channellist-item[data-channel="' + this.state.channel + '"]').addClass('active');

    $('.view_percentile').html(this.state.viewPercentile);

    this.showViews();
};

Viewer.prototype.drawChannelListItem = function(channel, info) {
    return '<div class="channellist-item" data-channel="' + channel +
            '">' +
                '<div class="channellist-item__wrap">' +
                    '<div class="channellist-item__bg"></div>' +
                '</div>' +
                '<div class="channellist-item__text">' + info.info.title + '</div>' +
            '</div>';
};

Viewer.prototype.drawChannelListItemStats = function(times, totals) {
    var duration = 100 * times.duration / totals.duration,
        sessions = 100 * times.sessions / totals.sessions,
        viewTime = times.viewTime ? 100 * times.viewTime / totals.viewTime : 0;

    return '<div data-stat="' + times.sessions + '" class="channellist-item__sessions" style="width:' + sessions + '%"></div>' +
            '<div data-stat="' + times.duration + '" class="channellist-item__duration" style="width:' + duration + '%"></div>' +
            '<div data-stat="' + times.viewTime + '" class="channellist-item__viewtime" style="width:' + viewTime + '%"></div>';
};

Viewer.prototype.addChannelStat = function(channelStats) {
    var maxChannelStats = {
            sessions: 0,
            duration: 0,
            viewTime: 0
        },
        stats,
        self = this;

    for (var channel in channelStats) {
        if (channelStats.hasOwnProperty(channel)) {
            stats = channelStats[channel];

            if (maxChannelStats.sessions < stats.sessions) {
                maxChannelStats.sessions = stats.sessions;
            }

            if (maxChannelStats.duration < stats.duration) {
                maxChannelStats.duration = stats.duration;
            }

            if (maxChannelStats.viewTime < stats.viewTime) {
                maxChannelStats.viewTime = stats.viewTime;
            }
        }
    }

    $('.channellist-item').each(function() {
        var channel = $(this).data('channel'),
            statsHtml = self.drawChannelListItemStats(channelStats[channel], maxChannelStats);

        $('[data-channel="' + channel + '"]', '.channels-list').find('.channellist-item__sessions, .channellist-item__duration, .channellist-item__viewtime').remove();
        $('[data-channel="' + channel + '"] .channellist-item__wrap', '.channels-list').append(statsHtml);
    });

    $('.channellist-item .stream__stats-hint').remove();
};

Viewer.prototype.addDimensionOptions = function() {
    var options,
        name,
        selected,
        i, l,
        self = this;

    this.programsArea.append('<div class="programs-viewtype">' +
                                    '<span class="dimension-label">Площадка</span><select data-dimension="from" class="dimensions-from"></select>' +
                                    '<span class="dimension-label">Тип просмотра</span><select data-dimension="viewType" class="dimensions-viewtype"></select>' +
                                    '<span class="dimension-label">Метрика</span><select data-dimension="metric" class="dimensions-metric"></select>' +
                                '</div>');

    for (var dimensionName in this.currDimensions) {
        if (this.currDimensions.hasOwnProperty(dimensionName)) {
            options = [];

            for (i = 0, l = this.currDimensions[dimensionName].length; i < l; i++) {
                name = this.currDimensions[dimensionName][i];
                selected = '';

                if (name === '_total_') {
                    name = 'Всего';
                }

                if (name === 'duration') {
                    name = 'Общее время просмотра';
                }

                if (name === 'sessions') {
                    name = 'Сессии';
                }

                if (name === 'sessions_durations') {
                    name = 'Сессии / Время просмотра';
                }

                if (name === 'density') {
                    name = 'Распределение времени';
                }

                if (this.currDimensions[dimensionName][i] === this.state.dimensions[dimensionName]) {
                    selected = ' selected="selected"';
                }

                options.push('<option value="' + this.currDimensions[dimensionName][i] + '"' + (selected) + '>' + name + '</option>');
            }

            $(('select.dimensions-' + dimensionName).toLowerCase(), this.programsArea).html(options.join(''));
        }
    }

    $('select[data-dimension]', this.programsArea).on('change', function() {
        var dimension = $(this).data('dimension');

        self.state.dimensions[dimension] = this.value;

        self.navigate();
    });
};

Viewer.prototype.getCurrDimensions = function() {
    var dimensions = {
        from: {
            '_total_': true
        },
        viewType: {
            '_total_': true
        }
    };

    for (var date in this.views) {
        if (this.views.hasOwnProperty(date)) {
            var dateViews = this.views[date].views;

            for (var i = 0, l = dateViews.length; i < l; i++) {
                dimensions.from[dateViews[i].ref_from] = true;
                dimensions.viewType[dateViews[i].view_type] = true;
            }
        }
    }

    this.currDimensions.from = Object.keys(dimensions.from);
    this.currDimensions.viewType = Object.keys(dimensions.viewType);
};

Viewer.prototype.getViewsByChannel = function(channel) {
    var viewsOfChannel = [];

    for (var date in this.views) {
        if (this.views.hasOwnProperty(date)) {
            var dateViews = this.views[date].views;

            for (var i = 0, l = dateViews.length; i < l; i++) {
                if (dateViews[i].channel === channel) {
                    viewsOfChannel.push({item: dateViews[i], date: date});
                }
            }
        }
    }

    return viewsOfChannel;
};

Viewer.prototype.getAllViewsAndChannels = function() {
    var views = [],
        channel,
        channels = {};

    for (var date in this.views) {
        if (this.views.hasOwnProperty(date)) {
            var dateViews = this.views[date].views;

            for (var i = 0, l = dateViews.length; i < l; i++) {
                views.push({item: dateViews[i], date: date});

                channel = dateViews[i].channel;

                if (!channels[channel]) {
                    channels[channel] = 1;
                }
            }
        }
    }

    var reportChannels = {};

    if (this.reportChannels.length) {
        this.reportChannels.map(function(item) {
            if (item._channel !== '_total_') {
                reportChannels[item._channel] = 1;
            }
        });
    }

    return {views: views, channels: this.reportChannels.length ? reportChannels : channels};
};

Viewer.prototype.getViewsByChannelFilter = function(channel, filter) {
    return this.getViewsByChannel(channel).filter(function(item) {
        for (var prop in filter) {
            if (filter.hasOwnProperty(prop)) {
                if (filter[prop] !== item.item[prop]) {
                    return false;
                }
            }
        }

        return true;
    });
};

Viewer.prototype.getViewsFilter = function(views, filter) {
    return views.filter(function(item) {
        for (var prop in filter) {
            if (filter.hasOwnProperty(prop)) {
                if (filter[prop] !== item.item[prop]) {
                    return false;
                }
            }
        }

        return true;
    });
};

Viewer.prototype.getReportHoursFilter = function(channel, filter) {
    filter.channel = channel;

    return this.reportHours.filter(function(item) {
        for (var prop in filter) {
            if (filter.hasOwnProperty(prop)) {
                if (filter[prop] !== item['_' + prop]) {
                    return false;
                }
            }
        }

        return true;
    });
};

Viewer.prototype.getReportChannelsFilter = function(channel, filter) {
    filter.channel = channel ? channel : '_total_';

    return this.reportChannels.filter(function(item) {
        for (var prop in filter) {
            if (filter.hasOwnProperty(prop)) {
                if (filter[prop] !== item['_' + prop]) {
                    return false;
                }
            }
        }

        return true;
    });
};

Viewer.prototype.addViewsIndex = function() {
    for (var date in this.views) {
        if (this.views.hasOwnProperty(date)) {
            var dateViews = this.views[date].views;

            for (var i = 0, l = dateViews.length; i < l; i++) {
                this.views[date].views[i]._index = i;
            }
        }
    }
};

Viewer.prototype.getFilterParams = function() {
    var filterParams = {},
        reportFilterParams = {};

    if (this.state.dimensions.from !== '_total_') {
        filterParams.ref_from = this.state.dimensions.from;
        reportFilterParams.ref_from = this.state.dimensions.from;
    } else {
        reportFilterParams.ref_from = '_total_';
    }

    if (this.state.dimensions.viewType !== '_total_') {
        filterParams.view_type = this.state.dimensions.viewType;
        reportFilterParams.view_type = this.state.dimensions.viewType;
    } else {
        reportFilterParams.view_type = '_total_';
    }

    return {filterParams: filterParams, reportFilterParams: reportFilterParams};
};

Viewer.prototype.showViews = function() {
    this.programsArea.html('');
    this.addDimensionOptions();

    var filterParams = this.getFilterParams();

    var filteredPrograms = this.getViewsByChannelFilter(this.state.channel, filterParams.filterParams),
        filteredReportsHours = this.getReportHoursFilter(this.state.channel, filterParams.reportFilterParams),
        channelStats = {},
        self = this;

    $('.channellist-item').each(function() {
        var channel = $(this).data('channel');

        if (self.reportChannels.length) {
            channelStats[channel] = self.getReportChannelStats(self.getReportChannelsFilter(channel, filterParams.reportFilterParams));
        } else {
            channelStats[channel] = self.getProgramMetricStats(null, null, self.getViewsByChannelFilter(channel, filterParams.filterParams));
        }
    });

    //console.log(filteredPrograms, filteredPrograms.length);

    this.showReportPrograms(filteredReportsHours);

    if (this.state.channel === '_total_') {
        this.showTopViews();
    } else {
        this.showViewsPrograms(filteredPrograms);
    }

    this.addChannelStat(channelStats);
};

Viewer.prototype.getReportChannelStats = function(filteredReportsChannels) {
    if (filteredReportsChannels.length) {
        return {
            sessions: filteredReportsChannels[0].count,
            duration: filteredReportsChannels[0].view_total_time,
            viewTime: filteredReportsChannels[0].view_percentiles[this.getViewPercentileIndex()]
        };
    }

    return {
        sessions: 0,
        duration: 0,
        viewTime: 0
    };
};

Viewer.prototype.showReportPrograms = function(filteredReportsHours) {
    if (filteredReportsHours.length < 1) {
        return;
    }

    filteredReportsHours = filteredReportsHours.sort(function(a, b) {
        return a._hour - b._hour;
    });

    this.programsArea.append('<div id="channel-hours" class="channel-hours"></div>');

    var dataCount = [],
        dataTotalTime = [],
        dataViewTime = [];

    for (var i = 0, l = filteredReportsHours.length; i < l; i++) {
        dataCount.push(filteredReportsHours[i].count || 0);
        dataViewTime.push(filteredReportsHours[i].view_percentiles[this.getViewPercentileIndex()] || 0);
        dataTotalTime.push(filteredReportsHours[i].view_total_time || 0);
    }

    var self = this,
        chart = new Highcharts.Chart({
        chart: {
            renderTo: 'channel-hours',
            defaultSeriesType: 'column',
            backgroundColor: '#fff',
            borderWidth: 0,
            borderColor: '#ccc',
            plotBackgroundColor: '#fff',
            plotBorderWidth: 1,
            plotBorderColor: '#ccc'
        },
        credits: {enabled: false},
        exporting: {enabled: false},
        title: {
            align: 'left',
            text: 'Просмотры по часам',
            y: 5
        },
        rangeSelector: {
            enabled: false
        },
        tooltip: {
            borderWidth: 1,
            formatter: function() {
                return '<span style="visibility:hidden;"></span><span>Сессии </span><b>' + this.points[0].y + '</b></span><br/>' +
                        '<span style="visibility:hidden;"></span><span>Общее время просмотра </span><b>' + self.getProgramDuration(this.points[1].y) + '</b></span><br/>' +
                        '<span style="visibility:hidden;"></span><span>' + self.state.viewPercentile + '% времени просмотра </span><b>' + self.getProgramDuration(this.points[2].y, true) + '</b></span>';
            },
            shared: true,
            followPointer: true
        },
        plotOptions: {
            column: {
                //stacking: 'normal',
                //grouping: false,
                shadow: false,
                borderWidth: 0,
                borderColor: '#666',
                pointPadding: 0,
                groupPadding: 0.1,
                animation: false
            }
        },
        xAxis: {
            min: 0,
            tickInterval: 1,
            max: 23
        },
        yAxis: [{
            title: {text: 'Число сессий'},
            gridLineColor: '#e9e9e9',
            tickWidth: 1,
            tickLength: 3,
            tickColor: '#ccc',
            lineColor: '#ccc'
        }, {
            title: {text: 'Общее время просмотра'},
            gridLineColor: '#e9e9e9',
            tickWidth: 1,
            tickLength: 3,
            tickColor: '#ccc',
            lineColor: '#ccc'
        }, {
            title: {text: self.state.viewPercentile + '% времени просмотра, сек'},
            gridLineColor: '#e9e9e9',
            opposite: true,
            tickWidth: 1,
            tickLength: 3,
            tickColor: '#ccc',
            lineColor: '#ccc'
        }],
        series: [{
            data: dataCount,
            color: 'rgba(0,0,255,.5)',
            name: 'Число сессий'
        }, {
            data: dataTotalTime,
            color: 'rgba(255,51,0,.5)',
            name: 'Общее время просмотра',
            yAxis: 1
        }, {
            data: dataViewTime,
            color: 'rgba(115,204,102,1)',
            name: self.state.viewPercentile + '% времени просмотра, сек',
            yAxis: 2
        }]
    });
};

Viewer.prototype.showViewsPrograms = function(filteredPrograms) {
    if (!this.channels[this.state.channel]) {
        return;
    }

    var maxProgramStats = {
            max: {
                sessions: 0,
                duration: 0,
                viewTime: 0
            }
        },
        maxProgramDensity = 0,
        programDensity,
        programStats,
        programs = this.channels[this.state.channel].programms,
        i, l,
        candidates = [];

    for (i = 0, l = programs.length; i < l; i++) {
        programStats = this.getProgramMetricStats(programs[i], null, filteredPrograms);
        programDensity = this.getProgramVisualDensityStats(programs[i], filteredPrograms).max;

        if (maxProgramStats.max.sessions < programStats.sessions) {
            maxProgramStats.max.sessions = programStats.sessions;
        }

        if (maxProgramStats.max.duration < programStats.duration) {
            maxProgramStats.max.duration = programStats.duration;
        }

        if (maxProgramStats.max.viewTime < programStats.viewTime) {
            maxProgramStats.max.viewTime = programStats.viewTime;
        }

        if (maxProgramDensity < programDensity) {
            maxProgramDensity = programDensity;
        }

        candidates.push({
            channel: this.state.channel,
            program: programs[i],
            stats: programStats
        });
    }

    candidates = this.filterPrograms(candidates);

    this.orderPrograms(candidates);

    var html = '';

    html += this.drawViewsProgramControl();
    html += this.drawViewsProgramHead(filteredPrograms, programStats);

    for (i = 0, l = candidates.length; i < l; i++) {
        html += this.drawViewsProgram(candidates[i].program, filteredPrograms, maxProgramStats, maxProgramDensity);
    }

    html = '<table class="stream__schedule-table"><tbody>' + html + '</tbody></table>';

    this.drawViewsProgramFilter();

    this.programsArea.append(html);

    this.addControlEvents();

    this.addProgramEvents();

    this.updateBodyProgramHintClass();
};

Viewer.prototype.drawViewsProgramHead = function(filteredPrograms) {
    return '<tr class="stream__schedule-item">' +
                '<td class="stream__time"></td>' +
                '<td class="stream__title"><div class="stream__title_wrap">' +
                    '<span class="stream__title-program stream__schedule-item-title">Всего</span>' +
                '</div></td>' +
                this.getProgramMetricHtml(null, filteredPrograms) +
                '<td></td>' +
            '</tr>';
};

Viewer.prototype.drawViewsProgramFilter = function() {
    var options = [],
        name,
        selected,
        self = this;

    this.programsArea.append('<div class="programs-filter">' +
                                    '<span class="filter-label">Время просмотра</span><select class="filter-percentile"></select>' +
                                    '<span class="filter-label">Сессий больше</span><input class="filter-sessions"></input>' +
                                    '<span class="filter-label">Время просмотра больше, сек</span><input class="filter-viewtime"></input>' +
                                    '<span class="filter-reload">Обновить</span>' +
                                '</div>');

    for (var i = 0, l = this.viewPercentiles.length; i < l; i++) {
        name = this.viewPercentiles[i];
        selected = '';

        if (name === this.state.viewPercentile) {
            selected = ' selected="selected"';
        }

        options.push('<option value="' + name + '"' + (selected) + '>' + name + '%</option>');
    }

    $('select.filter-percentile', this.programsArea).html(options.join(''));
    $('.filter-sessions', this.programsArea).val(this.state.limits.sessions);
    $('.filter-viewtime', this.programsArea).val(this.state.limits.viewTime);

    $('select.filter-percentile', this.programsArea).on('change', function() {
        var percentile = Number(this.value);

        self.state.viewPercentile = percentile;

        self.navigate();
    });

    $('.filter-reload').click(function() {
        self.state.limits.sessions = Number($('.filter-sessions', this.programsArea).val());
        self.state.limits.viewTime = Number($('.filter-viewtime', this.programsArea).val());

        self.navigate();
    });
};

Viewer.prototype.drawViewsProgramControl = function() {
    return '<tr class="stream__schedule-item">' +
                (this.state.channel === '_total_' ? '<td class="stream__time"></td>' : '') +
                (this.state.channel === '_total_' ? '<td class="stream__time"></td>' : '<td data-order="time" class="stream__control stream__time"><div class="stream__control_area"><div class="legend-item__line legend-item__line-starttime"></div></div></td>') +
                '<td class="stream__title"><div class="stream__title_wrap">' +
                    '<span class="stream__title-program stream__schedule-item-title"></span>' +
                '</div></td>' +
                this.getProgramMetricControl() +
                '<td></td>' +
            '</tr>';
};

Viewer.prototype.drawViewsProgram = function(program, filteredPrograms, totalStats, maxProgramDensity) {
    return '<tr class="stream__schedule-item" data-contentid="' + program.content_id + '">' +
                '<td class="stream__time">' + this.getProgramStartTime(program) + '</td>' +
                '<td class="stream__title"><div class="stream__title_wrap">' +
                    '<span class="stream__title-program stream__schedule-item-title">' + program.program_title + '</span>' +
                '</div></td>' +
                this.getProgramMetricHtml(program, filteredPrograms) +
                this.getProgramVisualStats(program, filteredPrograms, totalStats, maxProgramDensity) +
            '</tr>';
};

Viewer.prototype.drawTopProgram = function(channel, program, filteredPrograms) {
    return '<tr class="stream__schedule-item" data-contentid="' + program.content_id + '">' +
                '<td class="stream__channel">' + channel + '</td>' +
                '<td class="stream__time">' + this.getProgramStartTime(program) + '</td>' +
                '<td class="stream__title"><div class="stream__title_wrap">' +
                    '<span class="stream__title-program stream__schedule-item-title">' + program.program_title + '</span>' +
                '</div></td>' +
                this.getProgramMetricHtml(program, filteredPrograms) +
            '</tr>';
};

Viewer.prototype.addProgramEvents = function() {

};

Viewer.prototype.getProgramStartTime = function(program) {
    var date = new Date(program.start_time * 1000),
        hours = date.getHours(),
        minutes = date.getMinutes();

    return hours + ':' + (minutes < 10 ? '0' + minutes : minutes);
};

Viewer.prototype.getProgramDuration = function(sec, nonZeroFlag) {
    var hours = Math.floor(sec / 3600),
        minutes = Math.floor((sec - (hours * 3600)) / 60),
        seconds = Math.floor(sec - (hours * 3600) - (minutes * 60));

    if (hours < 10) {
        if (nonZeroFlag && !hours) {
            hours = '';
        } else {
            hours = '0' + hours;
        }
    }
    if (minutes < 10) {
        minutes = '0' + minutes;
    }
    if (seconds < 10) {
        seconds = '0' + seconds;
    }

    return (hours ? hours + ':' : '') + (minutes ? minutes + ':' : '') + seconds;
};

Viewer.prototype.getProgramMetricStats = function(program, date, filteredPrograms, addItems) {
    var sessions = 0,
        duration = 0,
        viewTime = 0,
        viewTimeCount = 0,
        sessionsMax = -1,
        durationMax = -1,
        viewTimeMax = 0,
        itemSession,
        itemDuration,
        itemViewTime,
        items = [];

    for (var i = 0, l = filteredPrograms.length; i < l; i++) {
        if (program && program.content_id !== filteredPrograms[i].item.content_id) {
            continue;
        }

        if (date && date !== filteredPrograms[i].date) {
            continue;
        }

        itemSession = filteredPrograms[i].item.count;
        itemDuration = filteredPrograms[i].item.view_total_time;
        itemViewTime = filteredPrograms[i].item.view_percentiles && filteredPrograms[i].item.view_percentiles.length ?
                                filteredPrograms[i].item.view_percentiles[this.getViewPercentileIndex()] :
                                0;

        if (itemSession > sessionsMax) {
            sessionsMax = itemSession;
        }

        if (itemDuration > durationMax) {
            durationMax = itemDuration;
        }

        if (itemViewTime > viewTimeMax) {
            viewTimeMax = itemViewTime;
        }

        sessions += itemSession;
        duration += itemDuration;
        viewTime += itemViewTime;
        viewTimeCount += 1;

        items.push(filteredPrograms[i]);
    }

    if (viewTimeCount) {
        viewTime = viewTime / viewTimeCount;
    }

    return {
        sessions: sessions,
        duration: duration,
        viewTime: viewTime,
        max: {
            sessions: sessionsMax,
            duration: durationMax,
            viewTime: viewTimeMax
        },
        items: addItems ? items : []
    };
};

Viewer.prototype.getProgramMetricControl = function() {
    if (this.state.dimensions.metric === 'sessions') {
        return '<td data-order="sessions" class="stream__control stream__stats-metric">' +
                    '<div class="stream__control_area"><div class="legend-item__line legend-item__line-sessions"></div></div>' +
                '</td>';
    } else if (this.state.dimensions.metric === 'duration') {
        return '<td data-order="duration" class="stream__control stream__stats-metric">' +
                    '<div class="stream__control_area"><div class="legend-item__line legend-item__line-duration"></div></div>' +
                '</td>';
    } else if (this.state.dimensions.metric === 'sessions_durations') {
        return '<td data-order="sessions" class="stream__control stream__stats-metric">' +
                    '<div class="stream__control_area"><div class="legend-item__line legend-item__line-sessions"></div></div>' +
                '</td>' +
                '<td data-order="duration" class="stream__control stream__stats-metric">' +
                    '<div class="stream__control_area"><div class="legend-item__line legend-item__line-duration"></div></div>' +
                '</td>' +
                '<td data-order="viewtime" class="stream__control stream__stats-metric">' +
                    '<div class="stream__control_area"><div class="legend-item__line legend-item__line-viewtime"></div></div>' +
                '</td>';
    } else if (this.state.dimensions.metric === 'density') {
        return '<td data-order="sessions" class="stream__control stream__stats-metric">' +
                    '<div class="stream__control_area"><div class="legend-item__line legend-item__line-sessions"></div></div>' +
                '</td>';
    }

    return '';
};

Viewer.prototype.getProgramMetricHtml = function(program, filteredPrograms) {
    var stats = this.getProgramMetricStats(program, null, filteredPrograms),
        metricValue = '';

    if (this.state.dimensions.metric === 'sessions') {
        metricValue = stats.sessions;
    } else if (this.state.dimensions.metric === 'duration') {
        metricValue = this.getProgramDuration(stats.duration);
    } else if (this.state.dimensions.metric === 'sessions_durations') {
        return '<td class="stream__stats-metric">' + stats.sessions + '</td>' +
                '<td class="stream__stats-metric">' + this.getProgramDuration(stats.duration) + '</td>' +
                '<td class="stream__stats-metric">' + this.getProgramDuration(stats.viewTime, true) + '</td>';
    } else if (this.state.dimensions.metric === 'density') {
        metricValue = this.getProgramVisualDensityStats(program, filteredPrograms).sum;
    }

    return '<td class="stream__stats-metric">' + metricValue + '</td>';
};

Viewer.prototype.getProgramVisualDateDiff = function(firstDate, date) {
    var firstDateParsed = firstDate.split('-'),
        dateParsed = date.split('-'),
        firstDateParsed2 = new Date(Number(firstDateParsed[0]), Number(firstDateParsed[1]) - 1, firstDateParsed[2]),
        dateParsed2 = new Date(Number(dateParsed[0]), Number(dateParsed[1]) - 1, dateParsed[2]);
    
    firstDateParsed2.setHours(12);
    dateParsed2.setHours(12);

    return (firstDateParsed2 - dateParsed2) / (1000 * 60 * 60 * 24);
};

Viewer.prototype.getProgramVisualStats = function(program, filteredPrograms, totalStats, maxProgramDensity) {
    if (!filteredPrograms.length) {
        return '<td class="stream__stats-visual"></td>';
    }

    var dates = {},
        i, l;

    for (i = 0, l = filteredPrograms.length; i < l; i++) {
        dates[filteredPrograms[i].date] = 1;
    }

    dates = Object.keys(dates).sort();

    var html = '',
        filteredProgramsByDate,
        programStats = this.getProgramMetricStats(program, null, filteredPrograms);

    if (this.state.dimensions.metric === 'sessions' || this.state.dimensions.metric === 'duration') {
        for (i = 0, l = dates.length; i < l; i++) {
            filteredProgramsByDate = this.getProgramMetricStats(program, dates[i], filteredPrograms, true);

            if (filteredProgramsByDate.sessions) {
                html += this.getProgramVisualStatsMetric(program, filteredProgramsByDate, programStats, totalStats, this.getProgramVisualDateDiff(dates[i], dates[0]), dates[i]);
            }
        }
    } else if (this.state.dimensions.metric === 'sessions_durations') {
        html += this.getProgramVisualStatsSessionsDurations(programStats, totalStats);
    } else if (this.state.dimensions.metric === 'density') {
        html += this.getProgramVisualDensity(program, filteredPrograms, maxProgramDensity);
    }

    return '<td class="stream__stats-visual"><div class="stream__stats-visual_wrap">' + html + '</div></td>';
};

Viewer.prototype.getProgramVisualStatsMetric = function(program, filteredProgramsByDate, programStats, totalStats, diff, date) {
    var live = this.getProgramMetricStats(null, null, this.getViewsFilter(filteredProgramsByDate.items, {view_type: 'live'})),
        dvr = this.getProgramMetricStats(null, null, this.getViewsFilter(filteredProgramsByDate.items, {view_type: 'dvr'})),
        metric = this.state.dimensions.metric;

    //console.log(program.title, program);

    var html = '',
        total = totalStats.max[metric];

    if (live[metric]) {
        html += '<div data-index="' + diff + '" class="stream__stats-visual_item stream__stats-visual_item_live" data-date="' + date + '" data-title="' + (metric === 'duration' ? this.getProgramDuration(live[metric]) : live[metric]) + '" style="width:' + (100 * live[metric] / total) + '%"></div>';
    }

    if (dvr[metric]) {
        html += '<div data-index="' + diff + '" class="stream__stats-visual_item stream__stats-visual_item_dvr" data-date="' + date + '" data-title="' + (metric === 'duration' ? this.getProgramDuration(dvr[metric]) : dvr[metric]) + '" style="width:' + (100 * dvr[metric] / total) + '%"></div>';
    }

    return html;
};

Viewer.prototype.getProgramVisualStatsSessionsDurations = function(programStats, totalStats) {
    return '<div class="stream__stats-visual_item2 stream__stats-visual_item2_sessions" style="width:' + (totalStats.max.sessions ? 100 * programStats.sessions / totalStats.max.sessions : 0) + '%"></div>' +
            '<div class="stream__stats-visual_item2 stream__stats-visual_item2_duration" style="width:' + (totalStats.max.duration ? 100 * programStats.duration / totalStats.max.duration : 0) + '%"></div>' +
            '<div class="stream__stats-visual_item2 stream__stats-visual_item2_viewtime" style="width:' + (totalStats.max.viewTime ? 100 * programStats.viewTime / totalStats.max.viewTime : 0) + '%"></div>';
};

Viewer.prototype.getProgramVisualDensityStats = function(program, filteredPrograms) {
    var stats = [],
        totals = [],
        maxValue = 0,
        sum = 0,
        i, l, j, m;

    for (i = 0, l = filteredPrograms.length; i < l; i++) {
        if (program && program.content_id !== filteredPrograms[i].item.content_id) {
            continue;
        }

        if (filteredPrograms[i].item.hist) {
            stats.push(filteredPrograms[i].item.hist);
        }
    }

    if (stats.length) {
        for (i = 0, l = stats.length; i < l; i++) {
            for (j = 0, m = stats[i].length; j < m; j++) {
                if (!totals[j]) {
                    totals[j] = 0;
                }

                totals[j] += stats[i][j];

                sum += stats[i][j];
            }
        }

        maxValue = Math.max.apply(Math, totals);
    }

    return {totals: totals, max: maxValue, sum: sum};
};

Viewer.prototype.showTopViews = function() {
    var totalStats = [],
        programStats,
        filteredPrograms,
        self = this,
        filterParams = this.getFilterParams();

    $('.channellist-item').each(function() {
        var channel = $(this).data('channel'),
            programs = channel !== '_total_' ? self.channels[channel].programms : [];

        if (programs && programs.length) {
            for (var i = 0, l = programs.length; i < l; i++) {
                filteredPrograms = self.getViewsByChannelFilter(channel, filterParams.filterParams);
                programStats = self.getProgramMetricStats(programs[i], null, filteredPrograms);

                totalStats.push({
                    channel: channel,
                    program: programs[i],
                    stats: programStats,
                    filteredPrograms: filteredPrograms
                });
            }
        }
    });

    var candidates = this.filterPrograms(totalStats);

    this.orderPrograms(candidates);

    var html = this.drawViewsProgramControl();

    for (var i = 0, l = Math.min(20, candidates.length); i < l; i++) {
        html += this.drawTopProgram(candidates[i].channel, candidates[i].program, candidates[i].filteredPrograms);
    }

    html = '<table class="stream__schedule-table"><tbody>' + html + '</tbody></table>';

    this.drawViewsProgramFilter();

    this.programsArea.append(html);

    this.addControlEvents();
};

Viewer.prototype.filterPrograms = function(arr) {
    var self = this;

    return arr.filter(function(item) {
        return item.stats.viewTime > self.state.limits.viewTime &&
                item.stats.sessions > self.state.limits.sessions;
    });
};

Viewer.prototype.orderPrograms = function(arr) {
    var self = this;

    arr.sort(function(a, b) {
        if (self.state.orderBy === 'sessions') {
            return b.stats.sessions - a.stats.sessions;
        } else if (self.state.orderBy === 'duration') {
            return b.stats.duration - a.stats.duration;
        } else if (self.state.orderBy === 'viewtime') {
            if (a.stats && typeof a.stats.viewTime !== 'undefined') {
                return b.stats.viewTime - a.stats.viewTime;
            } else {
                return b.stats.duration - a.stats.duration;
            }
        } else if (self.state.orderBy === 'time') {
            return a.program.start_time - b.program.start_time;
        }
    });
};

Viewer.prototype.addControlEvents = function() {
    $('.stream__control.active').removeClass('active');
    $('.stream__control[data-order="' + this.state.orderBy + '"]').addClass('active');

    var self = this;

    $('.stream__control:not(.active)').click(function() {
        var order = $(this).data('order');

        self.state.orderBy = order;

        self.navigate();
    });
};

function hslToRgb(h, s, l) {
    var r, g, b;

    if (s === 0) {
        r = g = b = l; // achromatic
    } else {
        var hue2rgb = function hue2rgb(p, q, t) {
            if (t < 0) {
                t += 1;
            }
            if (t > 1) {
                t -= 1;
            }
            if (t < 1 / 6) {
                return p + (q - p) * 6 * t;
            }
            if (t < 1 / 2) {
                return q;
            }
            if (t < 2 / 3) {
                return p + (q - p) * (2 / 3 - t) * 6;
            }
            return p;
        };

        var q = l < 0.5 ? l * (1 + s) : l + s - l * s;
        var p = 2 * l - q;
        r = hue2rgb(p, q, h + 1 / 3);
        g = hue2rgb(p, q, h);
        b = hue2rgb(p, q, h - 1 / 3);
    }

    return [Math.round(r * 255), Math.round(g * 255), Math.round(b * 255)];
}

function numberToColorHsl(i) {
    // as the function expects a value between 0 and 1, and red = 0° and green = 120°
    // we convert the input to the appropriate hue value
    var hue = i * 1.2 / 360;
    // we convert hsl to rgb (saturation 50%, lightness 60%)
    var rgb = hslToRgb(hue, .5, .6);
    // we format to css value and return
    return 'rgb(' + rgb[0] + ',' + rgb[1] + ',' + rgb[2] + ')';
}

Viewer.prototype.getProgramVisualDensity = function(program, filteredPrograms, maxProgramDensity) {
    var stats = this.getProgramVisualDensityStats(program, filteredPrograms),
        html = '',
        color,
        density;

    if (stats.totals) {
        for (var i = 0, l = stats.totals.length; i < l; i++) {
            density = stats.totals[i] / maxProgramDensity;

            if (density > this.densityLimit) {
                color = numberToColorHsl(100 * (1 - density));
            } else {
                color = 'rgba(0,0,0,0)';
            }
            html += '<div class="stream__stats-visual_item_density" data-statmax="' + maxProgramDensity + '" data-stat="' + stats.totals[i] + '" style="background-color:' + color + '"></div>';
        }
    }

    return html;
};

Viewer.prototype.getProgramsByContentid = function(program, filteredPrograms) {
    var programs = [];

    for (var i = 0, l = filteredPrograms.length; i < l; i++) {
        if (filteredPrograms[i].item.content_id === program.content_id) {
            programs.push(filteredPrograms[i].item);
        }
    }

    return programs;
};

Viewer.prototype.attachListEvents = function() {
    var self = this;

    $('.channellist-item').click(function() {
        var channel = $(this).data('channel');

        if (channel === '_total_') {
            self.state.dimensions.metric = 'sessions_durations';

            if (self.state.orderBy === 'time') {
                self.state.orderBy = 'sessions';
            }
        }

        self.state.channel = channel;
        self.navigate();
    });
};

Viewer.prototype.init = function() {
    this.datepicker();
    this.initRouter();
};

Viewer.prototype.navigate = function() {
    Router.navigate(JSON.stringify(this.getState()));
};

Viewer.prototype.initRouter = function() {
    // configuration
    Router.config({
        root: this.rootViewer
    });

    var self = this;

    // adding routes
    Router.add(/({.*})|(%7B.*%7D)/, function() {
        try {
            var state = JSON.parse(decodeURIComponent(arguments[0] || arguments[1])),
                dateChanged = self.state.date !== self.lastDate;

            self.setState(state);

            if (dateChanged) {
                $('.datepicker').datepicker('setDate', self.state.date);

                self.getData();
            } else {
                self.getProgramViews();
            }
        } catch(e) {
            self.setState(self.getDefaultState());

            self.getData();
        }
    }).add('', function() {
        var state = self.getDefaultState();

        state.orderBy = 'sessions';
        state.dimensions.metric = 'sessions_durations';

        self.setState(state);

        self.getData();
    })
    .listen();

    Router.check(location.hash);
};

Viewer.prototype.datepicker = function() {
    var self = this;

    $('.datepicker').datepicker({
        minDate: new Date(2017, 1 - 1, 10),
        maxDate: 0,
        dateFormat: 'yy-mm-dd',
        firstDay: 1,
        onSelect: function(date) {
            self.lastDate = self.state.date;
            self.state.date = date;
            self.navigate();
        }
    });
};

Viewer.prototype.bindProgramVisualHint = function() {
    var self = this;

    $('body').on('mouseenter', '.stream__stats-visual_wrap, .channels-list .channellist-item', function () {
        var stats = $(this);

        if (stats.find('.stream__stats-hint').length || stats.children().length === 0) {
            return;
        }

        var hint = '',
            hintForChannels = false;

        if (stats.hasClass('channellist-item')) {
            hint = self.drawChannelVisualHint(stats);
            hintForChannels = true;
        } else if (self.state.dimensions.metric === 'sessions' || self.state.dimensions.metric === 'duration') {
            hint = self.drawProgramVisualHint(stats);
        } else if (self.state.dimensions.metric === 'density') {
            hint = self.drawProgramVisualHintDensity(stats);
        }

        if (!hint) {
            return;
        }

        hint = $(hint).appendTo(stats);

        var hintBottom = hint.offset().top + hint.height();
        var tableBottom = self.programsArea.offset().top + self.programsArea.height();
        var channelsBottom = $('.channels-list').offset().top + $('.channels-list').height();

        if (hintForChannels) {
            if (hintBottom > channelsBottom) {
                hint.css('margin-top', channelsBottom - hintBottom);
            }
        } else {
            if (hintBottom > tableBottom) {
                hint.css('margin-top', tableBottom - hintBottom);
            }
        }
    });

    this.updateBodyProgramHintClass();
    $(window).on('resize', this.updateBodyProgramHintClass.bind(this));
};

Viewer.prototype.updateBodyProgramHintClass = function () {
    var stats = $('.stream__stats-visual').eq(0);

    if (!stats.length) {
        return;
    }

    var isLeft = $(window).width() < stats.offset().left + stats.width() * 2 + 10;

    $('body')
        .toggleClass('body__stream-hint_type_left', isLeft)
        .toggleClass('body__stream-hint_type_right', !isLeft);
};

Viewer.prototype.drawProgramVisualHint = function (stats) {
    var items = [];

    var sourceItems = stats.find('.stream__stats-visual_item');
    var widths = [],
        totalWidth = 0;

    sourceItems.each(function (index) {
        widths[index] = parseFloat(this.style.width);
        totalWidth += widths[index];
    });

    sourceItems.each(function (index) {
        var elem = $(this);
        var isLive = elem.hasClass('stream__stats-visual_item_live');
        var relativeWidth = widths[index] / totalWidth;
        var dvrIndex = elem.data('index');

        items.push('<div class="stream__stats-hint-row">' +
            '<div class="stream__stats-hint-bg ' +
                (isLive ? 'stream__stats-hint-bg_type_live' : 'stream__stats-hint-bg_type_dvr') + '"' +
                ' style="width:' + (relativeWidth * 100).toFixed(2) + '%"' +
                ' data-index="' + dvrIndex + '"></div>' +
            '<div>' + elem.data('title') + '</div>' +
            '<div>' + (isLive ? 'live ' : 'dvr ') + elem.data('date') + '</div>' +
            '</div>');
    });

    return '' +
        '<div class="stream__stats-hint">' +
            '<div class="stream__stats-hint-inner">' +
                items.join('') +
            '</div>' +
        '</div>';
};

Viewer.prototype.drawChannelVisualHint = function(stats) {
    var sessions = Number($('.channellist-item__sessions', stats).data('stat')),
        duration = Number($('.channellist-item__duration', stats).data('stat')),
        viewTime = Number($('.channellist-item__viewtime', stats).data('stat'));

    return '' +
        '<div class="stream__stats-hint">' +
            '<div class="stream__stats-hint-inner">' +
                '<table class="stream__stats-channel"><tbody>' +
                    '<tr>' +
                        '<td class="stream__stats-channel-text">Сессии</td>' +
                        '<td class="stream__stats-channel-value">' + sessions + '</td>' +
                    '<tr/>' +
                    '<tr class="stream__stats-channel">' +
                        '<td class="stream__stats-channel-text">Общее время просмотра</td>' +
                        '<td class="stream__stats-channel-value">' + this.getProgramDuration(duration) + '</td>' +
                    '<tr/>' +
                    '<tr class="stream__stats-channel">' +
                        '<td class="stream__stats-channel-text">' + this.state.viewPercentile + '% времени просмотра</td>' +
                        '<td class="stream__stats-channel-value">' + this.getProgramDuration(viewTime, true) + '</td>' +
                    '<tr/>' +
                '</tbody></table>' +
            '</div>' +
        '</div>';
};

Viewer.prototype.drawProgramVisualHintDensity = function(stats) {
    var items = [];

    var sourceItems = stats.find('.stream__stats-visual_item_density');
    var hist = [],
        maxValue = 0,
        limit = 0;

    if (!sourceItems.length) {
        return '';
    }

    sourceItems.each(function (index) {
        hist[index] = parseInt($(this).data('stat'), 10);
        limit = parseInt($(this).data('statmax'), 10);

        if (maxValue < hist[index]) {
            maxValue = hist[index];
        }
    });


    sourceItems.each(function (index) {
        var density = hist[index] / limit;
        items.push('<div class="stream__stats-hint-density__item">' +
                        '<div class="stream__stats-hint-density__item-wrap">' +
                            '<div class="stream__stats-hint-density__item-bar" style="height:' + (100 * density) + '%;background-color:' + numberToColorHsl(100 * (1 - density)) + '"></div>' +
                        '</div>' +
                        '<div class="stream__stats-hint-density__item-text">' + hist[index] + '</div>' +
                        '<div class="stream__stats-hint-density__item-percent">' + ((index + 1) * 10) + '%</div>' +
                    '</div>');
    });

    return '' +
            '<div class="stream__stats-hint stream__stats-hint-density">' +
                '<div class="stream__stats-hint-density__wrap">' +
                    items.join('') +
                    '<div class="stream__stats-hint-density__limit-wrap">' +
                        '<div class="stream__stats-hint-density__limit" style="bottom:' + (100 * this.densityLimit) + '%">' +
                        '</div>' +
                    '</div>' +
                '</div>' +
            '</div>';
};
