# -*- coding: utf-8 -*-

from __future__ import absolute_import, print_function, division
from collections import OrderedDict

from nile.api.v1 import Record, aggregators as na
from qb2.api.v1 import extractors as qe, filters as qf, typing as qt

from statbox_abt_metrics.metrics.common.schema import with_metrics_schema
from statbox_abt_metrics.metrics.metrics_group import (
    YTMetricsGroup,
    metrics_group,
)
from statbox_abt_metrics.meta import Meta, MetricsMetaRegistry
from statbox_abt_metrics.stattests.yt import MannWhitneyULocal


DOC_URL = "https://wiki.yandex-team.ru/stream/stream/speed/"
DOC_LINK = '<a target="_blank" href="{}">Wiki</a>'.format(DOC_URL)
NAMES = {
    "category_first_paint": {
        "en": "Timing of categories screen first paint",
        "ru": "Время первой отрисовки экрана категорий",
    },
    "category_redraw": {
        "en": "Timing of categories screen redraw",
        "ru": "Время перерисовки экрана категорий",
    },
    "channels_first_paint": {
        "en": "Timing of channels list screen first paint",
        "ru": "Время первой отрисовки экрана со списком каналов",
    },
    "channels_redraw": {
        "en": "Timing of channels list screen redraw",
        "ru": "Время перерисовки экрана со списком каналов",
    },
    "error404_first_paint": {
        "en": "Timing of 404 error screen first paint",
        "ru": "Время первой отрисовки экрана ошибки 404",
    },
    "error404_redraw": {
        "en": "Timing of 404 error screen redraw",
        "ru": "Время перерисовки экрана ошибки 404",
    },
    "error_first_paint": {
        "en": "Timing of fatal error first paint",
        "ru": "Скорость первой отрисовки экрана фатальной ошибки",
    },
    "error_redraw": {
        "en": "Timing of fatal error redraw",
        "ru": "Скорость перерисовки экрана фатальной ошибки",
    },
    "landing_first_paint": {
        "en": "Timing of publisher screen first paint",
        "ru": "Время первой отрисовки экрана паблишера",
    },
    "landing_redraw": {
        "en": "Timing of publisher screen redraw",
        "ru": "Время перерисовки экрана паблишера",
    },
    "mini_player_create": {
        "en": "player_create analog for mini player",
        "ru": "Аналог player_create для миниплеера",
    },
    "mini_player_init": {
        "en": "player_init analog for mini player",
        "ru": "Аналог player_init для миниплеера",
    },
    "mini_player_start": {
        "en": "player_start analog for mini player",
        "ru": "Аналог player_start для миниплеера",
    },
    "overlay_first_paint": {
        "en": "Timing of overlay first paint",
        "ru": "Время первой отрисовки паранджи",
    },
    "overlay_redraw": {
        "en": "Timing of overlay redraw",
        "ru": "Время перерисовки паранджи",
    },
    "player_create": {
        "en": "Metric for player initialization start",
        "ru": "Метрика начала инициализации плеера для экрана просмотра",
    },
    "player_init": {
        "en": "Metric for player initialization end",
        "ru": "Метрика окончания инициализации плеера для экрана просмотра",
    },
    "player_start": {
        "en": "Metric for playback start",
        "ru": "Метрика старта проигрывания плеера для экрана просмотра",
    },
    "schedule_first_paint": {
        "en": "Timing of schedule first paint",
        "ru": "Время первой отрисовки расписания",
    },
    "schedule_redraw": {
        "en": "Timing of schedule redraw",
        "ru": "Время перерисовки расписания",
    },
    "set_source": {
        "en": "Timing of initializing a new piece of content",
        "ru": "Время включения новой передачи в плеере",
    },
    "store_front_first_paint": {
        "en": "Timing of store front first paint",
        "ru": "Время первой отрисовки витрины",
    },
    "store_front_redraw": {
        "en": "Timing of store front redraw",
        "ru": "Время перерисовки витрины",
    },
    "watch_first_paint": {
        "en": "Timing of watch screen first paint",
        "ru": "Скорость первой отрисовки экрана просмотра",
    },
    "watch_redraw": {
        "en": "Timing of watch screen redraw",
        "ru": "Скорость перерисовки экрана просмотра",
    },
}


@metrics_group
class EtherMetrics(YTMetricsGroup):
    class RunFilter(YTMetricsGroup.RunFilter):
        allow_yql = True
        dates_decomposition = True
        full = True

    ru_name = "RUM-метрики Эфира"
    en_name = "Ether RUM metrics"
    description = """Метрики скорости отрисовки различных элементов Эфира"""

    marks_metrics = [
        "player_create",
        "player_init",
        "player_start",
        "mini_player_create",
        "mini_player_init",
        "mini_player_start",
    ]
    deltas_metrics = [
        "category_first_paint",
        "category_redraw",
        "channels_first_paint",
        "channels_redraw",
        "error404_first_paint",
        "error404_redraw",
        "error_first_paint",
        "error_redraw",
        "landing_first_paint",
        "landing_redraw",
        "overlay_first_paint",
        "overlay_redraw",
        "schedule_first_paint",
        "schedule_redraw",
        "set_source",
        "store_front_first_paint",
        "store_front_redraw",
        "watch_first_paint",
        "watch_redraw",
    ]
    metrics = {}
    for metric in marks_metrics + deltas_metrics:
        metrics[metric] = {
            "name_hits": "{}_hits".format(metric),
            "name_avg": "{}_avg".format(metric),
            "ru_name_hits": "{} (hits)".format(NAMES[metric]["ru"]),
            "en_name_hits": "{} (hits)".format(NAMES[metric]["en"]),
            "ru_name_avg": "{} (avg)".format(NAMES[metric]["ru"]),
            "en_name_avg": "{} (avg)".format(NAMES[metric]["en"]),
        }

    @staticmethod
    def extract_test_ids(tests):
        if not tests:
            return []
        res = []
        if ";" in tests:
            # Format like "id1,0,bucket1;id2,0,bucket2"
            for chunk in tests.split(";"):
                exp_id_data = chunk.split(",")
                if exp_id_data:
                    res.append(str(exp_id_data[0]).strip())
        else:
            # Format like "id1,id2"
            for exp_id in tests.split(","):
                res.append(str(exp_id).strip())

        return res

    @classmethod
    def create_metric_unfolder(cls, hit_aggregator_data, avg_aggregator_data):
        @with_metrics_schema
        def unfold_metrics(records):
            for base_record in records:
                key = Record(
                    datetime=base_record.datetime,
                    exp_id=base_record.exp_id,
                    user_id=base_record.user_id,
                )

                for aggregator_name in hit_aggregator_data:
                    value = getattr(base_record, aggregator_name, None)
                    if value is None:
                        continue

                    yield Record(
                        key,
                        metric_name=aggregator_name,
                        value=value,
                        test_measure="sum",
                    )

                for aggregator_name in avg_aggregator_data:
                    value = getattr(base_record, aggregator_name, None)
                    if value is None:
                        continue

                    yield Record(
                        key,
                        metric_name=aggregator_name,
                        value=value,
                        test_measure="mean",
                    )

        return unfold_metrics

    @staticmethod
    def get_marks_value(marks, field):
        if not marks:
            return
        val = marks.get(field)
        # Фикс для версии 1.2.2 rum с параллельными таймерами
        if val is not None and val != 0 and val != 1:
            return round(val, 0)
        return

    @classmethod
    def build_extractors(self):
        result = []
        for metric in self.marks_metrics:
            result += [
                qe.custom(
                    metric,
                    lambda x: self.get_marks_value(x, metric),
                    "time_marks",
                ).with_type(float)
            ]
        for metric in self.deltas_metrics:
            result += [
                qe.custom(
                    metric,
                    lambda x: self.get_marks_value(x, metric),
                    "time_deltas",
                ).with_type(float)
            ]
        return result

    def build_meta(self):
        registry = MetricsMetaRegistry()

        for metric_key, metric_meta in self.metrics.iteritems():
            registry.add(
                Meta(
                    name=metric_meta["name_hits"],
                    ru_name=metric_meta["ru_name_hits"],
                    en_name=metric_meta["en_name_hits"],
                    positive_direction="up",
                    description="{} ({})".format(metric_key, DOC_LINK),
                )
            )

            registry.add(
                Meta(
                    name=metric_meta["name_avg"],
                    ru_name=metric_meta["ru_name_avg"],
                    en_name=metric_meta["en_name_avg"],
                    positive_direction="down",
                    description="{} ({})".format(metric_key, DOC_LINK),
                )
            )

        return registry

    def build_stream(self, job):
        aggregators = {}

        hit_aggregator_data = []
        avg_aggregator_data = []

        for metric_key, metric_meta in self.metrics.iteritems():
            aggregator_name = metric_meta["name_hits"]
            aggregators[aggregator_name] = na.count(qf.defined(metric_key))
            hit_aggregator_data.append(aggregator_name)

            aggregator_name = metric_meta["name_avg"]
            aggregators[aggregator_name] = na.mean(metric_key)
            avg_aggregator_data.append(aggregator_name)

        return (
            job.log(log_path="//home/velocity/rum/1d/")
            .qb2(
                log="generic-log",
                fields=[
                    qe.log_field("time_deltas").with_type(qt.Json),
                    qe.log_field("time_marks").with_type(qt.Json),
                    qe.log_field("datetime").with_type(str),
                    qe.log_field("rum_id").with_type(str),
                    qe.log_field("uid").rename("user_id").with_type(str),
                    self.helpers.get_exp_id_extractor(
                        "exp_id",
                        qe.log_field("test_ids").rename("raw_test_ids").hide(),
                        qe.custom(
                            "test_ids_", self.extract_test_ids, "raw_test_ids"
                        ).hide(),
                        qe.unfold("exp_id", "test_ids_").with_type(str),
                    ),
                ]
                + self.build_extractors(),
                filters=[
                    qf.or_(
                        qf.contains("rum_id", ".stream."),
                        qf.contains("rum_id", ".videohub."),
                    ),
                    qf.one_of("exp_id", self.config.experiments),
                ],
            )
            .call(self.helpers.unfold_datetime, "datetime")
            .groupby("datetime", "exp_id", "user_id")
            .aggregate(**aggregators)
            .map(
                self.create_metric_unfolder(
                    hit_aggregator_data, avg_aggregator_data
                )
            )
        )
