#!/usr/bin/env python
# -*- coding: utf-8 -*-

from __future__ import absolute_import, print_function, division
from statbox_abt_metrics.metrics.metrics_group import (
    metrics_group, YTMetricsGroup
)
from nile.api.v1 import (
    aggregators as na
)
from statbox_abt_metrics.stattests.yt.symmetrical_binom_test import (
    SymmetricalBinomTest
)
from statbox_abt_metrics.meta import Meta, MetricsMetaRegistry
from .video_player_metrics_common import (
    video_player_prepare_stream, GenerateMetrics
)
from statbox_abt_metrics.metrics.common.schema import with_metrics_schema


class GenerateDisorderMetrics(GenerateMetrics):

    def __call__(self, recs):
        for rec in recs:

            for value in ['vsid', 'microsession']:
                yield self._generate_metric(
                    rec, 'video_player_{}_count'.format(value),
                    value='{}s'.format(value)
                )


@metrics_group
class VideoPlayerDisorders(YTMetricsGroup):
    ru_name = 'Метрики разладок видеоплеера'
    en_name = 'Video player disorder metrics'
    description = (
        """Количество сессий видеоплеера """
    )

    statistical_test = SymmetricalBinomTest

    class RunFilter(YTMetricsGroup.RunFilter):
        full = True
        dates_decomposition = True
        allow_yql = True

    def init(self):
        pass

    def build_meta(self):
        registry = MetricsMetaRegistry()
        vsids = Meta(
            name="video_player_vsid_count",
            ru_name="Количество запусков плеера",
            en_name="Video player instances",
            positive_direction="up",
            description="""
            Количество запусков плеера (video session id)
            """
        )
        microsessions = Meta(
            name="video_player_microsession_count",
            ru_name="Количество микросессий",
            en_name="Video player microsessions",
            positive_direction="up",
            description="""
            Количество микросессий плеера (непрерывное смотрение одной единицы контента)
            """
        )
        registry.add(vsids)
        registry.add(microsessions)
        return registry

    def build_stream(self, job):
        mapped_stream = video_player_prepare_stream(self, job)
        reduced_stream = mapped_stream.groupby(
            'datetime', 'exp_id'
        ).aggregate(
            microsessions=na.count(),
            vsids=na.count_distinct('user_id')
        ).map(
            with_metrics_schema(GenerateDisorderMetrics())
        )
        return reduced_stream
