#!/usr/bin/env python
# -*- coding: utf-8 -*-

from __future__ import absolute_import, print_function, division
from statbox_abt_metrics.metrics.metrics_group import (
    metrics_group, YTMetricsGroup
)
from nile.api.v1 import (
    aggregators as na,
    Record,
    with_hints,
    extended_schema
)
from statbox_abt_metrics.metrics.common.schema import with_metrics_schema
from statbox_abt_metrics.meta import Meta, MetricsMetaRegistry
from .video_player_metrics_common import (
    video_player_prepare_stream, GenerateMetrics
)


def get_fatals(errors, view_time):
    fatals = [x for x in errors if x['id'].endswith('_fatal')]
    if not fatals:
        return 0
    elif fatals[0]['rel_time'] < view_time:
        return 0
    else:
        return 1


def get_stalled_duration(x):
    try:
        return int(x['details']['stalledDuration'])
    except (KeyError, ValueError, TypeError):
        return 1


@with_hints(
    output_schema=extended_schema(
        fatals=int,
        stalleds=int,
        stalled_others=int,
        stalleds_duration=int,
        stalled_others_duration=int
    )
)
def add_error_metrics(recs):
    for rec in recs:
        errors = rec.get('errors') or []
        view_time = rec.get('view_time') or 0
        fatals = get_fatals(errors, view_time)
        _stalleds = [x for x in errors if x['id'].startswith('Stalled')]
        _stalled_others = [
            x for x in _stalleds if x['id'] == 'Stalled_Other' and
            x['details'].get('connection', 'OK') in {'O', 'OK'}
        ]
        stalleds = len(_stalleds)
        stalled_others = len(_stalled_others)
        stalleds_duration = sum(
            get_stalled_duration(x) for x in _stalleds
        )
        stalled_others_duration = sum(
            get_stalled_duration(x) for x in _stalled_others
        )
        yield Record(
            rec,
            fatals=fatals,
            stalleds=stalleds,
            stalled_others=stalled_others,
            stalleds_duration=stalleds_duration,
            stalled_others_duration=stalled_others_duration,
        )


class GenerateErrorMetrics(GenerateMetrics):

    def __init__(self, label):
        self.label = label

    def __call__(self, recs):
        label = self.label
        for rec in recs:

            if label == 'by_vsid':
                yield self._generate_metric(
                    rec, 'total_view_time'
                )
                yield self._generate_metric(
                    rec, 'fatals'
                )
                yield self._generate_metric(
                    rec, 'stalleds'
                )
                yield self._generate_metric(
                    rec, 'stalled_others'
                )
                yield self._generate_metric(
                    rec, 'total_stalleds_duration'
                )
                yield self._generate_metric(
                    rec, 'total_stalled_others_duration'
                )

            for value in ['fatals', 'stalleds', 'stalled_others']:
                yield self._generate_metric(
                    rec, '{}_{}'.format(value, label),
                    test_measure='mean',
                    value=value
                )
            for value in ['stalled', 'stalled_other']:
                yield self._generate_metric(
                    rec, '{}_duration_share_{}'.format(value, label),
                    test_measure='mean',
                    value='total_{}s_duration'.format(value),
                    count='total_view_time',
                )


@metrics_group
class VideoPlayerErrors(YTMetricsGroup):
    ru_name = 'Метрики ошибок видеоплеера'
    en_name = 'Video player error metrics'
    description = (
        "Количество фатальных ошибок видеоплеера (суммарно и на сессию), "
        "то же со столдами, длительность столдов на сессию етц "
    )

    class RunFilter(YTMetricsGroup.RunFilter):
        full = True
        dates_decomposition = True
        allow_yql = True

    def build_meta(self):
        registry = MetricsMetaRegistry()
        registry.update([
            Meta(
                name="total_view_time",
                ru_name="Суммарное время просмотра",
                en_name="Total view time",
                positive_direction="up",
                description="""
                Суммарное время просмотра
                """
            ),
            Meta(
                name="fatals",
                ru_name="Фатальные ошибки",
                en_name="Fatal errors",
                positive_direction="down",
                description="""
                Количество фатальных ошибок (= сессий с фатальными ошибками)
                """
            ),
            Meta(
                name="stalleds",
                ru_name="Столды",
                en_name="Stalleds",
                positive_direction="down",
                description="""
                Количество столдов (крутящихся индикаторов загрузки)
                """
            ),
            Meta(
                name="stalled_others",
                ru_name="Столды типа Other",
                en_name="Stalled_others",
                positive_direction="down",
                description="""
                Количество столдов типа Other на коннекшне OK
                (с неясной причиной)
                """
            ),
            Meta(
                name="total_stalleds_duration",
                ru_name="Суммарная длительность столдов",
                en_name="Total Stalled duration",
                positive_direction="down",
                description="""
                Суммарная длительность столдов
                """
            ),
            Meta(
                name="total_stalled_others_duration",
                ru_name="Суммарная длительность столдов типа Other",
                en_name="Total Stalled_Other duration",
                positive_direction="down",
                description="""
                Суммарная длительность столдов типа Other на коннекшне OK
                """
            )
        ])
        for label in ['vsid', 'session']:
            registry.update([
                Meta(
                    name="fatals_by_{}".format(label),
                    ru_name="Фатальных ошибок на {}".format(label),
                    en_name="Fatal errors per {}".format(label),
                    positive_direction="down",
                    description="""
                Среднее количество фатальных ошибок на {}
                """.format(label)
                ),
                Meta(
                    name="stalleds_by_{}".format(label),
                    ru_name="Столдов на {}".format(label),
                    en_name="Stalleds per {}".format(label),
                    positive_direction="down",
                    description="""
                Среднее количество столдов на {}
                """.format(label)
                ),
                Meta(
                    name="stalled_others_by_{}".format(label),
                    ru_name="Столдов типа Other на {}".format(label),
                    en_name="Stalled_Other per {}".format(label),
                    positive_direction="down",
                    description="""
                Среднее количество столдов типа Other на {}
                """.format(label)
                ),
                Meta(
                    name="stalled_duration_share_by_{}".format(label),
                    ru_name="Доля длительности столдов на {}".format(label),
                    en_name="Stalled duration share per {}".format(label),
                    positive_direction="down",
                    description="""
                Доля длительности столдов от длительности просмотра
                """.format(label)
                ),
                Meta(
                    name="stalled_other_duration_share_by_{}".format(label),
                    ru_name="Доля длительности столдов типа Other на {}".format(
                        label
                    ),
                    en_name="Stalled_Other duration share per {}".format(
                        label),
                    positive_direction="down",
                    description="""
                Доля длительности столдов типа Other от длительности просмотра
                """.format(label)
                )
            ])
        return registry

    def build_stream(self, job):
        mapped_stream = video_player_prepare_stream(self, job).map(
            add_error_metrics
        )

        to_concat = []

        for grouping in [
            ('by_vsid', ['user_id', 'exp_id', 'datetime']),
            ('by_session', [])
        ]:
            label, groupby_args = grouping

            if groupby_args:
                reduced_stream = mapped_stream.groupby(
                    *groupby_args
                ).aggregate(
                    fatals=na.sum('fatals'),
                    stalleds=na.sum('stalleds'),
                    stalled_others=na.sum('stalled_others'),
                    total_view_time=na.sum('view_time'),
                    total_stalleds_duration=na.sum(
                        'stalleds_duration'
                    ),
                    total_stalled_others_duration=na.sum(
                        'stalled_others_duration'
                    ),
                )
            else:
                reduced_stream = mapped_stream.project(
                    'user_id', 'exp_id', 'datetime',
                    'fatals', 'stalleds', 'stalled_others',
                    total_view_time='view_time',
                    total_stalleds_duration='stalleds_duration',
                    total_stalled_others_duration='stalled_others_duration',
                )

            to_concat.append(reduced_stream.map(
                with_metrics_schema(GenerateErrorMetrics(label))
            ))

        return job.concat(*to_concat)
