#!/usr/bin/env python
# -*- coding: utf-8 -*-

from __future__ import absolute_import, print_function, division
from statbox_abt_metrics.metrics.metrics_group import (
    metrics_group, YTMetricsGroup
)
from nile.api.v1 import (
    extractors as ne,
    aggregators as na,
)
from statbox_abt_metrics.meta import Meta, MetricsMetaRegistry
from statbox_abt_metrics.metrics.common.schema import with_metrics_schema
from .video_player_metrics_common import (
    video_player_prepare_stream, GenerateMetrics
)


class GenerateMoneyMetrics(GenerateMetrics):

    def __call__(self, recs):
        for rec in recs:

            for value in ['price', 'partner_price']:
                for test_measure in ['sum', 'mean']:
                    yield self._generate_metric(
                        rec, 'video_player_ad_{}_{}'.format(
                            test_measure, value
                        ),
                        test_measure=test_measure,
                        value=value
                    )
                    yield self._generate_metric(
                        rec, 'video_player_withmoney_{}_{}'.format(
                            test_measure, value
                        ),
                        test_measure=test_measure,
                        value='{}_nonzero'.format(value)
                    )


def money_reducer_pipeline(job, grouped_stream_):
    to_concat = []
    for fieldname in ['price', 'partner_price']:
        grouped_stream = grouped_stream_.aggregate(
            sum_=na.sum(fieldname),
        ).project(
            ne.all(),
            nonzero=ne.custom(lambda x: 1 if x else 0, 'sum_').with_type(
                int
            ),
        )

        sum_metric = grouped_stream.project(
            ne.all(exclude=['sum_', 'nonzero']),
            value='sum_',
            test_measure=ne.const('sum'),
            metric_name=ne.const('video_player_ad_sum_{}'.format(fieldname))
        )
        to_concat.append(sum_metric)

        mean_metric = grouped_stream.project(
            ne.all(exclude=['sum_', 'nonzero']),
            value='sum_',
            test_measure=ne.const('mean'),
            metric_name=ne.const('video_player_ad_mean_{}'.format(fieldname))
        )
        to_concat.append(mean_metric)

        players_with_money_sum_metric = grouped_stream.project(
            ne.all(exclude=['sum_', 'nonzero']),
            value='nonzero',
            test_measure=ne.const('sum'),
            metric_name=ne.const(
                'video_player_withmoney_sum_{}'.format(fieldname)
            )
        )
        to_concat.append(players_with_money_sum_metric)

        players_with_money_mean_metric = grouped_stream.project(
            ne.all(exclude=['sum_', 'nonzero']),
            value='nonzero',
            test_measure=ne.const('mean'),
            metric_name=ne.const(
                'video_player_withmoney_mean_{}'.format(fieldname)
            )
        )
        to_concat.append(players_with_money_mean_metric)
    return job.concat(*to_concat)


@metrics_group
class VideoPlayerMoney(YTMetricsGroup):
    ru_name = 'Метрики денег видеоплеера'
    en_name = 'Video player ad revenue metrics'
    description = (
        """Метрики видеоплеера — сумма и среднее price и partner_price, """
        """доля плееров с деньгами"""
    )

    class RunFilter(YTMetricsGroup.RunFilter):
        full = True
        dates_decomposition = True
        allow_yql = True

    def init(self):
        pass

    def build_meta(self):
        registry = MetricsMetaRegistry()
        for fieldname in ['price', 'partner_price']:
            sum_ = Meta(
                name="video_player_ad_sum_{}".format(fieldname),
                ru_name="Сумма денег по полю {}".format(fieldname),
                en_name="Total of ad revenue by field {}".format(fieldname),
                positive_direction="up",
                description="""
                Сумма денег в эксперименте по полю {} из dsp-log
                """.format(fieldname)
            )
            mean = Meta(
                name="video_player_ad_mean_{}".format(fieldname),
                ru_name="Среднее денег по полю {}".format(fieldname),
                en_name="Mean of ad revenue by field {}".format(fieldname),
                positive_direction="up",
                description="""
                Среднее денег в эксперименте по полю {} из dsp-log
                """.format(fieldname)
            )
            nz_sum_ = Meta(
                name="video_player_withmoney_sum_{}".format(fieldname),
                ru_name="Количество плееров с деньгами по полю {}".format(fieldname),
                en_name="Number of player instances with money by field {}".format(fieldname),
                positive_direction="up",
                description="""
                Количество плееров с деньгами в эксперименте по полю {} из dsp-log
                """.format(fieldname)
            )
            nz_mean = Meta(
                name="video_player_withmoney_mean_{}".format(fieldname),
                ru_name="Доля плееров с деньгами по полю {}".format(fieldname),
                en_name="Share of player instances with money by field {}".format(fieldname),
                positive_direction="up",
                description="""
                Доля плееров с деньгами в эксперименте по полю {} из dsp-log
                """.format(fieldname)
            )
            registry.add(sum_)
            registry.add(mean)
            registry.add(nz_sum_)
            registry.add(nz_mean)
        return registry

    def build_stream(self, job):
        mapped_stream = video_player_prepare_stream(self, job).groupby(
            'user_id', 'exp_id', 'datetime'
        ).aggregate(
            price=na.sum('price'),
            partner_price=na.sum('partner_price')
        ).project(
            ne.all(),
            price=ne.custom(lambda x: x / 1000000.0, 'price').with_type(float),
            partner_price=ne.custom(
                lambda x: x / 1000000.0, 'partner_price'
            ).with_type(float),
            price_nonzero=ne.custom(
                lambda x: 1 if x else 0, 'price'
            ).with_type(float),
            partner_price_nonzero=ne.custom(
                lambda x: 1 if x else 0, 'partner_price'
            ).with_type(float),
        )
        reduced_stream = mapped_stream.map(
            with_metrics_schema(GenerateMoneyMetrics())
        )
        return reduced_stream
