import ast


class VideoPageQualityAggregatedAverage:
    def __init__(self, depth=3):
        self.depth = depth

    def value_by_position(self, position_metric_params):
        res = position_metric_params.result

        pq_weights = {
                "HIGHEST": 1,
                "HIGH": 0.75,
                "MEDIUM": 0.5,
                "LOW": 0.25,
                "LOWEST": 0,
                "FOREIGN_LANG": 0,
                "404": 0
            }
        pq_marks = ast.literal_eval(res.get_scale("pq_grade_enrichment", "[]"))

        if not pq_marks:
            return 0

        return (1.0 * sum([pq_weights[mark] for mark in pq_marks]))/len(pq_marks)

    def aggregate_by_position(self, agg_metric_params):
        metric_values = agg_metric_params.pos_metric_values
        marks = [x.value for x in metric_values[:self.depth] if x.value is not None]

        if len(marks) == 0:
            return 0

        return 1.0 * sum(marks) / self.depth


class VideoPageQualityWithVideoRelevanceAggregatedAverage:
    def __init__(self, depth=3):
        self.depth = depth

    def value_by_position(self, position_metric_params):
        res = position_metric_params.result

        pq_weights = {
                "HIGHEST": 1,
                "HIGH": 0.75,
                "MEDIUM": 0.5,
                "LOW": 0.25,
                "LOWEST": 0,
                "FOREIGN_LANG": 0,
                "404": 0
            }
        relevance_weights = {'RELEVANT_PLUS': 1.0,
                'RELEVANT_PLUS_NO_PLAYER': 0.4,
                'RELEVANT_MINUS': 0.5,
                'RELEVANT_MINUS_NO_PLAYER': 0.25,
                'IRRELEVANT': 0.0,
                'IRRELEVANT_NO_PLAYER': 0.0,
                'SOFT_404': 0.0,
                'SOFT_404_NO_PLAYER': 0.0,
                '_404': 0.0,
                '_404_NO_PLAYER': 0.0,
                'VIRUS' : 0.0,
                'VIRUS_NO_PLAYER' : 0.0
                }
        pq_marks = ast.literal_eval(res.get_scale("pq_grade_enrichment", "[]"))
        relevance_mark = res.get_scale("video_relevance_with_video_player", '_404')


        if not pq_marks:
            return 0

        return relevance_weights[relevance_mark] * (1.0 * sum([pq_weights[mark] for mark in pq_marks]))/len(pq_marks)

    def aggregate_by_position(self, agg_metric_params):
        metric_values = agg_metric_params.pos_metric_values
        marks = [x.value for x in metric_values[:self.depth] if x.value is not None]

        if len(marks) == 0:
            return 0

        return 1.0 * sum(marks) / self.depth


class VideoPageQualityConsistency:
    def __init__(self, depth=3, inconsistency_threshold=2):
        self.depth = depth
        self.inconsistency_threshold = inconsistency_threshold

    def value_by_position(self, position_metric_params):
        res = position_metric_params.result

        pq_gradations = {
            "404": 0, "FOREIGN_LANG": 0, "LOWEST": 0,
            "LOW": 1,
            "MEDIUM": 2,
            "HIGH": 3,
            "HIGHEST": 4
        }

        pq_marks = ast.literal_eval(res.get_scale("pq_grade_enrichment", "[]"))

        if not pq_marks:
            return 0

        max_gradation = max(pq_gradations[mark] for mark in pq_marks)
        min_gradation = min(pq_gradations[mark] for mark in pq_marks)
        return 1.0 if max_gradation - min_gradation < self.inconsistency_threshold else 0.0

    def aggregate_by_position(self, agg_metric_params):
        metric_values = agg_metric_params.pos_metric_values
        marks = [x.value for x in metric_values[:self.depth] if x.value is not None]

        if len(marks) == 0:
            return 0

        return float(sum(marks)) / self.depth


class VideoPageQualityInconsistency:
    def __init__(self, depth=3, inconsistency_threshold=3):
        self.depth = depth
        self.inconsistency_threshold = inconsistency_threshold

    def value_by_position(self, position_metric_params):
        res = position_metric_params.result

        pq_gradations = {
            "404": 0, "FOREIGN_LANG": 0, "LOWEST": 0,
            "LOW": 1,
            "MEDIUM": 2,
            "HIGH": 3,
            "HIGHEST": 4
        }
        inappropriate_marks = {"404", "FOREIGN_LANG"}

        pq_marks = ast.literal_eval(res.get_scale("pq_grade_enrichment", "[]"))

        pq_marks = [mark for mark in pq_marks if mark not in inappropriate_marks]
        if not pq_marks:
            return 0.0

        max_gradation = max(pq_gradations[mark] for mark in pq_marks)
        min_gradation = min(pq_gradations[mark] for mark in pq_marks)
        return 1.0 if max_gradation - min_gradation >= self.inconsistency_threshold else 0.0

    def aggregate_by_position(self, agg_metric_params):
        metric_values = agg_metric_params.pos_metric_values
        marks = [x.value for x in metric_values[:self.depth] if x.value is not None]

        if len(marks) == 0:
            return 0

        return float(sum(marks)) / self.depth


class VideoPageQuality404:
    def __init__(self, depth=3):
        self.depth = depth

    def value_by_position(self, position_metric_params):
        res = position_metric_params.result
        pq_marks = ast.literal_eval(res.get_scale("pq_grade_enrichment", "[]"))
        return 1.0 if "404" in pq_marks else 0.0

    def aggregate_by_position(self, agg_metric_params):
        metric_values = agg_metric_params.pos_metric_values
        marks = [x.value for x in metric_values[:self.depth] if x.value is not None]

        if len(marks) == 0:
            return 0

        return float(sum(marks)) / self.depth


class VideoPageQualityAssessmentClassifierDiff:
    def __init__(self, depth=5):
        self.depth = depth

    def value_by_position(self, position_metric_params):
        res = position_metric_params.result

        pq_weights = {
            "HIGHEST": 1,
            "HIGH": 0.75,
            "MEDIUM": 0.5,
            "LOW": 0.25,
            "LOWEST": 0, "FOREIGN_LANG": 0, "404": 0
        }

        pq_marks = ast.literal_eval(res.get_scale("pq_grade_enrichment", "[]"))
        classifier_score = res.get_scale("video_pq_classifier_score")

        if not pq_marks or classifier_score is None:
            return 0.0

        assessment_score = float(sum([pq_weights[mark] for mark in pq_marks])) / len(pq_marks)

        return assessment_score - classifier_score

    def aggregate_by_position(self, agg_metric_params):
        metric_values = agg_metric_params.pos_metric_values
        marks = [x.value for x in metric_values[:self.depth] if x.value is not None]

        if len(marks) == 0:
            return 0.0

        return float(sum(marks)) / self.depth


class VideoDefectRateAssessmentAnyDefect:
    def __init__(self, depth=5):
        self.depth = depth

    def value(self, metric_params):
        pq_weights = {
            "HIGHEST": 1,
            "HIGH": 0.75,
            "MEDIUM": 0.5,
            "LOW": 0.25,
            "LOWEST": 0,
            "FOREIGN_LANG": 0,
            "404": 0
        }

        DEFECT_THRESHOLD = pq_weights["LOW"]

        for res in metric_params.results[:self.depth]:
            pq_marks = ast.literal_eval(res.get_scale("pq_grade_enrichment", "[]"))
            if not pq_marks:
                continue
            if min([pq_weights[mark] for mark in pq_marks]) <= DEFECT_THRESHOLD:
                return 1.0

        return 0.0


class VideoDefectRateAssessmentAllDefect:
    def __init__(self, depth=5):
        self.depth = depth

    def value(self, metric_params):
        pq_weights = {
            "HIGHEST": 1,
            "HIGH": 0.75,
            "MEDIUM": 0.5,
            "LOW": 0.25,
            "LOWEST": 0,
            "FOREIGN_LANG": 0,
            "404": 0
        }

        DEFECT_THRESHOLD = pq_weights["LOW"]

        correct_cnt = 0
        for res in metric_params.results[:self.depth]:
            pq_marks = ast.literal_eval(res.get_scale("pq_grade_enrichment", "[]"))
            if not pq_marks:
                continue
            correct_cnt += 1
            if min([pq_weights[mark] for mark in pq_marks]) > DEFECT_THRESHOLD:
                return 0.0

        return 1.0 if correct_cnt > 0 else 0.0


class VideoDefectRateClassifierAnyDefect:
    def __init__(self, depth=5, defect_threshold=0.25):
        self.depth = depth
        self.defect_threshold = defect_threshold

    def value(self, metric_params):
        for res in metric_params.results[:self.depth]:
            pq_classifier_score = res.get_scale("video_pq_classifier_score")
            if pq_classifier_score is None:
                continue
            if pq_classifier_score <= self.defect_threshold:
                return 1.0

        return 0.0


class VideoDefectRateClassifierAllDefect:
    def __init__(self, depth=5, defect_threshold=0.25):
        self.depth = depth
        self.defect_threshold = defect_threshold

    def value(self, metric_params):
        correct_cnt = 0
        for res in metric_params.results[:self.depth]:
            pq_classifier_score = res.get_scale("video_pq_classifier_score")
            if pq_classifier_score is None:
                continue
            correct_cnt += 1
            if pq_classifier_score > self.defect_threshold:
                return 0.0

        return 1.0 if correct_cnt > 0 else 0.0


class VideoPageQualityTop1BadDoc:
    def __init__(self, depth=5):
        self.depth = depth

    def value(self, metric_params):
        pq_gradations = {
            "404": 0, "FOREIGN_LANG": 0, "LOWEST": 0,
            "LOW": 1,
            "MEDIUM": 2,
            "HIGH": 3,
            "HIGHEST": 4
        }

        if not metric_params.results:
            return 0.0

        top1_pq_marks = ast.literal_eval(metric_params.results[0].get_scale("pq_grade_enrichment", "[]"))
        if not top1_pq_marks:
            return 0.0
        top1_pq_gradation = min(pq_gradations[mark] for mark in top1_pq_marks)

        if top1_pq_gradation <= pq_gradations["LOW"]:
            for res in metric_params.results[1:self.depth]:
                relevance = res.get_scale("video_relevance_with_video_player", '_404')
                pq_marks = ast.literal_eval(res.get_scale("pq_grade_enrichment", "[]"))
                if not pq_marks:
                    continue
                pq_gradation = min(pq_gradations[mark] for mark in pq_marks)

                if relevance == "RELEVANT_PLUS" and pq_gradation > pq_gradations["LOW"]:
                    return 1.0

        return 0.0

class VideoPageQualityClassifierAggregatedAverage:
    def __init__(self, depth=5):
        self.depth = depth

    def value_by_position(self, position_metric_params):
        res = position_metric_params.result
        return res.get_scale("video_pq_classifier_score", 0)


    def aggregate_by_position(self, agg_metric_params):
        metric_values = agg_metric_params.pos_metric_values
        marks = [x.value for x in metric_values[:self.depth] if x.value is not None]

        if len(marks) == 0:
            return 0

        return 1.0 * sum(marks) / self.depth


class VideoRelevanceAggregatedAverage:
    def __init__(self, depth=5):
        self.depth = depth

    def value_by_position(self, position_metric_params):
        res = position_metric_params.result

        relevance_weights = {'RELEVANT_PLUS': 1.0,
                'RELEVANT_PLUS_NO_PLAYER': 0.4,
                'RELEVANT_MINUS': 0.5,
                'RELEVANT_MINUS_NO_PLAYER': 0.25,
                'IRRELEVANT': 0.0,
                'IRRELEVANT_NO_PLAYER': 0.0,
                'SOFT_404': 0.0,
                'SOFT_404_NO_PLAYER': 0.0,
                '_404': 0.0,
                '_404_NO_PLAYER': 0.0,
                'VIRUS' : 0.0,
                'VIRUS_NO_PLAYER' : 0.0
                }
        return relevance_weights[res.get_scale("video_relevance_with_video_player", '_404')]

    def aggregate_by_position(self, agg_metric_params):
        metric_values = agg_metric_params.pos_metric_values
        marks = [x.value for x in metric_values[:self.depth] if x.value is not None]

        if len(marks) == 0:
            return 0

        return 1.0 * sum(marks) / self.depth


class VideoPageQualityClassifierWithVideoRelevanceAggregatedAverage:
    def __init__(self, depth=3):
        self.depth = depth

    def value_by_position(self, position_metric_params):
        res = position_metric_params.result

        relevance_weights = {'RELEVANT_PLUS': 1.0,
                'RELEVANT_PLUS_NO_PLAYER': 0.4,
                'RELEVANT_MINUS': 0.5,
                'RELEVANT_MINUS_NO_PLAYER': 0.25,
                'IRRELEVANT': 0.0,
                'IRRELEVANT_NO_PLAYER': 0.0,
                'SOFT_404': 0.0,
                'SOFT_404_NO_PLAYER': 0.0,
                '_404': 0.0,
                '_404_NO_PLAYER': 0.0,
                'VIRUS' : 0.0,
                'VIRUS_NO_PLAYER' : 0.0
                }
        pq_marks = res.get_scale("video_pq_classifier_score", 0)
        relevance_mark = res.get_scale("video_relevance_with_video_player", '_404')

        return relevance_weights[relevance_mark] * pq_marks

    def aggregate_by_position(self, agg_metric_params):
        metric_values = agg_metric_params.pos_metric_values
        marks = [x.value for x in metric_values[:self.depth] if x.value is not None]

        if len(marks) == 0:
            return 0

        return 1.0 * sum(marks) / self.depth

