from datetime import datetime


class PQVideoProxima:
    def __init__(self, rel_coeff, pq_boost_coeffs, relevance_field="video_relevance_with_video_player",
                 depth=5):
        self.rel_coeff = rel_coeff
        self.pq_boost_coeffs = pq_boost_coeffs
        self.relevance_field = relevance_field
        self.depth = depth

        self.serp_download_ts = None
        self.actuality_is_needed = None

    def precompute(self, metric_params):
        self.serp_download_ts = metric_params.serp_data \
                                             .get("headers", {}) \
                                             .get("timestamp", datetime.now().timestamp()) / 1000
        self.actuality_is_needed = metric_params.serp_data.get(
            "serp_query_param.video_query_actuality",
            "no_need_actuality"
        ) == "need_actuality"

    def value_by_position(self, position_metric_params):
        def rel_num_value(label, middle_mark_weight):
            v = {
                "RELEVANT_PLUS": 1.0,
                "RELEVANT_PLUS_NO_PLAYER": 0.5,
                "RELEVANT_MINUS": middle_mark_weight,
                "RELEVANT_MINUS_NO_PLAYER": middle_mark_weight / 2,
                "IRRELEVANT": 0.0,
                "IRRELEVANT_NO_PLAYER": 0.0,
                "SOFT_404": 0.0,
                "SOFT_404_NO_PLAYER": 0.0,
                "_404": 0.0,
                "_404_NO_PLAYER": 0.0,
                "VIRUS": 0.0,
                "VIRUS_NO_PLAYER": 0.0
            }
            return v[label]

        def act_coeff(rel, boost_val):
            v = {
                "RELEVANT_PLUS": boost_val,
                "RELEVANT_MINUS": boost_val,
                "IRRELEVANT": 0.0
            }
            return v.get(rel, 0.0) if self.actuality_is_needed else 0.0

        def pq_boost(relevance, pq):
            return self.pq_boost_coeffs.get(relevance, self.pq_boost_coeffs["IRRELEVANT"])

        BOOST_VALUE = 0.3
        REL_MINUS_WEIGHT = 0.7

        res = position_metric_params.result

        relevance = res.get_scale(self.relevance_field, "_404")
        rel_numeric = rel_num_value(relevance, REL_MINUS_WEIGHT)

        published_date = datetime.strptime(res.get_scale("text.publishedDate"), "%Y-%m-%dT%H:%M:%S%z") \
                                 .timestamp() / 1000
        delta_minutes = ((self.serp_download_ts - published_date) / 60) if published_date < self.serp_download_ts else 0.0
        delta_days = delta_minutes / (60 * 24) if delta_minutes > 30 * 60 * 24 else 0.0
        doc_age = max(0.5, 1 - delta_days * (0.5 / (3 * 365 - 30)))

        pq_classifier_score = res.get_scale("video_pq_classifier_score", 0.0)

        return self.rel_coeff * rel_numeric \
               + act_coeff(relevance, BOOST_VALUE) * doc_age \
               + pq_boost(relevance, pq_classifier_score) * pq_classifier_score

    def aggregate_by_position(self, agg_metric_params):
        metric_values = [m.value for m in agg_metric_params.pos_metric_values[:self.depth]]
        return sum(metric_values) / float(self.depth)
