import math
import json

class ProximaKernel:
    def __init__(self, depth=5,
                 video_quality_desktop_weight=1,
                 kernel_desktop_weight=1,
                 popularity_desktop_weight=1,
                 video_quality_touch_weight=1,
                 kernel_touch_weight=1,
                 popularity_touch_weight=1):
        self.depth = depth

        self.video_quality_desktop_weight = video_quality_desktop_weight
        self.kernel_desktop_weight = kernel_desktop_weight
        self.popularity_desktop_weight = popularity_desktop_weight

        self.video_quality_touch_weight = video_quality_touch_weight
        self.kernel_touch_weight = kernel_touch_weight
        self.popularity_touch_weight = popularity_touch_weight

    def value(self, metric_params):

        BOOST_VALUE = 0.3

        def coef(rel):

            v = {'RELEVANT_PLUS': BOOST_VALUE,
                 'RELEVANT_MINUS': BOOST_VALUE,
                 'IRRELEVANT': 0.0
                 }

            return v.get(rel, 0.0)

        def get_kernel(albin_host_bundle):

            visitors = json.loads(albin_host_bundle).get("visitors", 0.)

            if not visitors:
                visitors = 0
            kessel_odd_factor160 = json.loads(albin_host_bundle).get("kessel_odd_factor160", 0.)

            return (kessel_odd_factor160 * visitors + 0.486 * 0.0013) / (visitors + 0.0013) - 0.486

        def get_factors_combination(res, metric_params):

            IS_DESKTOP = 0
            is_desktop = metric_params.query_device == IS_DESKTOP

            if is_desktop:
                video_quality_weight = self.video_quality_desktop_weight
                kernel_weight = self.kernel_desktop_weight
                popularity_weight = self.popularity_desktop_weight
            else:
                video_quality_weight = self.video_quality_touch_weight
                kernel_weight = self.kernel_touch_weight
                popularity_weight = self.popularity_touch_weight

            kernel = get_kernel(res.scales.get('video_albin_host_bundle',"{}"))

            my_clickadd = kernel * kernel_weight

            my_clickadd = float(my_clickadd) / (video_quality_weight + kernel_weight * 0.5 + popularity_weight)

            return my_clickadd

        results = metric_params.results[:self.depth]

        max_dcg = (1.0 + BOOST_VALUE) * sum([1.0 / math.log(2 + pos) for pos in range(self.depth)])
        dcg = 0

        for res in results:
            pos_index = res.pos
            assert (pos_index >= 1)

            relevance = res.scales.get("video_relevance", '_404')
            rel_value = coef(relevance) * get_factors_combination(res, metric_params)

            dcg += rel_value / math.log(1 + pos_index)

        return dcg / max_dcg
