# -*- coding: utf-8 -*-

from nile.api.v1 import (
    filters as nf,
    aggregators as na,
    extractors as ne,
    statface as ns,
    clusters,
    Record
)

from qb2.api.v1 import extractors as se, filters as sf
import argparse
import nile
import datetime
import uatraits, json, re
import urllib, urlparse
import math,cgi
import pandas as pd
from itertools import product
import sys
import os


cluster = clusters.yt.Hahn(pool='vika-pavlova'
    ).env(templates=dict(job_root='home/videolog/vika-pavlova/3705-station_report'
                        ),
          yt_spec_defaults=dict(pool_trees=["physical"],
                                use_default_tentative_pool_trees = True),
          parallel_operations_limit=10
         )


def redir_parse(date):

    job = cluster.job()

    redir = job.table('logs/redir-log/1d/' + date)

    raw = redir.qb2(log = 'redir-log',
                    fields = ['path','yandexuid','date',
                              se.log_fields('from', "deviceID", "ticks", 'url', "duration", "x-req-id",'reqid'),
                              se.custom('init', lambda x: 1 if 'inited' in str(x) else 0, 'path'),
                              se.custom('start', lambda x: 1 if 'started' in x else 0,'path')
                             ],
                    filters = [
                               sf.equals('from', "yastation")
                              ],
                    mode = 'yamr_lines'
                   ).sort('yandexuid'
                         ).put('$job_root/raw_' + date)

    job.run()


def add_host(recs):

    for rec in recs:
        try:
            host = urlparse.urlparse(urlparse.unquote(rec['url'].replace('data=url%3D', ''))).netloc
        except:
            host = 'none'

        yield Record(rec, host = host)


def add_total(recs):

    for rec in recs:

        yield Record(fielddate = rec.date, host = rec.host, tvt = rec.tvt,
                     init_count = rec.init_count, start_count = rec.start_count,
                     device_count = rec.device_count
                    )

        yield Record(fielddate = rec.date, host = '_total_', tvt = rec.tvt,
                     init_count = rec.init_count, start_count = rec.start_count,
                     device_count = rec.device_count
                    )


def process_data(date):

    job = cluster.job()

    raw = job.table('$job_root/raw_' + date
                   ).project(ne.all(exclude = 'ticks'),
                             ticks = ne.custom(lambda x: int(x) if x else 0, 'ticks')
                            ).map(add_host)

    aggr = raw.groupby("date", "deviceID", "reqid", "url", 'host',
                      ).aggregate(ticks = na.max('ticks'),
                                  has_init = na.max('init'),
                                  has_start = na.max('start')
                                 ).filter(sf.defined('host'))

    aggr.groupby("date", 'host'
                ).aggregate(tvt = na.sum('ticks'),
                             init_count = na.sum('has_init'),
                             start_count = na.sum('has_start'),
                             device_count = na.count_distinct("deviceID")
                            ).map(add_total
                                 ).groupby("fielddate", 'host'
                                          ).aggregate(tvt = na.sum('tvt'),
                                                     init_count = na.sum('init_count'),
                                                     start_count = na.sum('start_count'),
                                                     device_count = na.sum("device_count")
                                                    ).put('$job_root/final_' + date)

    job.run()


def put_data_to_stat(date):

    client = ns.StatfaceClient(
        proxy = 'upload.stat.yandex-team.ru',
        token = os.environ['STAT_TOKEN']
    )

    ns.StatfaceReport().path('Video.All/station_tvt_report'
                            ).scale('daily'
                                   ).client(client
                                           ).remote_publish(proxy='hahn',
                                                            table_path='//home/videolog/vika-pavlova/3705-station_report/final_' + date,
                                                            async_mode=False,
                                                            upload_config=False
                                                           )


def main():

    parser = argparse.ArgumentParser()
    parser.add_argument('--start_date', type=str, required=True)
    parser.add_argument('--end_date', type=str, required=True)
    args = parser.parse_args()

    for date in pd.date_range(start=args.start_date, end=args.end_date):
        date_str = str(date)[:10]
        redir_parse(date_str)
        process_data(date_str)
        put_data_to_stat(date_str)


if __name__ == '__main__':
    main()
