// MalwareList class manages malware list subscription and request blocking.

var HOUR_IN_MS = 1000 * 60 * 60;

var malwareUrl = "https://easylist-downloads.adblockplus.org/malwaredomains_full.txt";

// Constructor: initialize the match cache
// Inputs: none.
// the init() function should be called after the constructor.
function MalwareList()
{
  this._matchCache = {};
}

// Update _malwaresub in case there are changes.
// Should be invoked right after creating a MalwareList object.
MalwareList.prototype.init = function()
{
  var that = this;
  ext.storage.get("malware_list", function(response)
  {
    if (response && response.malware_list)
    {
      that._malwaresub = response.malware_list;
    }
    if (!that._malwaresub)
    {
      // Brand new user. subscribe to the filter list
      that._malwaresub = {
        subscribed : true,
        url : malwareUrl
      };
    }
    // Build the list
    that._onSubscriptionChange(true);
    if (that._malwaresub.subscribed)
    {
      // start checking for updates periodically
      window.setInterval(function()
      {
        that.checkFilterUpdates();
      }, 60 * 60 * 1000);
      ext.webRequest.onBeforeRequest.addListener(that.onBeforeRequestHandler);
    }
  });
}

// Get the Subscription data
// used by the Options pages.
MalwareList.prototype.getSubscription = function()
{
  return this._malwaresub;
}
// When a subscription property changes, this function stores it
// Inputs: rebuild? boolean, true if the filterset should be rebuilt
MalwareList.prototype._onSubscriptionChange = function(rebuild)
{
  ext.storage.set('malware_list', this._malwaresub);

  // The only reasons to (re)build the filter set are
  // - when AdBlock starts
  // - when a filter list text is changed ([un]subscribed or updated a list)
  if (rebuild)
  {
    this.rebuild();
  }
}

// Rebuild filters based on the current settings and subscriptions.
MalwareList.prototype.rebuild = function()
{
  // if the user is subscribed to malware, then get it
  if (this._malwaresub && this._malwaresub.subscribed && this._malwaresub.text)
  {
    this.setMalwareDomains(this._malwaresub.text);
  }
  else if (this._malwaresub && this._malwaresub.subscribed && !this._malwaresub.text)
  {
    this._initializeMalwareDomains();
  }
}

// Change a property of the malware subscription or check if it has to be
// updated
// Inputs: subData: object containing all data that should be changed
// forceFetch: if the subscriptions have to be fetched again forced
MalwareList.prototype.changeSubscription = function(subData, forceFetch)
{
  var subscribeRequiredListToo = false;
  var listDidntExistBefore = false;

  // Check if the list has to be updated
  function out_of_date(subscription)
  {
    if (forceFetch)
      return true;
    // After a failure, wait at least a day to refetch (overridden below if
    // it's a new filter list, having no .text)
    var failed_at = subscription.last_update_failed_at || 0;
    if (Date.now() - failed_at < HOUR_IN_MS * 24)
      return false;
    // Don't let expiresAfterHours delay indefinitely (Issue 7443)
    var hardStop = subscription.expiresAfterHoursHard || 240;
    var smallerExpiry = Math.min(subscription.expiresAfterHours, hardStop);
    var millis = Date.now() - subscription.last_update;
    return (millis > HOUR_IN_MS * smallerExpiry);
  }
  // Apply all changes from subData
  for ( var property in subData)
  {
    if (subData[property] !== undefined)
    {
      this._malwaresub[property] = subData[property];
    }
  }

  if (this._malwaresub.subscribed)
  {
    // if forceFetch, set the last update timestamp of the malware to zero, so
    // it's updated now.
    if (forceFetch)
    {
      this._malwaresub.last_update = 0;
    }
    // load the malware domains
    this._loadMalwareDomains();
    ext.webRequest.onBeforeRequest.addListener(this.onBeforeRequestHandler);
    this._onSubscriptionChange(subData.subscribed === false);
    var subscription = this._malwaresub;
    subscription.id = "malware";
    subscription.filters = [{
      disable : true
    }];
    FilterNotifier.triggerListeners("subscription.added", subscription);
  }
  else
  {
    // If unsubscribed, remove properties
    delete this._malwaresub.text;
    delete this._malwaresub.rules;
    delete this._malwaresub.downloadCount;
    delete this._malwaresub.lastSuccess;
    delete this._malwaresub.last_update;
    delete this._malwaresub.expiresAfterHours;
    delete this._malwaresub.last_update_failed_at;
    delete this._malwaresub.last_modified;
    ext.webRequest.onBeforeRequest.removeListener(this.onBeforeRequestHandler);
    this._onSubscriptionChange(subData.subscribed === false);
    var subscription = this._malwaresub;
    subscription.id = "malware";
    subscription.filters = [{
      disable : true
    }];
    FilterNotifier.triggerListeners("subscription.removed", subscription);
  }
  return;
}

MalwareList.prototype.setMalwareDomains = function(malwareDoms)
{
  if (malwareDoms === null)
  {
    this.malwareDomains = null;
    return;
  }
  var domains = malwareDoms.adware;
  var result = {};
  for (var i = 0; i < domains.length; i++)
  {
    var domain = domains[i];
    var char = domain.charAt(0);
    if (!result[char])
    {
      result[char] = [];
    }
    result[char].push(domain);
  }
  this.malwareDomains = result;
}
// Checks if subscriptions have to be updated
// Inputs: force? (boolean), true if every filter has to be updated
MalwareList.prototype.checkFilterUpdates = function(force)
{
  var key = 'last_malwaresub_check';
  var now = Date.now();
  var that = this;
  ext.storage.get(key, function(value)
  {
    var delta = now - (value.last_malwaresub_check || now);
    var delta_hours = delta / HOUR_IN_MS;
    ext.storage.set(key, now);
    if (delta_hours > 24)
    {
      if (that._malwaresub.subscribed)
      {
        that._malwaresub.expiresAfterHours += delta_hours;
      }
      that._onSubscriptionChange(); // Store the change
    }
    if (that._malwaresub.subscribed)
    {
      that.changeSubscription({}, force);
    }
  });
}

// Retreive the list of malware domains from our site.
// and set the response (list of domains) on the blocking
// filter set for processing.
MalwareList.prototype._loadMalwareDomains = function()
{
  function out_of_date(subscription)
  {
    // After a failure, wait at least a day to refetch (overridden below if
    // it has no .text)
    var failed_at = subscription.last_update_failed_at || 0;
    if (Date.now() - failed_at < HOUR_IN_MS * 24)
      return false;
    var hardStop = subscription.expiresAfterHoursHard || 240;
    var smallerExpiry = Math.min((subscription.expiresAfterHours || 24), hardStop);
    var millis = Date.now() - (subscription.last_update || 0);
    return (millis > HOUR_IN_MS * smallerExpiry);
  }

  if (!this._malwaresub.text || !this.getMalwareDomains() || out_of_date(this._malwaresub))
  {
    var url = malwareUrl + "?_=" + new Date().getTime();
    // Fetch file with malware-known domains
    var xhr = new XMLHttpRequest();
    var that = this;
    xhr.onerror = function(e)
    {
      // if the request fail, retry the next time
      that._malwaresub.last_update_failed_at = Date.now();
    }
    xhr.onload = function(e)
    {
      if (!xhr.responseText)
      {
        that._malwaresub.last_update_failed_at = Date.now();
        return;
      }
      var malwareObj = that._convertMalwareDomains(xhr.responseText);
      that.setMalwareDomains(malwareObj);

      // set the malware object on the 'text' property so it's persisted to
      // storage
      that._malwaresub.text = malwareObj;
      that._malwaresub.last_update = Date.now();
      that._malwaresub.last_modified = Date.now();
      that._malwaresub.lastSuccess = that._malwaresub.lastDownload = Math.round(Date.now() / require("downloader").MILLIS_IN_SECOND);
      that._malwaresub.downloadStatus = "synchronize_ok";
      that._malwaresub.downloadCount = (that._malwaresub.downloadCount || 0) + 1;
      delete that._malwaresub.last_update_failed_at;
      // since the Malware file is updated ~1 day
      // expiration is around 24 hours.
      that._malwaresub.expiresAfterHours = 24;
      var smear = Math.random() * 0.4 + 0.8;
      that._malwaresub.expiresAfterHours *= smear;
      that._onSubscriptionChange(true);
      var subscription = that._malwaresub;
      subscription.id = "malware";
      subscription.filters = [{
        disable : true
      }];
      FilterNotifier.triggerListeners("subscription.downloadStatus", subscription);
      log("Fetched " + url);
    }
    xhr.open("GET", url);
    xhr.send();
  }
}
MalwareList.prototype._convertMalwareDomains = function(text)
{
  if (!text)
    return;

  // Convert the filter list to an array of domains
  var domains = [];
  var lines = text.split('\n');
  for (var i = 0; i < lines.length; i++)
  {
    var line = lines[i];
    // The entries we're interested in, should be of the form: ||domain.org^
    // Ignore all others
    if (line.charAt(0) !== '!' && line.charAt(0) !== '[' && line.length >= 4 && line.charAt(0) === '|' && line.charAt(1) === '|')
    {
      line = line.substr(2); // remove preceeding ||
      line = line.substring(0, line.length - 1); // # remove ending ^
      // convert any non-ascii characters to ascii (punycode)
      line = getUnicodeDomain(line.toLowerCase());
      domains.push(line);
    }
  }

  return ({
    "adware" : domains
  });
}
// Retreive the list of malware domains from our site.
// and set the response (list of domains) on the blocking
// filter set for processing.
MalwareList.prototype._initializeMalwareDomains = function()
{
  this._loadMalwareDomains();
}

// Get the current list of malware domains
// will return undefined, if the user is not subscribed to the Malware 'filter
// list'.
MalwareList.prototype.getMalwareDomains = function()
{
  return this._malwaresub.text;
}

// Does the hostname in the parameter URL match
// any of domains in the Malware list?
// Inputs: url (URL)
// Outputs: boolean - true if the request is NOT from a known malware domain
// false if the request is from a known malware domain
MalwareList.prototype.allowRequest = function(url)
{
  if (!url)
    return true;

  if (url in this._matchCache)
  {
    return this._matchCache[url];
  }
  var urlDomain = getUnicodeDomain(parseUri(url).hostname);

  if (!urlDomain)
    return true;

  if (this.malwareDomains && urlDomain && this.malwareDomains[urlDomain.charAt(0)] && this.malwareDomains[urlDomain.charAt(0)].indexOf(urlDomain) > -1)
  {
    log("matched malware domain", urlDomain);
    this._matchCache[url] = false;
    if (typeof createMalwareNotification === "function")
    {
      createMalwareNotification();
    }
    malwareBlockCounts.recordOneMalwareBlocked();
    return this._matchCache[url];
  }

  this._matchCache[url] = true;
  return this._matchCache[url];
}

// When a request starts, perhaps block it.
MalwareList.prototype.onBeforeRequestHandler = function(url, type, page, frame)
{
  if (adblockIsPaused())
  {
    return true;
  }

  if (checkWhitelisted(page, frame))
  {
    return true;
  }
  var blockedData = malwareList.allowRequest(url);
  return blockedData;
}
/*
 * subscription properties: subscribed (bool): if you are subscribed to the list
 * or not last_update (date): time of the last succesfull update last_modified
 * (string): time of the last change on the server last_update_failed_at (date):
 * if set, when the last update attempt failed text (object): array of the
 * domains expiresAfterHours (int): the time after which the subscription
 * expires expiresAfterHoursHard (int): we must redownload subscription after
 * this delay deleteMe (bool): if the subscription has to be deleted
 */

