/*
 * This file is part of Adblock Plus <https://adblockplus.org/>,
 * Copyright (C) 2006-present eyeo GmbH
 *
 * Adblock Plus is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * Adblock Plus is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Adblock Plus.  If not, see <http://www.gnu.org/licenses/>.
 */

/** @module devtools */

import {AllowingFilter,
        ElemHideException} from "../adblockpluscore/lib/filterClasses.js";
import {SpecialSubscription} from
  "../adblockpluscore/lib/subscriptionClasses.js";
import {contentTypes} from "../adblockpluscore/lib/contentTypes.js";
import {parseURL} from "../adblockpluscore/lib/url.js";
import {filterStorage} from "../adblockpluscore/lib/filterStorage.js";
import {defaultMatcher} from "../adblockpluscore/lib/matcher.js";
import {filterNotifier} from "../adblockpluscore/lib/filterNotifier.js";
import {port} from "./messaging.js";
import {HitLogger, nonRequestTypes} from "./hitLogger.js";
import * as info from "info";
import {compareVersions} from "../adblockpluscore/lib/versions.js";

let panels = new Map();

function getFilterInfo(filter)
{
  if (!filter)
    return null;

  let userDefined = false;
  let subscriptionTitle = null;

  for (let subscription of filterStorage.subscriptions(filter.text))
  {
    if (!subscription.disabled)
    {
      if (subscription instanceof SpecialSubscription)
        userDefined = true;
      else
        subscriptionTitle = subscription.title;
    }
  }

  return {
    text: filter.text,
    whitelisted: filter instanceof AllowingFilter ||
                 filter instanceof ElemHideException,
    userDefined,
    subscription: subscriptionTitle
  };
}

function hasRecord(request, filter, record)
{
  return record.request.url == request.url &&
    record.request.docDomain == request.docDomain &&

    // Ignore partial (e.g. ELEMHIDE) allowlisting if there is already
    // a DOCUMENT exception which disables all means of blocking.
    (record.request.type == "DOCUMENT" ?
       nonRequestTypes.includes(request.type) :
       record.request.type == request.type) &&

    // Matched element hiding filters don't relate to a particular request,
    // so we have to compare the selector in order to avoid duplicates.
    (record.filter && record.filter.selector) == (filter && filter.selector) &&

    // We apply multiple CSP filters to a document, but we must still remove
    // any duplicates. Two CSP filters are duplicates if both have identical
    // text.
    (record.filter && record.filter.csp && record.filter.text) ==
    (filter && filter.csp && filter.text);
}

function addRecord(panel, request, filter)
{
  let matchesAny = false;

  for (let i = 0; i < panel.records.length; i++)
  {
    let record = panel.records[i];

    let matches = hasRecord(request, filter, record);
    if (!matches)
      continue;

    matchesAny = true;

    // Update record without filters, if filter matches on later checks
    if (!filter)
      break;

    if (record.filter)
      continue;

    record.filter = filter;

    panel.port.postMessage({
      type: "update-record",
      initialize: true,
      index: i,
      request: record.request,
      filter: getFilterInfo(record.filter)
    });
  }

  if (matchesAny)
    return;

  panel.port.postMessage({
    type: "add-record",
    request,
    filter: getFilterInfo(filter)
  });

  panel.records.push({request, filter});
}

function matchRequest(request)
{
  return defaultMatcher.match(
    parseURL(request.url),
    contentTypes[request.type],
    request.docDomain,
    request.sitekey,
    request.specificOnly
  );
}

function onBeforeRequest(details)
{
  let panel = panels.get(details.tabId);

  // Clear the devtools panel and reload the inspected tab without caching
  // when a new request is issued. However, make sure that we don't end up
  // in an infinite recursion if we already triggered a reload.
  if (panel.reloading)
  {
    panel.reloading = false;
  }
  else
  {
    panel.records = [];
    panel.port.postMessage({type: "reset"});

    // We can't repeat the request if it isn't a GET request. Chrome would
    // prompt the user to confirm reloading the page, and POST requests are
    // known to cause issues on many websites if repeated.
    if (details.method == "GET")
      panel.reload = true;
  }
}

function onLoading(page)
{
  let tabId = page.id;
  let panel = panels.get(tabId);

  // Reloading the tab is the only way that allows bypassing all caches, in
  // order to see all requests in the devtools panel. Reloading must not be
  // performed before the tab changes to "loading", otherwise it will load the
  // previous URL.
  if (panel && panel.reload)
  {
    browser.tabs.reload(tabId, {bypassCache: true});

    panel.reload = false;
    panel.reloading = true;
  }
}

function updateFilters(subscription, filters, added)
{
  let includes = subscription ?
    filter => filter && subscription.findFilterIndex(filter) != -1 :
    filters.includes.bind(filters);

  for (let panel of panels.values())
  {
    for (let i = 0; i < panel.records.length; i++)
    {
      let record = panel.records[i];

      // If an added filter matches a request shown in the devtools panel,
      // update that record to show the new filter. Ignore filters that aren't
      // associated with any sub-resource request. There is no record for these
      // if they don't already match. In particular, in case of element hiding
      // filters, we also wouldn't know if any new element matches.
      if (added)
      {
        if (nonRequestTypes.includes(record.request.type))
          continue;

        let filter = matchRequest(record.request);

        if (!includes(filter))
          continue;

        record.filter = filter;
      }

      // If a filter shown in the devtools panel got removed, update that
      // record to show the filter that matches now, or none, instead.
      // For filters that aren't associated with any sub-resource request,
      // just remove the record. We wouldn't know whether another filter
      // matches instead until the page is reloaded.
      else
      {
        if (!includes(record.filter))
          continue;

        if (nonRequestTypes.includes(record.request.type))
        {
          panel.port.postMessage({
            type: "remove-record",
            index: i
          });
          panel.records.splice(i--, 1);
          continue;
        }

        record.filter = matchRequest(record.request);
      }

      panel.port.postMessage({
        type: "update-record",
        index: i,
        request: record.request,
        filter: getFilterInfo(record.filter)
      });
    }
  }
}

function onFilterAdded(filter)
{
  updateFilters(null, [filter], true);
}

function onFilterRemoved(filter)
{
  updateFilters(null, [filter], false);
}

function onSubscriptionAdded(subscription)
{
  if (subscription instanceof SpecialSubscription)
    updateFilters(subscription, null, true);
}

browser.runtime.onConnect.addListener(newPort =>
{
  let match = newPort.name.match(/^devtools-(\d+)$/);
  if (!match)
    return;

  let inspectedTabId = parseInt(match[1], 10);
  let localOnBeforeRequest = onBeforeRequest.bind();
  let panel = {port: newPort, records: []};
  let hitListener = addRecord.bind(null, panel);

  browser.webRequest.onBeforeRequest.addListener(
    localOnBeforeRequest,
    {
      urls: ["http://*/*", "https://*/*"],
      types: ["main_frame"],
      tabId: inspectedTabId
    }
  );

  if (panels.size == 0)
  {
    ext.pages.onLoading.addListener(onLoading);
    filterNotifier.on("filter.added", onFilterAdded);
    filterNotifier.on("filter.removed", onFilterRemoved);
    filterNotifier.on("subscription.added", onSubscriptionAdded);
  }

  newPort.onDisconnect.addListener(() =>
  {
    HitLogger.removeListener(inspectedTabId, hitListener);
    panels.delete(inspectedTabId);
    browser.webRequest.onBeforeRequest.removeListener(localOnBeforeRequest);

    if (panels.size == 0)
    {
      ext.pages.onLoading.removeListener(onLoading);
      filterNotifier.off("filter.added", onFilterAdded);
      filterNotifier.off("filter.removed", onFilterRemoved);
      filterNotifier.off("subscription.added", onSubscriptionAdded);
    }
  });

  HitLogger.addListener(inspectedTabId, hitListener);
  panels.set(inspectedTabId, panel);
});

/**
 * Returns true if our devtools panel is supported by the browser.
 *
 * @event "devtools.supported"
 * @returns {boolean}
 */
port.on("devtools.supported", (message, sender) =>
  info.platform == "chromium" ||
  info.application == "firefox" &&
  compareVersions(info.applicationVersion, "54") >= 0
);
