#include "cacher_factors.h"
#include "environment.h"

#include <util/generic/xrange.h>

namespace NAntiRobot {

namespace {

THashMap<TString, size_t> MakeCacherFactorNameToIndex() {
    THashMap<TString, size_t> nameToIndex;
    nameToIndex.reserve(TRawCacherFactors::NumFactors());

    for (size_t i = 0; i < static_cast<size_t>(ECacherFactors::NumStaticFactors); ++i) {
        auto name = ToString(static_cast<ECacherFactors>(i));
        nameToIndex.emplace(std::move(name), i);
    }

    for (const auto& [i, rule] : Enumerate(ANTIROBOT_DAEMON_CONFIG.GlobalJsonConfig.LastVisitsRules)) {
        nameToIndex[rule.Name] =
            static_cast<size_t>(ECacherFactors::NumStaticFactors) + i;
    }

    return nameToIndex;
}

} // anonymous namespace

size_t TRawCacherFactors::GetIndexByName(const TString& name) {
    static const auto nameToIndex = MakeCacherFactorNameToIndex();

    const auto index = nameToIndex.FindPtr(name);
    Y_ENSURE(index, "unknown cacher feature name: " << name);

    return *index;
}

void TRawCacherFactors::FillCacherRequestFeatures(const TCacherRequestFeatures& features) {
    Factors[static_cast<size_t>(ECacherFactors::NumDocs)]                = features.NumDocs;
    Factors[static_cast<size_t>(ECacherFactors::PageNum)]                = features.PageNum;
    Factors[static_cast<size_t>(ECacherFactors::HeadersCount)]           = features.HeadersCount;
    Factors[static_cast<size_t>(ECacherFactors::IsProxy)]                = static_cast<float>(features.IsProxy);
    Factors[static_cast<size_t>(ECacherFactors::IsTor)]                  = static_cast<float>(features.IsTor);
    Factors[static_cast<size_t>(ECacherFactors::IsVpn)]                  = static_cast<float>(features.IsVpn);
    Factors[static_cast<size_t>(ECacherFactors::IsHosting)]              = static_cast<float>(features.IsHosting);
    Factors[static_cast<size_t>(ECacherFactors::RefererFromYandex)]      = static_cast<float>(features.RefererFromYandex);
    Factors[static_cast<size_t>(ECacherFactors::IsBadProtocol)]          = static_cast<float>(features.IsBadProtocol);
    Factors[static_cast<size_t>(ECacherFactors::IsBadUserAgent)]         = static_cast<float>(features.IsBadUserAgent);
    Factors[static_cast<size_t>(ECacherFactors::IsBadUserAgentNew)]      = static_cast<float>(features.IsBadUserAgentNew);
    Factors[static_cast<size_t>(ECacherFactors::IsConnectionKeepAlive)]  = static_cast<float>(features.IsConnectionKeepAlive);
    Factors[static_cast<size_t>(ECacherFactors::HaveUnknownHeaders)]     = static_cast<float>(features.HaveUnknownHeaders);
    Factors[static_cast<size_t>(ECacherFactors::IpSubnetMatch)]          = features.IpSubnetMatch;
    Factors[static_cast<size_t>(ECacherFactors::SpravkaLifetime)]        = features.SpravkaLifetime;
    Factors[static_cast<size_t>(ECacherFactors::FraudJa3)]               = features.FraudJa3;
    Factors[static_cast<size_t>(ECacherFactors::FraudSubnet)]            = features.FraudSubnet;
    Factors[static_cast<size_t>(ECacherFactors::SecFetchDest)]           = features.SecFetchDest;
    Factors[static_cast<size_t>(ECacherFactors::AcceptLanguage)]         = features.AcceptLanguage;
    Factors[static_cast<size_t>(ECacherFactors::Cookie)]                 = features.Cookie;
    Factors[static_cast<size_t>(ECacherFactors::UpgradeInsecureRequests)]= features.UpgradeInsecureRequests;
    Factors[static_cast<size_t>(ECacherFactors::AcceptEncoding)]         = features.AcceptEncoding;
    Factors[static_cast<size_t>(ECacherFactors::Dnt)]                    = features.Dnt;
    Factors[static_cast<size_t>(ECacherFactors::Origin)]                 = features.Origin;
    Factors[static_cast<size_t>(ECacherFactors::UserAgent)]              = features.UserAgent;
    Factors[static_cast<size_t>(ECacherFactors::Host)]                   = features.Host;
    Factors[static_cast<size_t>(ECacherFactors::Referer)]                = features.Referer;
    Factors[static_cast<size_t>(ECacherFactors::Authority)]              = features.Authority;
    Factors[static_cast<size_t>(ECacherFactors::CacheControl)]           = features.CacheControl;
    Factors[static_cast<size_t>(ECacherFactors::XForwardedProto)]        = features.XForwardedProto;
    Factors[static_cast<size_t>(ECacherFactors::KeepAlive)]              = features.KeepAlive;
    Factors[static_cast<size_t>(ECacherFactors::Pragma)]                 = features.Pragma;
    Factors[static_cast<size_t>(ECacherFactors::ProxyConnection)]        = features.ProxyConnection;
    Factors[static_cast<size_t>(ECacherFactors::Rtt)]                    = features.Rtt;
    Factors[static_cast<size_t>(ECacherFactors::ChromeHodor)]            = static_cast<float>(features.ChromeHodor);
    Factors[static_cast<size_t>(ECacherFactors::CgiParamText)]           = features.CgiParamPresent[FindArrayIndex(CacherCgiParamName, "text")];
    Factors[static_cast<size_t>(ECacherFactors::CgiParamLr)]             = features.CgiParamPresent[FindArrayIndex(CacherCgiParamName, "lr")];
    Factors[static_cast<size_t>(ECacherFactors::CgiParamClid)]           = features.CgiParamPresent[FindArrayIndex(CacherCgiParamName, "clid")];
    Factors[static_cast<size_t>(ECacherFactors::CgiParamTld)]            = features.CgiParamPresent[FindArrayIndex(CacherCgiParamName, "tld")];
    Factors[static_cast<size_t>(ECacherFactors::CgiParamUrl)]            = features.CgiParamPresent[FindArrayIndex(CacherCgiParamName, "url")];
    Factors[static_cast<size_t>(ECacherFactors::CgiParamSite)]           = features.CgiParamPresent[FindArrayIndex(CacherCgiParamName, "site")];
    Factors[static_cast<size_t>(ECacherFactors::CgiParamLang)]           = features.CgiParamPresent[FindArrayIndex(CacherCgiParamName, "lang")];
    Factors[static_cast<size_t>(ECacherFactors::CookieL)]                = features.CookiePresent[FindArrayIndex(CacherCookieName, "L")];
    Factors[static_cast<size_t>(ECacherFactors::CookieMy)]               = features.CookiePresent[FindArrayIndex(CacherCookieName, "my")];
    Factors[static_cast<size_t>(ECacherFactors::CookieSessionId)]        = features.CookiePresent[FindArrayIndex(CacherCookieName, "Session_id")];
    Factors[static_cast<size_t>(ECacherFactors::CookieYabsFrequency)]    = features.CookiePresent[FindArrayIndex(CacherCookieName, "yabs-frequency")];
    Factors[static_cast<size_t>(ECacherFactors::CookieYandexLogin)]      = features.CookiePresent[FindArrayIndex(CacherCookieName, "yandex_login")];
    Factors[static_cast<size_t>(ECacherFactors::CookieYandexuid)]        = features.CookiePresent[FindArrayIndex(CacherCookieName, "yandexuid")];
    Factors[static_cast<size_t>(ECacherFactors::CookieYs)]               = features.CookiePresent[FindArrayIndex(CacherCookieName, "ys")];
    Factors[static_cast<size_t>(ECacherFactors::CookieFuid01)]           = features.CookiePresent[FindArrayIndex(CacherCookieName, "fuid01")];
    Factors[static_cast<size_t>(ECacherFactors::HeaderAcceptEncoding)]   = features.HttpHeaderPresent[FindArrayIndex(CacherHttpHeaderName, "Accept-Encoding")];
    Factors[static_cast<size_t>(ECacherFactors::HeaderAcceptLanguage)]   = features.HttpHeaderPresent[FindArrayIndex(CacherHttpHeaderName, "Accept-Language")];
    Factors[static_cast<size_t>(ECacherFactors::HeaderAuthorization)]    = features.HttpHeaderPresent[FindArrayIndex(CacherHttpHeaderName, "Authorization")];
    Factors[static_cast<size_t>(ECacherFactors::HeaderCacheControl)]     = features.HttpHeaderPresent[FindArrayIndex(CacherHttpHeaderName, "Cache-Control")];
    Factors[static_cast<size_t>(ECacherFactors::HeaderCookie)]           = features.HttpHeaderPresent[FindArrayIndex(CacherHttpHeaderName, "Cookie")];
    Factors[static_cast<size_t>(ECacherFactors::HeaderConnection)]       = features.HttpHeaderPresent[FindArrayIndex(CacherHttpHeaderName, "Connection")];
    Factors[static_cast<size_t>(ECacherFactors::HeaderContentLength)]    = features.HttpHeaderPresent[FindArrayIndex(CacherHttpHeaderName, "Content-Length")];
    Factors[static_cast<size_t>(ECacherFactors::HeaderContentType)]      = features.HttpHeaderPresent[FindArrayIndex(CacherHttpHeaderName, "Content-Type")];
    Factors[static_cast<size_t>(ECacherFactors::HeaderReferer)]          = features.HttpHeaderPresent[FindArrayIndex(CacherHttpHeaderName, "Referer")];
    Factors[static_cast<size_t>(ECacherFactors::HeaderUserAgent)]        = features.HttpHeaderPresent[FindArrayIndex(CacherHttpHeaderName, "User-Agent")];
    Factors[static_cast<size_t>(ECacherFactors::HeaderXForwardedFor)]    = features.HttpHeaderPresent[FindArrayIndex(CacherHttpHeaderName, "X-Forwarded-For")];
    Factors[static_cast<size_t>(ECacherFactors::InRobotSet)]             = static_cast<float>(features.InRobotSet);
    Factors[static_cast<size_t>(ECacherFactors::HasValidSpravka)]        = static_cast<float>(features.HasValidSpravka);
    Factors[static_cast<size_t>(ECacherFactors::FraudSubnetNew)]         = features.FraudSubnetNew;
    Factors[static_cast<size_t>(ECacherFactors::IsRobot)]                = static_cast<float>(features.IsRobot);
    Factors[static_cast<size_t>(ECacherFactors::IsMobile)]               = static_cast<float>(features.IsMobile);
    Factors[static_cast<size_t>(ECacherFactors::IsBrowser)]              = static_cast<float>(features.IsBrowser);
    Factors[static_cast<size_t>(ECacherFactors::HistorySupport)]         = static_cast<float>(features.HistorySupport);
    Factors[static_cast<size_t>(ECacherFactors::IsEmulator)]             = static_cast<float>(features.IsEmulator);
    Factors[static_cast<size_t>(ECacherFactors::IsBrowserEngine)]        = static_cast<float>(features.IsBrowserEngine);
    Factors[static_cast<size_t>(ECacherFactors::IsBrowserEngineVersion)] = static_cast<float>(features.IsBrowserEngineVersion);
    Factors[static_cast<size_t>(ECacherFactors::IsBrowserVersion)]       = static_cast<float>(features.IsBrowserVersion);
    Factors[static_cast<size_t>(ECacherFactors::IsOSName)]               = static_cast<float>(features.IsOSName);
    Factors[static_cast<size_t>(ECacherFactors::IsOSVersion)]            = static_cast<float>(features.IsOSVersion);
    Factors[static_cast<size_t>(ECacherFactors::IsOSFamily)]             = static_cast<float>(features.IsOSFamily);
    Factors[static_cast<size_t>(ECacherFactors::IsOSFamilyAndroid)]      = static_cast<float>(features.IsOSFamilyAndroid);
    Factors[static_cast<size_t>(ECacherFactors::IsOSFamilyWindows)]      = static_cast<float>(features.IsOSFamilyWindows);
    Factors[static_cast<size_t>(ECacherFactors::IsOSFamilyiOS)]          = static_cast<float>(features.IsOSFamilyiOS);
    Factors[static_cast<size_t>(ECacherFactors::IsOSFamilyMacOS)]        = static_cast<float>(features.IsOSFamilyMacOS);
    Factors[static_cast<size_t>(ECacherFactors::IsOSFamilyLinux)]        = static_cast<float>(features.IsOSFamilyLinux);
    Factors[static_cast<size_t>(ECacherFactors::ITP)]                    = static_cast<float>(features.ITP);
    Factors[static_cast<size_t>(ECacherFactors::ITPFakeCookie)]          = static_cast<float>(features.ITPFakeCookie);
    Factors[static_cast<size_t>(ECacherFactors::localStorageSupport)]    = static_cast<float>(features.localStorageSupport);
    Factors[static_cast<size_t>(ECacherFactors::P0fOlen)]                = features.P0fOlen;
    Factors[static_cast<size_t>(ECacherFactors::P0fVersion)]             = features.P0fVersion;
    Factors[static_cast<size_t>(ECacherFactors::P0fObservedTTL)]         = features.P0fObservedTTL;
    Factors[static_cast<size_t>(ECacherFactors::P0fEOL)]                 = features.P0fEOL;
    Factors[static_cast<size_t>(ECacherFactors::P0fITTLDistance)]        = features.P0fITTLDistance;
    Factors[static_cast<size_t>(ECacherFactors::P0fUnknownOptionID)]     = features.P0fUnknownOptionID;
    Factors[static_cast<size_t>(ECacherFactors::P0fMSS)]                 = static_cast<float>(features.P0fMSS);
    Factors[static_cast<size_t>(ECacherFactors::P0fWSize)]               = static_cast<float>(features.P0fWSize);
    Factors[static_cast<size_t>(ECacherFactors::P0fScale)]               = static_cast<float>(features.P0fScale);
    Factors[static_cast<size_t>(ECacherFactors::P0fLayoutNOP)]           = static_cast<float>(features.P0fLayoutNOP);
    Factors[static_cast<size_t>(ECacherFactors::P0fLayoutMSS)]           = static_cast<float>(features.P0fLayoutMSS);
    Factors[static_cast<size_t>(ECacherFactors::P0fLayoutWS)]            = static_cast<float>(features.P0fLayoutWS);
    Factors[static_cast<size_t>(ECacherFactors::P0fLayoutSOK)]           = static_cast<float>(features.P0fLayoutSOK);
    Factors[static_cast<size_t>(ECacherFactors::P0fLayoutSACK)]          = static_cast<float>(features.P0fLayoutSACK);
    Factors[static_cast<size_t>(ECacherFactors::P0fLayoutTS)]            = static_cast<float>(features.P0fLayoutTS);
    Factors[static_cast<size_t>(ECacherFactors::P0fQuirksDF)]            = static_cast<float>(features.P0fQuirksDF);
    Factors[static_cast<size_t>(ECacherFactors::P0fQuirksIDp)]           = static_cast<float>(features.P0fQuirksIDp);
    Factors[static_cast<size_t>(ECacherFactors::P0fQuirksIDn)]           = static_cast<float>(features.P0fQuirksIDn);
    Factors[static_cast<size_t>(ECacherFactors::P0fQuirksECN)]           = static_cast<float>(features.P0fQuirksECN);
    Factors[static_cast<size_t>(ECacherFactors::P0fQuirks0p)]            = static_cast<float>(features.P0fQuirks0p);
    Factors[static_cast<size_t>(ECacherFactors::P0fQuirksFlow)]          = static_cast<float>(features.P0fQuirksFlow);
    Factors[static_cast<size_t>(ECacherFactors::P0fQuirksSEQn)]          = static_cast<float>(features.P0fQuirksSEQn);
    Factors[static_cast<size_t>(ECacherFactors::P0fQuirksACKp)]          = static_cast<float>(features.P0fQuirksACKp);
    Factors[static_cast<size_t>(ECacherFactors::P0fQuirksACKn)]          = static_cast<float>(features.P0fQuirksACKn);
    Factors[static_cast<size_t>(ECacherFactors::P0fQuirksUPTRp)]         = static_cast<float>(features.P0fQuirksUPTRp);
    Factors[static_cast<size_t>(ECacherFactors::P0fQuirksURGFp)]         = static_cast<float>(features.P0fQuirksURGFp);
    Factors[static_cast<size_t>(ECacherFactors::P0fQuirksPUSHFp)]        = static_cast<float>(features.P0fQuirksPUSHFp);
    Factors[static_cast<size_t>(ECacherFactors::P0fQuirksTS1n)]          = static_cast<float>(features.P0fQuirksTS1n);
    Factors[static_cast<size_t>(ECacherFactors::P0fQuirksTS2p)]          = static_cast<float>(features.P0fQuirksTS2p);
    Factors[static_cast<size_t>(ECacherFactors::P0fQuirksOPTp)]          = static_cast<float>(features.P0fQuirksOPTp);
    Factors[static_cast<size_t>(ECacherFactors::P0fQuirksEXWS)]          = static_cast<float>(features.P0fQuirksEXWS);
    Factors[static_cast<size_t>(ECacherFactors::P0fQuirksBad)]           = static_cast<float>(features.P0fQuirksBad);
    Factors[static_cast<size_t>(ECacherFactors::P0fPClass)]              = static_cast<float>(features.P0fPClass);

    Factors[static_cast<size_t>(ECacherFactors::Ja3TlsVersion)]          = features.Ja3.TlsVersion;
    Factors[static_cast<size_t>(ECacherFactors::Ja3CiphersLen)]          = features.Ja3.CiphersLen;
    Factors[static_cast<size_t>(ECacherFactors::Ja3ExtensionsLen)]       = features.Ja3.ExtensionsLen;

    Factors[static_cast<size_t>(ECacherFactors::Ja3C159)]                = features.Ja3.C159;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C57_61)]              = features.Ja3.C57_61;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C53)]                 = features.Ja3.C53;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C60_49187)]           = features.Ja3.C60_49187;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C53_49187)]           = features.Ja3.C53_49187;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C52393_103)]          = features.Ja3.C52393_103;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C49162)]              = features.Ja3.C49162;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C50)]                 = features.Ja3.C50;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C51)]                 = features.Ja3.C51;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C255)]                = features.Ja3.C255;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C52392)]              = features.Ja3.C52392;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C10)]                 = features.Ja3.C10;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C157_49200)]          = features.Ja3.C157_49200;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C49171_103)]          = features.Ja3.C49171_103;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C49200)]              = features.Ja3.C49200;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C49191_52394)]        = features.Ja3.C49191_52394;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C49192_52394)]        = features.Ja3.C49192_52394;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C65_52394)]           = features.Ja3.C65_52394;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C157)]                = features.Ja3.C157;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C52393_49200)]        = features.Ja3.C52393_49200;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C49159)]              = features.Ja3.C49159;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C4865)]               = features.Ja3.C4865;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C158_61)]             = features.Ja3.C158_61;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C49196_47)]           = features.Ja3.C49196_47;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C103)]                = features.Ja3.C103;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C103_49196)]          = features.Ja3.C103_49196;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C52393_49188)]        = features.Ja3.C52393_49188;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C60_65)]              = features.Ja3.C60_65;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C49195_69)]           = features.Ja3.C49195_69;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C154)]                = features.Ja3.C154;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C49187_49188)]        = features.Ja3.C49187_49188;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C49199_60)]           = features.Ja3.C49199_60;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C49195_51)]           = features.Ja3.C49195_51;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C49235)]              = features.Ja3.C49235;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C47)]                 = features.Ja3.C47;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C49169)]              = features.Ja3.C49169;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C49249)]              = features.Ja3.C49249;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C49171_60)]           = features.Ja3.C49171_60;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C49188_49196)]        = features.Ja3.C49188_49196;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C61)]                 = features.Ja3.C61;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C156_255)]            = features.Ja3.C156_255;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C47_57)]              = features.Ja3.C47_57;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C186)]                = features.Ja3.C186;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C49245)]              = features.Ja3.C49245;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C156_52394)]          = features.Ja3.C156_52394;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C20)]                 = features.Ja3.C20;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C49188_49195)]        = features.Ja3.C49188_49195;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C52394_52392)]        = features.Ja3.C52394_52392;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C53_49162)]           = features.Ja3.C53_49162;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C49191)]              = features.Ja3.C49191;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C49245_49249)]        = features.Ja3.C49245_49249;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C49171)]              = features.Ja3.C49171;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C53_52393)]           = features.Ja3.C53_52393;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C51_49199)]           = features.Ja3.C51_49199;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C49234)]              = features.Ja3.C49234;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C49315)]              = features.Ja3.C49315;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C158)]                = features.Ja3.C158;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C49187_49161)]        = features.Ja3.C49187_49161;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C107)]                = features.Ja3.C107;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C52394)]              = features.Ja3.C52394;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C49162_61)]           = features.Ja3.C49162_61;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C153)]                = features.Ja3.C153;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C49170)]              = features.Ja3.C49170;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C156)]                = features.Ja3.C156;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C52393_60)]           = features.Ja3.C52393_60;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C49195_49192)]        = features.Ja3.C49195_49192;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C7)]                  = features.Ja3.C7;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C49187_103)]          = features.Ja3.C49187_103;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C61_49172)]           = features.Ja3.C61_49172;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C159_49188)]          = features.Ja3.C159_49188;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C49171_49187)]        = features.Ja3.C49171_49187;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C49196_49188)]        = features.Ja3.C49196_49188;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C158_49161)]          = features.Ja3.C158_49161;
    Factors[static_cast<size_t>(ECacherFactors::Ja3C49193)]              = features.Ja3.C49193;

    Factors[static_cast<size_t>(ECacherFactors::AcceptUniqueKeysNumber)]           = features.AcceptUniqueKeysNumber;
    Factors[static_cast<size_t>(ECacherFactors::AcceptEncodingUniqueKeysNumber)]   = features.AcceptEncodingUniqueKeysNumber;
    Factors[static_cast<size_t>(ECacherFactors::AcceptCharsetUniqueKeysNumber)]    = features.AcceptCharsetUniqueKeysNumber;
    Factors[static_cast<size_t>(ECacherFactors::AcceptLanguageUniqueKeysNumber)]   = features.AcceptLanguageUniqueKeysNumber;
    Factors[static_cast<size_t>(ECacherFactors::AcceptAnySpace)]                   = features.AcceptAnySpace;
    Factors[static_cast<size_t>(ECacherFactors::AcceptEncodingAnySpace)]           = features.AcceptEncodingAnySpace;
    Factors[static_cast<size_t>(ECacherFactors::AcceptCharsetAnySpace)]            = features.AcceptCharsetAnySpace;
    Factors[static_cast<size_t>(ECacherFactors::AcceptLanguageAnySpace)]           = features.AcceptLanguageAnySpace;
    Factors[static_cast<size_t>(ECacherFactors::AcceptLanguageHasRussian)]         = features.AcceptLanguageHasRussian;

    Factors[static_cast<size_t>(ECacherFactors::MarketJwsStateIsDefaultExpiredRatio)]   = features.MarketJwsStatesStats.DefaultExpiredRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketJwsStateIsDefaultRatio)]          = features.MarketJwsStatesStats.DefaultRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketJwsStateIsInvalidRatio)]          = features.MarketJwsStatesStats.JwsStateIsInvalidRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketJwsStateIsSuspExpiredRatio)]      = features.MarketJwsStatesStats.SuspExpiredRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketJwsStateIsSuspRatio)]             = features.MarketJwsStatesStats.SuspRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketJwsStateIsValidExpiredRatio)]     = features.MarketJwsStatesStats.ValidExpiredRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketJwsStateIsValidRatio)]            = features.MarketJwsStatesStats.ValidRatio;

    Factors[static_cast<size_t>(ECacherFactors::MarketJa3BlockedCntRatio)]              = features.MarketJa3Stats.BlockedCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketJa3CatalogReqsCntRatio)]          = features.MarketJa3Stats.CatalogReqsCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketJa3EnemyCntRatio)]                = features.MarketJa3Stats.EnemyCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketJa3EnemyRedirectsCntRatio)]       = features.MarketJa3Stats.EnemyRedirectsCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketJa3FuidCntRatio)]                 = features.MarketJa3Stats.FuidCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketJa3HostingCntRatio)]              = features.MarketJa3Stats.HostingCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketJa3IcookieCntRatio)]              = features.MarketJa3Stats.IcookieCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketJa3Ipv4CntRatio)]                 = features.MarketJa3Stats.Ipv4CntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketJa3Ipv6CntRatio)]                 = features.MarketJa3Stats.Ipv6CntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketJa3LoginCntRatio)]                = features.MarketJa3Stats.LoginCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketJa3MobileCntRatio)]               = features.MarketJa3Stats.MobileCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketJa3OtherHandlesReqsCntRatio)]     = features.MarketJa3Stats.OtherHandlesReqsCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketJa3ProductReqsCntRatio)]          = features.MarketJa3Stats.ProductReqsCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketJa3ProxyCntRatio)]                = features.MarketJa3Stats.ProxyCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketJa3RefererIsEmptyCntRatio)]       = features.MarketJa3Stats.RefererIsEmptyCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketJa3RefererIsNotYandexCntRatio)]   = features.MarketJa3Stats.RefererIsNotYandexCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketJa3RefererIsYandexCntRatio)]      = features.MarketJa3Stats.RefererIsYandexCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketJa3RobotsCntRatio)]               = features.MarketJa3Stats.RobotsCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketJa3SearchReqsCntRatio)]           = features.MarketJa3Stats.SearchReqsCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketJa3SpravkaCntRatio)]              = features.MarketJa3Stats.SpravkaCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketJa3TorCntRatio)]                  = features.MarketJa3Stats.TorCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketJa3VpnCntRatio)]                  = features.MarketJa3Stats.VpnCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketJa3YndxIpCntRatio)]               = features.MarketJa3Stats.YndxIpCntRatio;

    Factors[static_cast<size_t>(ECacherFactors::MarketSubnetBlockedCntRatio)]              = features.MarketSubnetStats.BlockedCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketSubnetCatalogReqsCntRatio)]          = features.MarketSubnetStats.CatalogReqsCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketSubnetEnemyCntRatio)]                = features.MarketSubnetStats.EnemyCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketSubnetEnemyRedirectsCntRatio)]       = features.MarketSubnetStats.EnemyRedirectsCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketSubnetFuidCntRatio)]                 = features.MarketSubnetStats.FuidCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketSubnetHostingCntRatio)]              = features.MarketSubnetStats.HostingCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketSubnetIcookieCntRatio)]              = features.MarketSubnetStats.IcookieCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketSubnetIpv4CntRatio)]                 = features.MarketSubnetStats.Ipv4CntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketSubnetIpv6CntRatio)]                 = features.MarketSubnetStats.Ipv6CntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketSubnetLoginCntRatio)]                = features.MarketSubnetStats.LoginCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketSubnetMobileCntRatio)]               = features.MarketSubnetStats.MobileCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketSubnetOtherHandlesReqsCntRatio)]     = features.MarketSubnetStats.OtherHandlesReqsCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketSubnetProductReqsCntRatio)]          = features.MarketSubnetStats.ProductReqsCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketSubnetProxyCntRatio)]                = features.MarketSubnetStats.ProxyCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketSubnetRefererIsEmptyCntRatio)]       = features.MarketSubnetStats.RefererIsEmptyCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketSubnetRefererIsNotYandexCntRatio)]   = features.MarketSubnetStats.RefererIsNotYandexCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketSubnetRefererIsYandexCntRatio)]      = features.MarketSubnetStats.RefererIsYandexCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketSubnetRobotsCntRatio)]               = features.MarketSubnetStats.RobotsCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketSubnetSearchReqsCntRatio)]           = features.MarketSubnetStats.SearchReqsCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketSubnetSpravkaCntRatio)]              = features.MarketSubnetStats.SpravkaCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketSubnetTorCntRatio)]                  = features.MarketSubnetStats.TorCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketSubnetVpnCntRatio)]                  = features.MarketSubnetStats.VpnCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketSubnetYndxIpCntRatio)]               = features.MarketSubnetStats.YndxIpCntRatio;

    Factors[static_cast<size_t>(ECacherFactors::MarketUABlockedCntRatio)]              = features.MarketUAStats.BlockedCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketUACatalogReqsCntRatio)]          = features.MarketUAStats.CatalogReqsCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketUAEnemyCntRatio)]                = features.MarketUAStats.EnemyCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketUAEnemyRedirectsCntRatio)]       = features.MarketUAStats.EnemyRedirectsCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketUAFuidCntRatio)]                 = features.MarketUAStats.FuidCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketUAHostingCntRatio)]              = features.MarketUAStats.HostingCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketUAIcookieCntRatio)]              = features.MarketUAStats.IcookieCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketUAIpv4CntRatio)]                 = features.MarketUAStats.Ipv4CntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketUAIpv6CntRatio)]                 = features.MarketUAStats.Ipv6CntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketUALoginCntRatio)]                = features.MarketUAStats.LoginCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketUAMobileCntRatio)]               = features.MarketUAStats.MobileCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketUAOtherHandlesReqsCntRatio)]     = features.MarketUAStats.OtherHandlesReqsCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketUAProductReqsCntRatio)]          = features.MarketUAStats.ProductReqsCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketUAProxyCntRatio)]                = features.MarketUAStats.ProxyCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketUARefererIsEmptyCntRatio)]       = features.MarketUAStats.RefererIsEmptyCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketUARefererIsNotYandexCntRatio)]   = features.MarketUAStats.RefererIsNotYandexCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketUARefererIsYandexCntRatio)]      = features.MarketUAStats.RefererIsYandexCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketUARobotsCntRatio)]               = features.MarketUAStats.RobotsCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketUASearchReqsCntRatio)]           = features.MarketUAStats.SearchReqsCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketUASpravkaCntRatio)]              = features.MarketUAStats.SpravkaCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketUATorCntRatio)]                  = features.MarketUAStats.TorCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketUAVpnCntRatio)]                  = features.MarketUAStats.VpnCntRatio;
    Factors[static_cast<size_t>(ECacherFactors::MarketUAYndxIpCntRatio)]               = features.MarketUAStats.YndxIpCntRatio;

    Factors[static_cast<size_t>(ECacherFactors::AutoruJa3)]                    = features.AutoruJa3;
    Factors[static_cast<size_t>(ECacherFactors::AutoruSubnet)]                 = features.AutoruSubnet;
    Factors[static_cast<size_t>(ECacherFactors::AutoruUA)]                     = features.AutoruUA;

    Factors[static_cast<size_t>(ECacherFactors::HasCookieAmcuid)]                      = static_cast<float>(features.HasCookieAmcuid);
    Factors[static_cast<size_t>(ECacherFactors::HasCookieCurrentRegionId)]             = static_cast<float>(features.HasCookieCurrentRegionId);
    Factors[static_cast<size_t>(ECacherFactors::HasCookieCycada)]                      = static_cast<float>(features.HasCookieCycada);
    Factors[static_cast<size_t>(ECacherFactors::HasCookieLocalOffersFirst)]            = static_cast<float>(features.HasCookieLocalOffersFirst);
    Factors[static_cast<size_t>(ECacherFactors::HasCookieLr)]                          = static_cast<float>(features.HasCookieLr);
    Factors[static_cast<size_t>(ECacherFactors::HasCookieMarketYs)]                    = static_cast<float>(features.HasCookieMarketYs);
    Factors[static_cast<size_t>(ECacherFactors::HasCookieOnstock)]                     = static_cast<float>(features.HasCookieOnstock);
    Factors[static_cast<size_t>(ECacherFactors::HasCookieYandexHelp)]                  = static_cast<float>(features.HasCookieYandexHelp);
    Factors[static_cast<size_t>(ECacherFactors::HasCookieCmpMerge)]                    = static_cast<float>(features.HasCookieCmpMerge);
    Factors[static_cast<size_t>(ECacherFactors::HasCookieComputer)]                    = static_cast<float>(features.HasCookieComputer);
    Factors[static_cast<size_t>(ECacherFactors::HasCookieHeadBaner)]                   = static_cast<float>(features.HasCookieHeadBaner);
    Factors[static_cast<size_t>(ECacherFactors::HasCookieUtmSource)]                   = static_cast<float>(features.HasCookieUtmSource);

    Factors[static_cast<size_t>(ECacherFactors::CookieYoungerThanMinute)]                   = features.CookieYoungerMinute;
    Factors[static_cast<size_t>(ECacherFactors::CookieYoungerThanHour)]                     = features.CookieYoungerHour;
    Factors[static_cast<size_t>(ECacherFactors::CookieYoungerThanDay)]                      = features.CookieYoungerDay;
    Factors[static_cast<size_t>(ECacherFactors::CookieYoungerThanWeek)]                     = features.CookieYoungerWeek;
    Factors[static_cast<size_t>(ECacherFactors::CookieYoungerThanMonth)]                    = features.CookieYoungerMonth;
    Factors[static_cast<size_t>(ECacherFactors::CookieYoungerThanThreeMonthes)]             = features.CookieYoungerThreeMonthes;
    Factors[static_cast<size_t>(ECacherFactors::CookieOlderThanMonth)]                      = features.CookieOlderMonth;
    Factors[static_cast<size_t>(ECacherFactors::CookieOlderThanThreeMonthes)]               = features.CookieOlderThreeMonthes;

    for (size_t i = 0; i < features.LastVisits.size(); ++i) {
        Factors[static_cast<size_t>(ECacherFactors::NumStaticFactors) + i] = features.LastVisits[i];
    }
}

void FillCacherFactors(NAntirobotEvClass::TCacherFactors& cf, const TCacherRequestFeatures& features, float result, float threshold) {
    cf.SetNumDocs(               features.NumDocs);
    cf.SetPageNum(               features.PageNum);
    cf.SetHeadersCount(          features.HeadersCount);
    cf.SetIsProxy(               features.IsProxy);
    cf.SetIsTor(                 features.IsTor);
    cf.SetIsVpn(                 features.IsVpn);
    cf.SetIsHosting(             features.IsHosting);
    cf.SetRefererFromYandex(     features.RefererFromYandex);
    cf.SetIsBadProtocol(         features.IsBadProtocol);
    cf.SetIsBadUserAgent(        features.IsBadUserAgent);
    cf.SetIsBadUserAgentNew(     features.IsBadUserAgentNew);
    cf.SetIsConnectionKeepAlive( features.IsConnectionKeepAlive);
    cf.SetHaveUnknownHeaders(    features.HaveUnknownHeaders);
    cf.SetIpSubnetMatch(         features.IpSubnetMatch);
    cf.SetSpravkaLifetime(       features.SpravkaLifetime);
    cf.SetFraudJa3(              features.FraudJa3);
    cf.SetFraudSubnet(           features.FraudSubnet);
    cf.SetFraudSubnetNew(        features.FraudSubnetNew);
    cf.SetSecFetchDest(          features.SecFetchDest);
    cf.SetAcceptLanguage(        features.AcceptLanguage);
    cf.SetCookie(                features.Cookie);
    cf.SetUpgradeInsecureRequests(features.UpgradeInsecureRequests);
    cf.SetAcceptEncoding(        features.AcceptEncoding);
    cf.SetDnt(                   features.Dnt);
    cf.SetOrigin(                features.Origin);
    cf.SetUserAgent(             features.UserAgent);
    cf.SetHost(                  features.Host);
    cf.SetReferer(               features.Referer);
    cf.SetAuthority(             features.Authority);
    cf.SetCacheControl(          features.CacheControl);
    cf.SetXForwardedProto(       features.XForwardedProto);
    cf.SetKeepAlive(             features.KeepAlive);
    cf.SetPragma(                features.Pragma);
    cf.SetProxyConnection(       features.ProxyConnection);
    cf.SetRtt(                   features.Rtt);
    cf.SetChromeHodor(           features.ChromeHodor);
    cf.SetCgiParamText(          features.CgiParamPresent[FindArrayIndex(CacherCgiParamName, "text")]);
    cf.SetCgiParamLr(            features.CgiParamPresent[FindArrayIndex(CacherCgiParamName, "lr")]);
    cf.SetCgiParamClid(          features.CgiParamPresent[FindArrayIndex(CacherCgiParamName, "clid")]);
    cf.SetCgiParamTld(           features.CgiParamPresent[FindArrayIndex(CacherCgiParamName, "tld")]);
    cf.SetCgiParamUrl(           features.CgiParamPresent[FindArrayIndex(CacherCgiParamName, "url")]);
    cf.SetCgiParamSite(          features.CgiParamPresent[FindArrayIndex(CacherCgiParamName, "site")]);
    cf.SetCgiParamLang(          features.CgiParamPresent[FindArrayIndex(CacherCgiParamName, "lang")]);
    cf.SetCookieL(               features.CookiePresent[FindArrayIndex(CacherCookieName, "L")]);
    cf.SetCookieMy(              features.CookiePresent[FindArrayIndex(CacherCookieName, "my")]);
    cf.SetCookieSessionId(       features.CookiePresent[FindArrayIndex(CacherCookieName, "Session_id")]);
    cf.SetCookieYabsFrequency(   features.CookiePresent[FindArrayIndex(CacherCookieName, "yabs-frequency")]);
    cf.SetCookieYandexLogin(     features.CookiePresent[FindArrayIndex(CacherCookieName, "yandex_login")]);
    cf.SetCookieYandexuid(       features.CookiePresent[FindArrayIndex(CacherCookieName, "yandexuid")]);
    cf.SetCookieYs(              features.CookiePresent[FindArrayIndex(CacherCookieName, "ys")]);
    cf.SetCookieFuid01(          features.CookiePresent[FindArrayIndex(CacherCookieName, "fuid01")]);
    cf.SetHeaderAcceptEncoding(  features.HttpHeaderPresent[FindArrayIndex(CacherHttpHeaderName, "Accept-Encoding")]);
    cf.SetHeaderAcceptLanguage(  features.HttpHeaderPresent[FindArrayIndex(CacherHttpHeaderName, "Accept-Language")]);
    cf.SetHeaderAuthorization(   features.HttpHeaderPresent[FindArrayIndex(CacherHttpHeaderName, "Authorization")]);
    cf.SetHeaderCacheControl(    features.HttpHeaderPresent[FindArrayIndex(CacherHttpHeaderName, "Cache-Control")]);
    cf.SetHeaderCookie(          features.HttpHeaderPresent[FindArrayIndex(CacherHttpHeaderName, "Cookie")]);
    cf.SetHeaderConnection(      features.HttpHeaderPresent[FindArrayIndex(CacherHttpHeaderName, "Connection")]);
    cf.SetHeaderContentLength(   features.HttpHeaderPresent[FindArrayIndex(CacherHttpHeaderName, "Content-Length")]);
    cf.SetHeaderContentType(     features.HttpHeaderPresent[FindArrayIndex(CacherHttpHeaderName, "Content-Type")]);
    cf.SetHeaderReferer(         features.HttpHeaderPresent[FindArrayIndex(CacherHttpHeaderName, "Referer")]);
    cf.SetHeaderUserAgent(       features.HttpHeaderPresent[FindArrayIndex(CacherHttpHeaderName, "User-Agent")]);
    cf.SetHeaderXForwardedFor(   features.HttpHeaderPresent[FindArrayIndex(CacherHttpHeaderName, "X-Forwarded-For")]);
    cf.SetInRobotSet(            features.InRobotSet);
    cf.SetHasValidSpravka(       features.HasValidSpravka);
    cf.SetIsRobot(               features.IsRobot);
    cf.SetIsMobile(              features.IsMobile);
    cf.SetIsBrowser(             features.IsBrowser);
    cf.SetHistorySupport(        features.HistorySupport);
    cf.SetIsEmulator(            features.IsEmulator);
    cf.SetIsBrowserEngine(       features.IsBrowserEngine);
    cf.SetIsBrowserEngineVersion(features.IsBrowserEngineVersion);
    cf.SetIsBrowserVersion(      features.IsBrowserVersion);
    cf.SetIsOSName(              features.IsOSName);
    cf.SetIsOSVersion(           features.IsOSVersion);
    cf.SetIsOSFamily(            features.IsOSFamily);
    cf.SetIsOSFamilyAndroid(     features.IsOSFamilyAndroid);
    cf.SetIsOSFamilyWindows(     features.IsOSFamilyWindows);
    cf.SetIsOSFamilyiOS(         features.IsOSFamilyiOS);
    cf.SetIsOSFamilyMacOS(       features.IsOSFamilyMacOS);
    cf.SetIsOSFamilyLinux(       features.IsOSFamilyLinux);
    cf.SetITP(                   features.ITP);
    cf.SetITPFakeCookie(         features.ITPFakeCookie);
    cf.SetlocalStorageSupport(   features.localStorageSupport);
    cf.SetFormulaResult(         result);
    cf.SetFormulaThreshold(      threshold);
    cf.SetP0fOlen(               features.P0fOlen);
    cf.SetP0fVersion(            features.P0fVersion);
    cf.SetP0fObservedTTL(        features.P0fObservedTTL);
    cf.SetP0fEOL(                features.P0fEOL);
    cf.SetP0fITTLDistance(       features.P0fITTLDistance);
    cf.SetP0fUnknownOptionID(    features.P0fUnknownOptionID);
    cf.SetP0fMSS(                features.P0fMSS);
    cf.SetP0fWSize(              features.P0fWSize);
    cf.SetP0fScale(              features.P0fScale);
    cf.SetP0fLayoutNOP(          features.P0fLayoutNOP);
    cf.SetP0fLayoutMSS(          features.P0fLayoutMSS);
    cf.SetP0fLayoutWS(           features.P0fLayoutWS);
    cf.SetP0fLayoutSOK(          features.P0fLayoutSOK);
    cf.SetP0fLayoutSACK(         features.P0fLayoutSACK);
    cf.SetP0fLayoutTS(           features.P0fLayoutTS);
    cf.SetP0fQuirksDF(           features.P0fQuirksDF);
    cf.SetP0fQuirksIDp(          features.P0fQuirksIDp);
    cf.SetP0fQuirksIDn(          features.P0fQuirksIDn);
    cf.SetP0fQuirksECN(          features.P0fQuirksECN);
    cf.SetP0fQuirks0p(           features.P0fQuirks0p);
    cf.SetP0fQuirksFlow(         features.P0fQuirksFlow);
    cf.SetP0fQuirksSEQn(         features.P0fQuirksSEQn);
    cf.SetP0fQuirksACKp(         features.P0fQuirksACKp);
    cf.SetP0fQuirksACKn(         features.P0fQuirksACKn);
    cf.SetP0fQuirksUPTRp(        features.P0fQuirksUPTRp);
    cf.SetP0fQuirksURGFp(        features.P0fQuirksURGFp);
    cf.SetP0fQuirksPUSHFp(       features.P0fQuirksPUSHFp);
    cf.SetP0fQuirksTS1n(         features.P0fQuirksTS1n);
    cf.SetP0fQuirksTS2p(         features.P0fQuirksTS2p);
    cf.SetP0fQuirksOPTp(         features.P0fQuirksOPTp);
    cf.SetP0fQuirksEXWS(         features.P0fQuirksEXWS);
    cf.SetP0fQuirksBad(          features.P0fQuirksBad);
    cf.SetP0fPClass(             features.P0fPClass);
    cf.SetJa3TlsVersion(         features.Ja3.TlsVersion);
    cf.SetJa3CiphersLen(         features.Ja3.CiphersLen);
    cf.SetJa3ExtensionsLen(      features.Ja3.ExtensionsLen);
    cf.SetJa3C159(               features.Ja3.C159);
    cf.SetJa3C57_61(             features.Ja3.C57_61);
    cf.SetJa3C53(                features.Ja3.C53);
    cf.SetJa3C60_49187(          features.Ja3.C60_49187);
    cf.SetJa3C53_49187(          features.Ja3.C53_49187);
    cf.SetJa3C52393_103(         features.Ja3.C52393_103);
    cf.SetJa3C49162(             features.Ja3.C49162);
    cf.SetJa3C50(                features.Ja3.C50);
    cf.SetJa3C51(                features.Ja3.C51);
    cf.SetJa3C255(               features.Ja3.C255);
    cf.SetJa3C52392(             features.Ja3.C52392);
    cf.SetJa3C10(                features.Ja3.C10);
    cf.SetJa3C157_49200(         features.Ja3.C157_49200);
    cf.SetJa3C49171_103(         features.Ja3.C49171_103);
    cf.SetJa3C49200(             features.Ja3.C49200);
    cf.SetJa3C49191_52394(       features.Ja3.C49191_52394);
    cf.SetJa3C49192_52394(       features.Ja3.C49192_52394);
    cf.SetJa3C65_52394(          features.Ja3.C65_52394);
    cf.SetJa3C157(               features.Ja3.C157);
    cf.SetJa3C52393_49200(       features.Ja3.C52393_49200);
    cf.SetJa3C49159(             features.Ja3.C49159);
    cf.SetJa3C4865(              features.Ja3.C4865);
    cf.SetJa3C158_61(            features.Ja3.C158_61);
    cf.SetJa3C49196_47(          features.Ja3.C49196_47);
    cf.SetJa3C103(               features.Ja3.C103);
    cf.SetJa3C103_49196(         features.Ja3.C103_49196);
    cf.SetJa3C52393_49188(       features.Ja3.C52393_49188);
    cf.SetJa3C60_65(             features.Ja3.C60_65);
    cf.SetJa3C49195_69(          features.Ja3.C49195_69);
    cf.SetJa3C154(               features.Ja3.C154);
    cf.SetJa3C49187_49188(       features.Ja3.C49187_49188);
    cf.SetJa3C49199_60(          features.Ja3.C49199_60);
    cf.SetJa3C49195_51(          features.Ja3.C49195_51);
    cf.SetJa3C49235(             features.Ja3.C49235);
    cf.SetJa3C47(                features.Ja3.C47);
    cf.SetJa3C49169(             features.Ja3.C49169);
    cf.SetJa3C49249(             features.Ja3.C49249);
    cf.SetJa3C49171_60(          features.Ja3.C49171_60);
    cf.SetJa3C49188_49196(       features.Ja3.C49188_49196);
    cf.SetJa3C61(                features.Ja3.C61);
    cf.SetJa3C156_255(           features.Ja3.C156_255);
    cf.SetJa3C47_57(             features.Ja3.C47_57);
    cf.SetJa3C186(               features.Ja3.C186);
    cf.SetJa3C49245(             features.Ja3.C49245);
    cf.SetJa3C156_52394(         features.Ja3.C156_52394);
    cf.SetJa3C20(                features.Ja3.C20);
    cf.SetJa3C49188_49195(       features.Ja3.C49188_49195);
    cf.SetJa3C52394_52392(       features.Ja3.C52394_52392);
    cf.SetJa3C53_49162(          features.Ja3.C53_49162);
    cf.SetJa3C49191(             features.Ja3.C49191);
    cf.SetJa3C49245_49249(       features.Ja3.C49245_49249);
    cf.SetJa3C49171(             features.Ja3.C49171);
    cf.SetJa3C53_52393(          features.Ja3.C53_52393);
    cf.SetJa3C51_49199(          features.Ja3.C51_49199);
    cf.SetJa3C49234(             features.Ja3.C49234);
    cf.SetJa3C49315(             features.Ja3.C49315);
    cf.SetJa3C158(               features.Ja3.C158);
    cf.SetJa3C49187_49161(       features.Ja3.C49187_49161);
    cf.SetJa3C107(               features.Ja3.C107);
    cf.SetJa3C52394(             features.Ja3.C52394);
    cf.SetJa3C49162_61(          features.Ja3.C49162_61);
    cf.SetJa3C153(               features.Ja3.C153);
    cf.SetJa3C49170(             features.Ja3.C49170);
    cf.SetJa3C156(               features.Ja3.C156);
    cf.SetJa3C52393_60(          features.Ja3.C52393_60);
    cf.SetJa3C49195_49192(       features.Ja3.C49195_49192);
    cf.SetJa3C7(                 features.Ja3.C7);
    cf.SetJa3C49187_103(         features.Ja3.C49187_103);
    cf.SetJa3C61_49172(          features.Ja3.C61_49172);
    cf.SetJa3C159_49188(         features.Ja3.C159_49188);
    cf.SetJa3C49171_49187(       features.Ja3.C49171_49187);
    cf.SetJa3C49196_49188(       features.Ja3.C49196_49188);
    cf.SetJa3C158_49161(         features.Ja3.C158_49161);
    cf.SetJa3C49193(             features.Ja3.C49193);
    cf.SetAcceptUniqueKeysNumber(           features.AcceptUniqueKeysNumber);
    cf.SetAcceptEncodingUniqueKeysNumber(   features.AcceptEncodingUniqueKeysNumber);
    cf.SetAcceptCharsetUniqueKeysNumber(    features.AcceptCharsetUniqueKeysNumber);
    cf.SetAcceptLanguageUniqueKeysNumber(   features.AcceptLanguageUniqueKeysNumber);
    cf.SetAcceptAnySpace(                   features.AcceptAnySpace);
    cf.SetAcceptEncodingAnySpace(           features.AcceptEncodingAnySpace);
    cf.SetAcceptCharsetAnySpace(            features.AcceptCharsetAnySpace);
    cf.SetAcceptLanguageAnySpace(           features.AcceptLanguageAnySpace);
    cf.SetAcceptLanguageHasRussian(         features.AcceptLanguageHasRussian);

    cf.SetMarketJwsStateIsDefaultExpiredRatio(  features.MarketJwsStatesStats.DefaultExpiredRatio);
    cf.SetMarketJwsStateIsDefaultRatio(         features.MarketJwsStatesStats.DefaultRatio);
    cf.SetMarketJwsStateIsInvalidRatio(         features.MarketJwsStatesStats.JwsStateIsInvalidRatio);
    cf.SetMarketJwsStateIsSuspExpiredRatio(     features.MarketJwsStatesStats.SuspExpiredRatio);
    cf.SetMarketJwsStateIsSuspRatio(            features.MarketJwsStatesStats.SuspRatio);
    cf.SetMarketJwsStateIsValidExpiredRatio(    features.MarketJwsStatesStats.ValidExpiredRatio);
    cf.SetMarketJwsStateIsValidRatio(           features.MarketJwsStatesStats.ValidRatio);

    cf.SetMarketJa3BlockedCntRatio(             features.MarketJa3Stats.BlockedCntRatio);
    cf.SetMarketJa3CatalogReqsCntRatio(         features.MarketJa3Stats.CatalogReqsCntRatio);
    cf.SetMarketJa3EnemyCntRatio(               features.MarketJa3Stats.EnemyCntRatio);
    cf.SetMarketJa3EnemyRedirectsCntRatio(      features.MarketJa3Stats.EnemyRedirectsCntRatio);
    cf.SetMarketJa3FuidCntRatio(                features.MarketJa3Stats.FuidCntRatio);
    cf.SetMarketJa3HostingCntRatio(             features.MarketJa3Stats.HostingCntRatio);
    cf.SetMarketJa3IcookieCntRatio(             features.MarketJa3Stats.IcookieCntRatio);
    cf.SetMarketJa3Ipv4CntRatio(                features.MarketJa3Stats.Ipv4CntRatio);
    cf.SetMarketJa3Ipv6CntRatio(                features.MarketJa3Stats.Ipv6CntRatio);
    cf.SetMarketJa3LoginCntRatio(               features.MarketJa3Stats.LoginCntRatio);
    cf.SetMarketJa3MobileCntRatio(              features.MarketJa3Stats.MobileCntRatio);
    cf.SetMarketJa3OtherHandlesReqsCntRatio(    features.MarketJa3Stats.OtherHandlesReqsCntRatio);
    cf.SetMarketJa3ProductReqsCntRatio(         features.MarketJa3Stats.ProductReqsCntRatio);
    cf.SetMarketJa3ProxyCntRatio(               features.MarketJa3Stats.ProxyCntRatio);
    cf.SetMarketJa3RefererIsEmptyCntRatio(      features.MarketJa3Stats.RefererIsEmptyCntRatio);
    cf.SetMarketJa3RefererIsNotYandexCntRatio(  features.MarketJa3Stats.RefererIsNotYandexCntRatio);
    cf.SetMarketJa3RefererIsYandexCntRatio(     features.MarketJa3Stats.RefererIsYandexCntRatio);
    cf.SetMarketJa3RobotsCntRatio(              features.MarketJa3Stats.RobotsCntRatio);
    cf.SetMarketJa3SearchReqsCntRatio(          features.MarketJa3Stats.SearchReqsCntRatio);
    cf.SetMarketJa3SpravkaCntRatio(             features.MarketJa3Stats.SpravkaCntRatio);
    cf.SetMarketJa3TorCntRatio(                 features.MarketJa3Stats.TorCntRatio);
    cf.SetMarketJa3VpnCntRatio(                 features.MarketJa3Stats.VpnCntRatio);
    cf.SetMarketJa3YndxIpCntRatio(              features.MarketJa3Stats.YndxIpCntRatio);

    cf.SetMarketSubnetBlockedCntRatio(             features.MarketSubnetStats.BlockedCntRatio);
    cf.SetMarketSubnetCatalogReqsCntRatio(         features.MarketSubnetStats.CatalogReqsCntRatio);
    cf.SetMarketSubnetEnemyCntRatio(               features.MarketSubnetStats.EnemyCntRatio);
    cf.SetMarketSubnetEnemyRedirectsCntRatio(      features.MarketSubnetStats.EnemyRedirectsCntRatio);
    cf.SetMarketSubnetFuidCntRatio(                features.MarketSubnetStats.FuidCntRatio);
    cf.SetMarketSubnetHostingCntRatio(             features.MarketSubnetStats.HostingCntRatio);
    cf.SetMarketSubnetIcookieCntRatio(             features.MarketSubnetStats.IcookieCntRatio);
    cf.SetMarketSubnetIpv4CntRatio(                features.MarketSubnetStats.Ipv4CntRatio);
    cf.SetMarketSubnetIpv6CntRatio(                features.MarketSubnetStats.Ipv6CntRatio);
    cf.SetMarketSubnetLoginCntRatio(               features.MarketSubnetStats.LoginCntRatio);
    cf.SetMarketSubnetMobileCntRatio(              features.MarketSubnetStats.MobileCntRatio);
    cf.SetMarketSubnetOtherHandlesReqsCntRatio(    features.MarketSubnetStats.OtherHandlesReqsCntRatio);
    cf.SetMarketSubnetProductReqsCntRatio(         features.MarketSubnetStats.ProductReqsCntRatio);
    cf.SetMarketSubnetProxyCntRatio(               features.MarketSubnetStats.ProxyCntRatio);
    cf.SetMarketSubnetRefererIsEmptyCntRatio(      features.MarketSubnetStats.RefererIsEmptyCntRatio);
    cf.SetMarketSubnetRefererIsNotYandexCntRatio(  features.MarketSubnetStats.RefererIsNotYandexCntRatio);
    cf.SetMarketSubnetRefererIsYandexCntRatio(     features.MarketSubnetStats.RefererIsYandexCntRatio);
    cf.SetMarketSubnetRobotsCntRatio(              features.MarketSubnetStats.RobotsCntRatio);
    cf.SetMarketSubnetSearchReqsCntRatio(          features.MarketSubnetStats.SearchReqsCntRatio);
    cf.SetMarketSubnetSpravkaCntRatio(             features.MarketSubnetStats.SpravkaCntRatio);
    cf.SetMarketSubnetTorCntRatio(                 features.MarketSubnetStats.TorCntRatio);
    cf.SetMarketSubnetVpnCntRatio(                 features.MarketSubnetStats.VpnCntRatio);
    cf.SetMarketSubnetYndxIpCntRatio(              features.MarketSubnetStats.YndxIpCntRatio);

    cf.SetMarketUABlockedCntRatio(             features.MarketUAStats.BlockedCntRatio);
    cf.SetMarketUACatalogReqsCntRatio(         features.MarketUAStats.CatalogReqsCntRatio);
    cf.SetMarketUAEnemyCntRatio(               features.MarketUAStats.EnemyCntRatio);
    cf.SetMarketUAEnemyRedirectsCntRatio(      features.MarketUAStats.EnemyRedirectsCntRatio);
    cf.SetMarketUAFuidCntRatio(                features.MarketUAStats.FuidCntRatio);
    cf.SetMarketUAHostingCntRatio(             features.MarketUAStats.HostingCntRatio);
    cf.SetMarketUAIcookieCntRatio(             features.MarketUAStats.IcookieCntRatio);
    cf.SetMarketUAIpv4CntRatio(                features.MarketUAStats.Ipv4CntRatio);
    cf.SetMarketUAIpv6CntRatio(                features.MarketUAStats.Ipv6CntRatio);
    cf.SetMarketUALoginCntRatio(               features.MarketUAStats.LoginCntRatio);
    cf.SetMarketUAMobileCntRatio(              features.MarketUAStats.MobileCntRatio);
    cf.SetMarketUAOtherHandlesReqsCntRatio(    features.MarketUAStats.OtherHandlesReqsCntRatio);
    cf.SetMarketUAProductReqsCntRatio(         features.MarketUAStats.ProductReqsCntRatio);
    cf.SetMarketUAProxyCntRatio(               features.MarketUAStats.ProxyCntRatio);
    cf.SetMarketUARefererIsEmptyCntRatio(      features.MarketUAStats.RefererIsEmptyCntRatio);
    cf.SetMarketUARefererIsNotYandexCntRatio(  features.MarketUAStats.RefererIsNotYandexCntRatio);
    cf.SetMarketUARefererIsYandexCntRatio(     features.MarketUAStats.RefererIsYandexCntRatio);
    cf.SetMarketUARobotsCntRatio(              features.MarketUAStats.RobotsCntRatio);
    cf.SetMarketUASearchReqsCntRatio(          features.MarketUAStats.SearchReqsCntRatio);
    cf.SetMarketUASpravkaCntRatio(             features.MarketUAStats.SpravkaCntRatio);
    cf.SetMarketUATorCntRatio(                 features.MarketUAStats.TorCntRatio);
    cf.SetMarketUAVpnCntRatio(                 features.MarketUAStats.VpnCntRatio);
    cf.SetMarketUAYndxIpCntRatio(              features.MarketUAStats.YndxIpCntRatio);

    cf.SetAutoruJa3(                    features.AutoruJa3);
    cf.SetAutoruSubnet(                 features.AutoruSubnet);
    cf.SetAutoruUA(                     features.AutoruUA);

    cf.SetHasCookieAmcuid(                     features.HasCookieAmcuid);
    cf.SetHasCookieCurrentRegionId(            features.HasCookieCurrentRegionId);
    cf.SetHasCookieCycada(                     features.HasCookieCycada);
    cf.SetHasCookieLocalOffersFirst(           features.HasCookieLocalOffersFirst);
    cf.SetHasCookieLr(                         features.HasCookieLr);
    cf.SetHasCookieMarketYs(                   features.HasCookieMarketYs);
    cf.SetHasCookieOnstock(                    features.HasCookieOnstock);
    cf.SetHasCookieYandexHelp(                 features.HasCookieYandexHelp);
    cf.SetHasCookieCmpMerge(                   features.HasCookieCmpMerge);
    cf.SetHasCookieComputer(                   features.HasCookieComputer);
    cf.SetHasCookieHeadBaner(                  features.HasCookieHeadBaner);
    cf.SetHasCookieUtmSource(                  features.HasCookieUtmSource);

    cf.SetCookieYoungerThanMinute(                    features.CookieYoungerMinute);
    cf.SetCookieYoungerThanHour(                      features.CookieYoungerHour);
    cf.SetCookieYoungerThanDay(                       features.CookieYoungerDay);
    cf.SetCookieYoungerThanWeek(                      features.CookieYoungerWeek);
    cf.SetCookieYoungerThanMonth(                     features.CookieYoungerMonth);
    cf.SetCookieYoungerThanThreeMonthes(              features.CookieYoungerThreeMonthes);
    cf.SetCookieOlderThanMonth(                       features.CookieOlderMonth);
    cf.SetCookieYoungerThanThreeMonthes(              features.CookieOlderThreeMonthes);

    const auto lastVisits = cf.MutableLastVisits();
    lastVisits->Reserve(static_cast<int>(features.LastVisits.size()));

    for (const auto lastVisit : features.LastVisits) {
        lastVisits->Add(lastVisit);
    }
}

} // namespace NAntiRobot
