#pragma once

#include <util/system/defaults.h>

namespace NSrvKernel {

    // Some of the errors will result in cookie rejection by a modern chromium.
    // Some are actually recoverable or have recoverable cases yet most probably they are signs of bugs potentially
    // resulting in a corrupted state on the client. Wherefore we will stop such cookies from reaching the user agents.
    enum class ESetCookieSyntaxError : ui32 {
        // A control or nonascii character in the cookie.
        // The cookie will be rejected by modern browsers
        InvalidOctet,

        // Neither name nor value provided.
        // The cookie will be rejected by most browsers.
        CookieEmpty,

        // A nameless cookie: '=value;' or 'value;'.
        // There can be only one nameless cookie and in some browsers without a '=' it is treated as a valueless cookie.
        // Ok in most browsers but probably a bug.
        CookieNameEmpty,

        // A double semicolon: '; ;' in the cookie.
        // Ok in most browsers but definitely a bug.
        AttrEmpty,
        // An attr name is empty: ';=attrValue' in the cookie.
        // Ok in most browsers but definitely a bug.
        AttrNameEmpty,
        // An unknown attr (not one of Path, Domain, Expires, Max-Age, SameSite, Secure, HttpOnly).
        // The cookie will be rejected by some old browsers.
        // This is probably a bug.
        AttrUnknown,

        // An empty or invalid Path is treated as if no path was specified.
        // Ok in most browsers but probably a bug.
        PathValueInvalid,
        // Conflicting Paths.
        // Ok in most browsers but probably a bug.
        PathValueConflict,

        // An empty or invalid Domain.
        // The cookie will be rejected by all browsers. An empty Domain is ok in most browers but probably a bug.
        DomainValueInvalid,
        // Conflicting Domains.
        // Ok in most browsers but probably a bug.
        DomainValueConflict,

        // An empty or invalid Expires.
        // Accepted by most browsers but the cookie will be treated as a session cookie.
        // We consider this to be a bug.
        ExpiresValueInvalid,
        // Conflicting Expires.
        // Ok in most browsers but probably a bug.
        ExpiresValueConflict,

        // An empty or invalid Max-Age.
        // Negative ones and those out of 32bit range are considered invalid as well.
        // Accepted in most browsers but the cookie will be treated as a session cookie.
        // We consider this to be a bug.
        MaxAgeValueInvalid,
        // Conflicting Max-Age.
        // Ok in most browsers but probably a bug.
        MaxAgeValueConflict,

        // An unknown or missing SameSite value (not one of None, Lax, Strict).
        // Will be treated as either SameSite=Lax or SameSite=None depending on a browser.
        // Most probably a bug.
        SameSiteValueInvalid,
        // Conflicting SameSite.
        // Ok in most browsers but probably a bug.
        SameSiteValueConflict,
    };

    // Most of them will result in cookie rejection by a modern chromium
    enum class ESetCookieSemanticError : ui32 {
        // The cookie is longer than 4096 bytes.
        // The cookie will be rejected by most browsers.
        TooLong,

        // The cookie name is one of Path, Expires, Domain, Max-Age, SameSite, Secure, HttpOnly.
        // Ok but likely a bug.
        CookieNameSuspicious,
        // A __Host- or __Secure- cookie on an insecure connection.
        // The cookie will be rejected by modern browsers.
        CookieNameRequiresHttps,

        // The request url is not on cookie Path.
        // The cookie will be rejected by all browsers.
        PathValueUrlMismatch,
        // A Path which differs from '/' in a __Host- cookie.
        // The cookie will be rejected by modern browsers.
        PathValueForbidden,
        // No Path specified in a __Host- cookie.
        // The cookie will be rejected by modern browsers.
        PathAttrRequired,

        // The request host does not match the cookie Domain.
        // The cookie will be rejected by all browsers.
        DomainValueHostMismatch,
        // Having a Domain in a __Host- cookie.
        // The cookie will be rejected by modern browsers.
        DomainAttrForbidden,

        // A Secure attr is required (a SameSite=None, __Secure- or __Host- cookie).
        // The cookie will be rejected by modern browsers.
        SecureAttrRequired,

        // Max-Age and Expires attrs contradict each other.
        // Max-Age is prioritized in the modern browsers but the situation is likely a bug.
        DeletionConflict,
    };
}
