#pragma once

#include <balancer/kernel/net/addr.h>
#include <util/datetime/base.h>
#include <util/string/builder.h>

namespace NSrvKernel::NDns {

struct TDnsServer {
    bool IsSet() const { return !IPv4.empty() && Port; }

    TString  IPv4;
    uint16_t Port = 0;
};

struct TResolverConfig {
    TResolverConfig() {}

    TResolverConfig(TDuration dnsTimeout, TDuration recordTtl, TDuration errorTtl,
                    const TString& ip, uint16_t port,
                    TString itsSwitchDnsFile, TDuration itsSwitchDnsCheckTimeout, bool useAsync)
        : DnsTimeout(dnsTimeout)
        , RecordTtl(recordTtl)
        , ErrorTtl(errorTtl)
        , DnsServer{ip, port}
        , SwitchDnsFile(itsSwitchDnsFile)
        , SwitchDnsCheckTimeout(itsSwitchDnsCheckTimeout)
        , UseAsync(useAsync)
    {
        // Check params for limited values
        Y_ASSERT(DnsTimeout < TDuration::Hours(1));
        Y_ASSERT(RecordTtl < TDuration::Days(30));
        Y_ASSERT(ErrorTtl < TDuration::Days(1));
        Y_ASSERT(!SwitchDnsFile.Empty());
        Y_ASSERT(SwitchDnsCheckTimeout < TDuration::Minutes(15));
    }

    TString ToString() const
    {
        return TStringBuilder() << (UseAsync ? "Async " : "Sync ") << ", DNS resolve(s) " << DnsTimeout
                << ", TTL(h) " << RecordTtl << ", Error TTL(s) " << ErrorTtl
                << (DnsServer.IsSet() ? ", has server" : "")
                << ", ITS " << SwitchDnsFile << ", ITS check(s)" << SwitchDnsCheckTimeout;
    }

    TDuration  DnsTimeout     = TDuration::Seconds(1);          //!< max time to resolve by DNS
    TDuration  RecordTtl      = TDuration::Hours(3);            //!< TTL for record in DNS cache
    TDuration  ErrorTtl       = TDuration::Seconds(1);          //!< TTL for error record in DNS cache
    TDnsServer DnsServer;                                       //!< DNS server, for functional tests
    TString    SwitchDnsFile  = "./controls/async_dns";         //!< ITS file
    TDuration  SwitchDnsCheckTimeout = TDuration::Seconds(1);   //!< check ITS timeout
    bool       UseAsync       = false;                          //!< use c-ares resolver
};

}