#include <balancer/modules/cookie_policy/policies/default_yandex_policies.h>
#include <balancer/modules/cookie_policy/ut_common/ut_common.h>
#include <balancer/kernel/helpers/yuid.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NModCookiePolicy;

// TODO(velavokr): better tests, split the cases apart

Y_UNIT_TEST_SUITE(TDefaultYandexPoliciesTest) {
    namespace {
        auto MakeWatchOnlyConfig() {
            auto cfg = MakeCombinedPolicy(R"=(instance={
                uuid="xxx";
                default_yandex_policies="stable";
            })=");
            AddDefaultYandexPolicies(cfg);
            return cfg;
        }

        auto MakeStableFixConfig() {
            auto cfg = MakeCombinedPolicy(R"=(instance={
                uuid="xxx";
                default_yandex_policies="stable_fix";
            })=");
            AddDefaultYandexPolicies(cfg);
            return cfg;
        }

        auto MakeUnstableConfig() {
            auto cfg = MakeCombinedPolicy(R"=(instance={
                uuid="xxx";
                default_yandex_policies="unstable";
            })=");
            AddDefaultYandexPolicies(cfg);
            return cfg;
        }

        auto MakeFixAllConfig() {
            auto cfg = MakeCombinedPolicy(R"=(instance={
                uuid="xxx";
                default_yandex_policies="fix_all";
            })=");
            AddDefaultYandexPolicies(cfg);
            return cfg;
        }

        TStrVec MakeCookie() {
            return {"is_gdpr_b=CJq6ShDIARgB; gdpr_popup=1; yandexuid=5432102345678987654; client_other=uuu"};
        }

        TStrVec MakeSetCookie() {
            return {
                "yandexuid=5432102345678987654; httponly; samesite=None; max-age=1",
                "yandexuid=1111111111111111111; max-age=1",
                "server_other=yyy; max-age=160000000",
                "lah=vvv",
                "invalid_format",
                "gdpr_popup=0",
            };
        }
    }

    Y_UNIT_TEST(TestWatchOnly) {
        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeWatchOnlyConfig(),
                .Host={"yandex.ru"},
            },
            .Out={
                .Stats=FixStatsNoGdpr({
                    {"cpol-xxx-eprivacy_lifetime-off_summ", 1},
                    {"cpol-xxx-eprivacy_lifetime-total_summ", 1},
                    {"cpol-xxx-eprivacy_client-off_summ", 1},
                    {"cpol-xxx-eprivacy_client-total_summ", 1},
                    {"cpol-xxx-eprivacy_server-off_summ", 1},
                    {"cpol-xxx-eprivacy_server-total_summ", 1},
                    {"cpol-xxx-gdpr_protected-pass_summ", 1},
                    {"cpol-xxx-gdpr_protected-total_summ", 1},
                    {"cpol-xxx-passport_protected-pass_summ", 1},
                    {"cpol-xxx-passport_protected-total_summ", 1},
                    {"cpol-xxx-yandex_js-pass_summ", 1},
                    {"cpol-xxx-yandex_js-total_summ", 1},
                    {"cpol-xxx-yandexuid_life-pass_summ", 1},
                    {"cpol-xxx-yandexuid_life-total_summ", 1},
                    {"cpol-xxx-yandex_root-pass_summ", 1},
                    {"cpol-xxx-yandex_root-total_summ", 1},
                    {"cpol-xxx-yandex_sec-pass_summ", 1},
                    {"cpol-xxx-yandex_sec-total_summ", 1},
                    {"cpol-xxx-yandex_ssn-pass_summ", 1},
                    {"cpol-xxx-yandex_ssn-total_summ", 1},
                    {"cpol-xxx-yandexuid_new-off_summ", 1},
                    {"cpol-xxx-yandexuid_new-total_summ", 1},
                    {"cpol-xxx-yandexuid_val-pass_summ", 1},
                    {"cpol-xxx-yandexuid_val-total_summ", 1},
                }, 0),
            },
        }));

        const auto now = TInstant::Seconds(1600327485);
        // max amount of violations
        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeWatchOnlyConfig(),
                .Now=now,
                .Host={"yandex.eu"},
                .UserAgent={TString(ChromimUA)},
                .Cookie=MakeCookie(),
                .SetCookie=MakeSetCookie(),
                .WithStats=false,
            },
            .Out={
                .Gdpr={.Value={.Mode=EGdpr::KeepTech,.IsGdpr=1},.SafeValue={.Mode=EGdpr::KeepTech,.IsGdpr=1}},
                .Cookie=MakeCookie(),
                .SetCookie=MakeSetCookie(),
                .ReqLog=" GdprDomain=1 GdprPopup=1 IsGdprB=1,?",
                .RespLog=" [cpol u:xxx-parser fail+dryRun (invalid_format)=Drop:1!CookieNameEmpty,]"
                         " [cpol u:xxx-eprivacy_client fail+dryRun client_other=Add:2!GdprDelete,]"
                         " [cpol u:xxx-eprivacy_lifetime fail+dryRun server_other=Fix:1!BigLifetime,]"
                         " [cpol u:xxx-eprivacy_server fail+dryRun server_other=Drop:1!GdprDelete,]"
                         " [cpol u:xxx-gdpr_protected fail+dryRun gdpr_popup=Drop:1!Forbidden,]"
                         " [cpol u:xxx-passport_protected fail+dryRun lah=Drop:1!Forbidden,]"
                         " [cpol u:xxx-yandexuid_val fail+dryRun yandexuid=Drop:1!ValueChange,]"
                         " [cpol u:xxx-yandexuid_life fail+dryRun yandexuid=Fix:1!SmallLifetime, yandexuid=Fix:1!SmallLifetime,]"
                         " [cpol u:xxx-yandex_root fail+dryRun yandexuid=Fix:1!NotGlobalScope, yandexuid=Fix:1!NotGlobalScope,]"
                         " [cpol u:xxx-yandex_js fail+dryRun yandexuid=Fix:1!HttpOnly,]"
                         " [cpol u:xxx-yandex_sec fail+dryRun yandexuid=Fix:1!NotSecure, yandexuid=Fix:1!NotSecure,]"
                         " [cpol u:xxx-yandex_ssn fail+dryRun yandexuid=Fix:1!NotSecure, yandexuid=Fix:1!BadSameSite,]",
            },
        }));
    }

    Y_UNIT_TEST(TestStableFix) {
        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeStableFixConfig(),
                .Host={"yandex.ru"},
            },
            .Out={
                .Stats=FixStatsNoGdpr({
                    {"cpol-xxx-eprivacy_lifetime-off_summ", 1},
                    {"cpol-xxx-eprivacy_lifetime-total_summ", 1},
                    {"cpol-xxx-eprivacy_client-off_summ", 1},
                    {"cpol-xxx-eprivacy_client-total_summ", 1},
                    {"cpol-xxx-eprivacy_server-off_summ", 1},
                    {"cpol-xxx-eprivacy_server-total_summ", 1},
                    {"cpol-xxx-gdpr_protected-pass_summ", 1},
                    {"cpol-xxx-gdpr_protected-total_summ", 1},
                    {"cpol-xxx-passport_protected-pass_summ", 1},
                    {"cpol-xxx-passport_protected-total_summ", 1},
                    {"cpol-xxx-yandex_js-pass_summ", 1},
                    {"cpol-xxx-yandex_js-total_summ", 1},
                    {"cpol-xxx-yandexuid_life-pass_summ", 1},
                    {"cpol-xxx-yandexuid_life-total_summ", 1},
                    {"cpol-xxx-yandex_root-pass_summ", 1},
                    {"cpol-xxx-yandex_root-total_summ", 1},
                    {"cpol-xxx-yandex_sec-pass_summ", 1},
                    {"cpol-xxx-yandex_sec-total_summ", 1},
                    {"cpol-xxx-yandex_ssn-pass_summ", 1},
                    {"cpol-xxx-yandex_ssn-total_summ", 1},
                    {"cpol-xxx-yandexuid_new-off_summ", 1},
                    {"cpol-xxx-yandexuid_new-total_summ", 1},
                    {"cpol-xxx-yandexuid_val-pass_summ", 1},
                    {"cpol-xxx-yandexuid_val-total_summ", 1},
                }, 0),
            },
        }));

        const auto now = TInstant::Seconds(1600327485);
        // max amount of violations
        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeStableFixConfig(),
                .Now=now,
                .Path="/",
                .Host={"yandex.eu"},
                .UserAgent={TString(ChromimUA)},
                .Cookie=MakeCookie(),
                .SetCookie={
                    "yandexuid=5432102345678987654; httponly; samesite=None; max-age=1",
                    "yandexuid=1111111111111111111; max-age=1",
                    "server_other=yyy; max-age=123",
                    "lah=vvv",
                    "i=www; max-age=160000000",
                    "invalid_format",
                },
                .WithStats=false,
            },
            .Out={
                .GeoCache={.XIpProperties=Nothing(),.XYandexEURequest=Nothing()},
                .Gdpr={.Value={.Mode=EGdpr::KeepTech,.IsGdpr=1},.SafeValue={.Mode=EGdpr::KeepTech,.IsGdpr=1}},
                .Cookie={
                    "is_gdpr_b=CJq6ShDIARgB; gdpr_popup=1; yandexuid=5432102345678987654; client_other=uuu",
                },
                .SetCookie={
                    "yandexuid=5432102345678987654; Max-Age=1; SameSite=None; Secure",
                    "yandexuid=1111111111111111111; Max-Age=1; SameSite=None; Secure",
                    "server_other=yyy; max-age=123",
                    "i=www; Max-Age=63072000; SameSite=None; Secure",
                    "invalid_format",
                },
                .ReqLog=" GdprDomain=1 GdprPopup=1 IsGdprB=1,?",
                .RespLog=" [cpol u:xxx-parser fail+dryRun (invalid_format)=Drop:1!CookieNameEmpty,]"
                         " [cpol u:xxx-eprivacy_client fail+dryRun client_other=Add:2!GdprDelete,]"
                         " [cpol u:xxx-eprivacy_lifetime fail+fix i=Fix:1!BigLifetime,]"
                         " [cpol u:xxx-eprivacy_server fail+dryRun server_other=Drop:1!GdprDelete,]"
                         " [cpol u:xxx-passport_protected fail+fix lah=Drop:1!Forbidden,]"
                         " [cpol u:xxx-yandexuid_val fail+dryRun yandexuid=Drop:1!ValueChange,]"
                         " [cpol u:xxx-yandexuid_life fail+dryRun yandexuid=Fix:1!SmallLifetime, yandexuid=Fix:1!SmallLifetime,]"
                         " [cpol u:xxx-yandex_root fail+dryRun yandexuid=Fix:1!NotGlobalScope, yandexuid=Fix:1!NotGlobalScope, i=Fix:1!NotGlobalScope,]"
                         " [cpol u:xxx-yandex_js fail+fix yandexuid=Fix:1!HttpOnly,]"
                         " [cpol u:xxx-yandex_sec fail+dryRun yandexuid=Fix:1!NotSecure, yandexuid=Fix:1!NotSecure, i=Fix:1!NotSecure,]"
                         " [cpol u:xxx-yandex_ssn fail+fix yandexuid=Fix:1!NotSecure, yandexuid=Fix:1!BadSameSite, i=Fix:1!BadSameSite,]",
            }
        }));
    }

    Y_UNIT_TEST(TestUnstable) {
        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeUnstableConfig(),
                .Host={"yandex.ru"},
            },
            .Out={
                .Stats=FixStatsNoGdpr({
                    {"cpol-xxx-eprivacy_lifetime-off_summ", 1},
                    {"cpol-xxx-eprivacy_lifetime-total_summ", 1},
                    {"cpol-xxx-eprivacy_client-off_summ", 1},
                    {"cpol-xxx-eprivacy_client-total_summ", 1},
                    {"cpol-xxx-eprivacy_server-off_summ", 1},
                    {"cpol-xxx-eprivacy_server-total_summ", 1},
                    {"cpol-xxx-is_gdpr-off_summ", 1},
                    {"cpol-xxx-is_gdpr-total_summ", 1},
                    {"cpol-xxx-is_gdpr_b-off_summ", 1},
                    {"cpol-xxx-is_gdpr_b-total_summ", 1},
                    {"cpol-xxx-gdpr_protected-pass_summ", 1},
                    {"cpol-xxx-gdpr_protected-total_summ", 1},
                    {"cpol-xxx-passport_protected-pass_summ", 1},
                    {"cpol-xxx-passport_protected-total_summ", 1},
                    {"cpol-xxx-yandex_js-pass_summ", 1},
                    {"cpol-xxx-yandex_js-total_summ", 1},
                    {"cpol-xxx-yandexuid_life-pass_summ", 1},
                    {"cpol-xxx-yandexuid_life-total_summ", 1},
                    {"cpol-xxx-yandex_root-pass_summ", 1},
                    {"cpol-xxx-yandex_root-total_summ", 1},
                    {"cpol-xxx-yandex_sec-pass_summ", 1},
                    {"cpol-xxx-yandex_sec-total_summ", 1},
                    {"cpol-xxx-yandex_ssn-pass_summ", 1},
                    {"cpol-xxx-yandex_ssn-total_summ", 1},
                    {"cpol-xxx-yandexuid_new-off_summ", 1},
                    {"cpol-xxx-yandexuid_new-total_summ", 1},
                    {"cpol-xxx-yandexuid_val-pass_summ", 1},
                    {"cpol-xxx-yandexuid_val-total_summ", 1},
                }, 0),
            },
        }));

        const auto now = TInstant::Seconds(1600327485);
        TIsGdprB isGdprB = {
            .BalancerVerHash=BalancerVerHash(),
            .Tstamp=now,
            .IsGdpr=1,
            .IsGdprNoVpn=1,
        };
        // max amount of violations
        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeUnstableConfig(),
                .Now=now,
                .Path="/",
                .Host={"yandex.eu"},
                .UserAgent={TString(ChromimUA)},
                .XIpProperties={"OABAAQ=="},
                .Cookie=MakeCookie(),
                .SetCookie=MakeSetCookie(),
                .WithStats=false,
            },
            .Out={
                .GeoCache={.XIpProperties=TXIpProperties{.IsGdpr=1,.IsVpn=0},.XYandexEURequest=Nothing()},
                .Gdpr={.Value={.Mode=EGdpr::KeepTech,.IsGdpr=1},.SafeValue={.Mode=EGdpr::KeepTech,.IsGdpr=1}},
                .Cookie={
                    "gdpr_popup=1; yandexuid=5432102345678987654; client_other=uuu; is_gdpr=1; is_gdpr_b=" + isGdprB.Render(),
                },
                .SetCookie={
                    "yandexuid=5432102345678987654; Path=/; Domain=.yandex.eu; Max-Age=1; SameSite=None; Secure",
                    "yandexuid=1111111111111111111; Path=/; Domain=.yandex.eu; Max-Age=1; SameSite=None; Secure",
                    "gdpr_popup=0",
                    "client_other=; Path=/; Expires=Fri, 21 Dec 2012 12:12:12 GMT; Max-Age=0; Secure",
                    "client_other=; Path=/; Domain=.yandex.eu; Expires=Fri, 21 Dec 2012 12:12:12 GMT; Max-Age=0; Secure",
                    "is_gdpr=1; Path=/; Domain=.yandex.eu; Expires=Sat, 17 Sep 2022 07:24:45 GMT; SameSite=None; Secure",
                    "is_gdpr_b=" + isGdprB.Render() + "; Path=/; Domain=.yandex.eu; Expires=Sat, 17 Sep 2022 07:24:45 GMT; SameSite=None; Secure",
                },
                .ReqLog=" GdprDomain=1 GdprPopup=1 XIpProperties=1,1 IsGdprB=1,?"
                        " [cpol-req u:xxx-is_gdpr fail+fix is_gdpr=Add:1!Create,]"
                        " [cpol-req u:xxx-is_gdpr_b fail+fix is_gdpr_b=Fix:1!Update,]",
                .RespLog=" [cpol u:xxx-parser fail+fix (invalid_format)=Drop:1!CookieNameEmpty,]"
                         " [cpol u:xxx-eprivacy_client fail+fix client_other=Add:2!GdprDelete,]"
                         " [cpol u:xxx-eprivacy_lifetime fail+fix server_other=Fix:1!BigLifetime,]"
                         " [cpol u:xxx-eprivacy_server fail+fix server_other=Drop:1!GdprDelete,]"
                         " [cpol u:xxx-is_gdpr fail+fix is_gdpr=Add:1!Create,]"
                         " [cpol u:xxx-is_gdpr_b fail+fix is_gdpr_b=Add:1!Update,]"
                         " [cpol u:xxx-gdpr_protected fail+dryRun gdpr_popup=Drop:1!Forbidden,]"
                         " [cpol u:xxx-passport_protected fail+fix lah=Drop:1!Forbidden,]"
                         " [cpol u:xxx-yandexuid_val fail+dryRun yandexuid=Drop:1!ValueChange,]"
                         " [cpol u:xxx-yandexuid_life fail+dryRun yandexuid=Fix:1!SmallLifetime, yandexuid=Fix:1!SmallLifetime,]"
                         " [cpol u:xxx-yandex_root fail+fix yandexuid=Fix:1!NotGlobalScope, yandexuid=Fix:1!NotGlobalScope,]"
                         " [cpol u:xxx-yandex_js fail+fix yandexuid=Fix:1!HttpOnly,]"
                         " [cpol u:xxx-yandex_sec fail+fix yandexuid=Fix:1!NotSecure, yandexuid=Fix:1!NotSecure,]"
                         " [cpol u:xxx-yandex_ssn fail+fix yandexuid=Fix:1!BadSameSite,]",
            },
        }));
    }

    Y_UNIT_TEST(TestFixAll) {
        const auto now = TInstant::Seconds(1600327485);
        const auto rnd = 555555555;
        const auto val = GetYuid(rnd, now);

        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeFixAllConfig(),
                .Now=now,
                .Random=rnd,
                .Host={"yandex.ru"},
            },
            .Out={
                .Cookie={"yandexuid=" + val},
                .SetCookie=ToCookie({TSetCookie{
                    .Name="yandexuid",
                    .Value=TBlob::NoCopy(val.data(), val.size()),
                    .Path="/",
                    .Domain=".yandex.ru",
                    .Expires=TExpires::FromInstant(now + TDuration::Days(365*10)),
                    .Secure=true,
                }}),
                .ReqLog=" [cpol-req u:xxx-yandexuid_new fail+fix yandexuid=Add:1!Create,]",
                .RespLog=" [cpol u:xxx-yandexuid_new fail+fix yandexuid=Add:1!Create,]",
                .Stats=FixStatsNoGdpr({
                    {"cpol-xxx-modified-req_summ", 1},
                    {"cpol-xxx-modified-resp_summ", 1},
                    {"cpol-xxx-eprivacy_lifetime-off_summ", 1},
                    {"cpol-xxx-eprivacy_lifetime-total_summ", 1},
                    {"cpol-xxx-eprivacy_client-off_summ", 1},
                    {"cpol-xxx-eprivacy_client-total_summ", 1},
                    {"cpol-xxx-eprivacy_server-off_summ", 1},
                    {"cpol-xxx-eprivacy_server-total_summ", 1},
                    {"cpol-xxx-is_gdpr-off_summ", 1},
                    {"cpol-xxx-is_gdpr-total_summ", 1},
                    {"cpol-xxx-is_gdpr_b-off_summ", 1},
                    {"cpol-xxx-is_gdpr_b-total_summ", 1},
                    {"cpol-xxx-gdpr_protected-pass_summ", 1},
                    {"cpol-xxx-gdpr_protected-total_summ", 1},
                    {"cpol-xxx-passport_protected-pass_summ", 1},
                    {"cpol-xxx-passport_protected-total_summ", 1},
                    {"cpol-xxx-yandex_js-pass_summ", 1},
                    {"cpol-xxx-yandex_js-total_summ", 1},
                    {"cpol-xxx-yandexuid_life-pass_summ", 1},
                    {"cpol-xxx-yandexuid_life-total_summ", 1},
                    {"cpol-xxx-yandex_root-pass_summ", 1},
                    {"cpol-xxx-yandex_root-total_summ", 1},
                    {"cpol-xxx-yandex_sec-pass_summ", 1},
                    {"cpol-xxx-yandex_sec-total_summ", 1},
                    {"cpol-xxx-yandex_ssn-pass_summ", 1},
                    {"cpol-xxx-yandex_ssn-total_summ", 1},
                    {"cpol-xxx-yandexuid_new-cookie-add_summ", 1},
                    {"cpol-xxx-yandexuid_new-fail_summ", 1},
                    {"cpol-xxx-yandexuid_new-fix_summ", 1},
                    {"cpol-xxx-yandexuid_new-req-cookie-add_summ", 1},
                    {"cpol-xxx-yandexuid_new-total_summ", 1},
                    {"cpol-xxx-yandexuid_val-pass_summ", 1},
                    {"cpol-xxx-yandexuid_val-total_summ", 1},
                }, 0),
            },
        }));

        TIsGdprB isGdprB = {
            .BalancerVerHash=BalancerVerHash(),
            .Tstamp=now,
            .IsGdpr=1
        };
        // max amount of violations
        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeFixAllConfig(),
                .Now=now,
                .Path="/",
                .Host={"yandex.eu"},
                .UserAgent={TString(ChromimUA)},
                .XYandexEURequest={"1"},
                .Cookie={"gdpr_popup=1; yandexuid=5432102345678987654; client_other=uuu"},
                .SetCookie={
                    "yandexuid=5432102345678987654; httponly; samesite=None; max-age=1",
                    "yandexuid=1111111111111111111; max-age=1",
                    "server_other=yyy; max-age=160000000",
                    "lah=vvv",
                    "i=www; max-age=160000000",
                    "invalid_format",
                },
                .WithStats=false,
            },
            .Out={
                .GeoCache={.XIpProperties=Nothing(),.XYandexEURequest=1},
                .Gdpr={.Value={.Mode=EGdpr::KeepTech,.IsGdpr=1},.SafeValue={.Mode=EGdpr::KeepTech,.IsGdpr=1}},
                .Cookie={
                    "gdpr_popup=1; yandexuid=5432102345678987654; client_other=uuu; is_gdpr=1; is_gdpr_b="
                    + isGdprB.Render(),
                },
                .SetCookie={
                    "yandexuid=5432102345678987654; Path=/; Domain=.yandex.eu; Max-Age=30240000; SameSite=None; Secure",
                    "i=www; Path=/; Domain=.yandex.eu; Max-Age=63072000; SameSite=None; Secure",
                    "client_other=; Path=/; Expires=Fri, 21 Dec 2012 12:12:12 GMT; Max-Age=0; Secure",
                    "client_other=; Path=/; Domain=.yandex.eu; Expires=Fri, 21 Dec 2012 12:12:12 GMT; Max-Age=0; Secure",
                    "is_gdpr=1; Path=/; Domain=.yandex.eu; Expires=Sat, 17 Sep 2022 07:24:45 GMT; SameSite=None; Secure",
                    "is_gdpr_b=" + isGdprB.Render() + "; Path=/; Domain=.yandex.eu; Expires=Sat, 17 Sep 2022 07:24:45 GMT; SameSite=None; Secure",
                },
                .ReqLog=" GdprDomain=1 GdprPopup=1 XYandexEURequest=1"
                        " [cpol-req u:xxx-is_gdpr fail+fix is_gdpr=Add:1!Create,]"
                        " [cpol-req u:xxx-is_gdpr_b fail+fix is_gdpr_b=Add:1!Create,]",
                .RespLog=" [cpol u:xxx-parser fail+fix (invalid_format)=Drop:1!CookieNameEmpty,]"
                         " [cpol u:xxx-eprivacy_client fail+fix client_other=Add:2!GdprDelete,]"
                         " [cpol u:xxx-eprivacy_lifetime fail+fix server_other=Fix:1!BigLifetime, i=Fix:1!BigLifetime,]"
                         " [cpol u:xxx-eprivacy_server fail+fix server_other=Drop:1!GdprDelete,]"
                         " [cpol u:xxx-is_gdpr fail+fix is_gdpr=Add:1!Create,]"
                         " [cpol u:xxx-is_gdpr_b fail+fix is_gdpr_b=Add:1!Create,]"
                         " [cpol u:xxx-passport_protected fail+fix lah=Drop:1!Forbidden,]"
                         " [cpol u:xxx-yandexuid_val fail+fix yandexuid=Drop:1!ValueChange,]"
                         " [cpol u:xxx-yandexuid_life fail+fix yandexuid=Fix:1!SmallLifetime,]"
                         " [cpol u:xxx-yandex_root fail+fix yandexuid=Fix:1!NotGlobalScope, i=Fix:1!NotGlobalScope,]"
                         " [cpol u:xxx-yandex_js fail+fix yandexuid=Fix:1!HttpOnly,]"
                         " [cpol u:xxx-yandex_sec fail+fix yandexuid=Fix:1!NotSecure, i=Fix:1!NotSecure,]"
                         " [cpol u:xxx-yandex_ssn fail+fix i=Fix:1!BadSameSite,]",
            },
        }));
    }
}
