#include <balancer/modules/cookie_policy/ut_common/ut_common.h>

#include <balancer/kernel/cookie/gdpr/lifetime.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NModCookiePolicy;

Y_UNIT_TEST_SUITE(TPolicyGdprLifetimeTest) {
    namespace {
        auto MakeConfig(bool fix) {
            return MakeCombinedPolicy(Sprintf(R"=(instance={
                uuid="xxx";
                gdpr_lifetime={
                    yyy={
                        %s
                    };
                };
            })=", (fix ? "mode='fix';" : "")));
        }

        auto MakeCookies(TInstant now) {
            const auto y3 = TDuration::Days(3 * 365);
            const auto expires3y = *TExpires::FromInstant(now + TDuration::Days(3 * 365));
            return ToCookie({
                {.Name="i", .Expires=expires3y, .MaxAge=y3.Seconds()},
                {.Name="yandexuid", .Expires=TMaxAllowedExpires::Get(now), .MaxAge=MaxAllowedAge.Seconds()},
                {.Name="whatever_other", .Expires=expires3y, .MaxAge=y3.Seconds()},
                {.Name="whatever_sess",},
                {.Name="whatever_del_max_age", .MaxAge=0},
                {.Name="whatever_del_expires", .Expires=TExpires::Past()},
            });
        }

        auto MakeFixedCookies(TInstant now) {
            return ToCookie({
                {.Name="i", .Expires=TMaxAllowedExpires::Get(now), .MaxAge=MaxAllowedAge.Seconds()},
                {.Name="yandexuid", .Expires=TMaxAllowedExpires::Get(now), .MaxAge=MaxAllowedAge.Seconds()},
                {.Name="whatever_other", .Expires=TMaxAllowedExpires::Get(now), .MaxAge=MaxAllowedAge.Seconds()},
                {.Name="whatever_sess",},
                {.Name="whatever_del_max_age", .MaxAge=0},
                {.Name="whatever_del_expires", .Expires=TExpires::Past()},
            });
        }
    }

    Y_UNIT_TEST(TestSkipNonGdpr) {
        TInstant now = TInstant::Seconds(1600092462);
        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeConfig(true),
                .Now=now,
                .Host={"yandex.eu"},
                .SetCookie=MakeCookies(now)
            },
            .Out={
                .SetCookie=MakeCookies(now),
                .ReqLog=" GdprDomain=1",
                .Stats=FixStatsNoGdpr({
                    {"cpol-xxx-gdpr-domain_summ", 1},
                    {"cpol-xxx-yyy-off_summ", 1},
                    {"cpol-xxx-yyy-total_summ", 1},
                }, MakeCookies(now).size()),
            },
        }));

        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeConfig(true),
                .Now=now,
                .Host={"yandex.eu"},
                // {"BalancerVerHash":1361575,"Tstamp":1599858000,"IsGdpr":0,"IsGdprNoVpn":1}
                .Cookie={"is_gdpr_b=CKeNUxBIKAE="},
                .SetCookie=MakeCookies(now)
            },
            .Out={
                .Cookie={"is_gdpr_b=CKeNUxBIKAE="},
                .SetCookie=MakeCookies(now),
                .ReqLog=" GdprDomain=1 IsGdprB=0,1",
                .Stats=FixStats({
                    {"cpol-xxx-gdpr-domain_summ", 1},
                    {"cpol-xxx-gdpr-geo-0_summ", 1},
                    {"cpol-xxx-gdpr-mod-0_summ", 1},
                    {"cpol-xxx-gdpr-src-IsGdprB_summ", 1},
                    {"cpol-xxx-gdpr-safe-geo-0_summ", 1},
                    {"cpol-xxx-gdpr-safe-mod-0_summ", 1},
                    {"cpol-xxx-gdpr-safe-src-IsGdprB_summ", 1},
                    {"cpol-xxx-IsGdprB-cookie_summ", 1},
                    {"cpol-xxx-yyy-off_summ", 1},
                    {"cpol-xxx-yyy-total_summ", 1},
                }, MakeCookies(now).size()),
            },
        }));
    }

    Y_UNIT_TEST(TestGdpr) {
        TInstant now = TInstant::Seconds(1600092462);

        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeConfig(true),
                .Now=now,
                .XYandexEURequest={"1"},
                .SetCookie=MakeCookies(now)
            },
            .Out={
                .GeoCache={.XYandexEURequest=1},
                .Gdpr={
                    .Value={.Mode=EGdpr::KeepAll, .IsGdpr=1},
                    .SafeValue={.Mode=EGdpr::KeepAll, .IsGdpr=1},
                },
                .SetCookie=MakeFixedCookies(now),
                .ReqLog=" XYandexEURequest=1",
                .RespLog=" [cpol u:xxx-yyy fail+fix i=Fix:1!BigLifetime, whatever_other=Fix:1!BigLifetime,]",
                .Stats=FixStats({
                    {"cpol-xxx-modified-resp_summ", 1},
                    {"cpol-xxx-gdpr-geo-1_summ", 1},
                    {"cpol-xxx-gdpr-mod-0_summ", 1},
                    {"cpol-xxx-gdpr-src-XYandexEURequest_summ", 1},
                    {"cpol-xxx-gdpr-safe-geo-1_summ", 1},
                    {"cpol-xxx-gdpr-safe-mod-0_summ", 1},
                    {"cpol-xxx-gdpr-safe-src-XYandexEURequest_summ", 1},
                    {"cpol-xxx-XYandexEURequest_summ", 1},
                    {"cpol-xxx-yyy-cookie-fail_summ", 2},
                    {"cpol-xxx-yyy-cookie-fix_summ", 2},
                    {"cpol-xxx-yyy-cookie-pass_summ", 1},
                    {"cpol-xxx-yyy-cookie-skip_summ", 3},
                    {"cpol-xxx-yyy-cookie-total_summ", 6},
                    {"cpol-xxx-yyy-fail_summ", 1},
                    {"cpol-xxx-yyy-fix_summ", 1},
                    {"cpol-xxx-yyy-total_summ", 1},
                }, MakeCookies(now).size()),
            },
        }));
    }

    Y_UNIT_TEST(TestGdprDryRun) {
        TInstant now = TInstant::Seconds(1600092462);

        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeConfig(false),
                .Now=now,
                .Host={"yandex.eu"},
                .XYandexEURequest={"1"},
                .SetCookie=MakeCookies(now)
            },
            .Out={
                .GeoCache={.XYandexEURequest=1},
                .Gdpr={
                    .Value={.Mode=EGdpr::KeepAll, .IsGdpr=1},
                    .SafeValue={.Mode=EGdpr::KeepAll, .IsGdpr=1},
                },
                .SetCookie=MakeCookies(now),
                .ReqLog=" GdprDomain=1 XYandexEURequest=1",
                .RespLog=" [cpol u:xxx-yyy fail+dryRun i=Fix:1!BigLifetime, whatever_other=Fix:1!BigLifetime,]",
                .Stats=FixStats({
                    {"cpol-xxx-gdpr-domain_summ", 1},
                    {"cpol-xxx-gdpr-geo-1_summ", 1},
                    {"cpol-xxx-gdpr-mod-0_summ", 1},
                    {"cpol-xxx-gdpr-src-XYandexEURequest_summ", 1},
                    {"cpol-xxx-gdpr-safe-geo-1_summ", 1},
                    {"cpol-xxx-gdpr-safe-mod-0_summ", 1},
                    {"cpol-xxx-gdpr-safe-src-XYandexEURequest_summ", 1},
                    {"cpol-xxx-XYandexEURequest_summ", 1},
                    {"cpol-xxx-yyy-cookie-fail_summ", 2},
                    {"cpol-xxx-yyy-cookie-fix-dryRun_summ", 2},
                    {"cpol-xxx-yyy-cookie-pass_summ", 1},
                    {"cpol-xxx-yyy-cookie-skip_summ", 3},
                    {"cpol-xxx-yyy-cookie-total_summ", 6},
                    {"cpol-xxx-yyy-fail_summ", 1},
                    {"cpol-xxx-yyy-fix-dryRun_summ", 1},
                    {"cpol-xxx-yyy-total_summ", 1},
                }, MakeCookies(now).size()),
            },
        }));
    }
}
