#include <balancer/modules/cookie_policy/ut_common/ut_common.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NModCookiePolicy;

Y_UNIT_TEST_SUITE(TPolicyGlobalScopeTest) {
    namespace {
        auto MakeConfig(bool fix) {
            return MakeCombinedPolicy(Sprintf(R"=(instance={
                uuid="xxx";
                global_scope={
                    yyy={
                        name_re="yandexuid.*";
                        %s
                    };
                };
            })=", (fix ? "mode='fix'" : "")));
        }

        auto MakeCookies() {
            return ToCookie({
                {.Name="yandexuid_ok", .Path="/", .Domain=".yandex.ru"},
                {.Name="yandexuid_ok2", .Path="/", .Domain="yandex.ru"},
                {.Name="yandexuid_nopath", .Domain=".yandex.ru"},
                {.Name="yandexuid_locpath", .Path="/xxx/", .Domain=".yandex.ru"},
                {.Name="yandexuid_locpathdomain", .Path="/xxx/", .Domain=".blogs.yandex.ru"},
                {.Name="yandexuid_wrongdomain", .Path="/", .Domain=".yandex.com"},
                {.Name="yandexuid_baddomain", .Path="/", .Domain="127.0.0.1"},
                {.Name="yandexuid_nodomain", .Path="/",},
                {.Name="other",},
            });
        }
    }

    Y_UNIT_TEST(TestBadHost) {
        for (auto&& h : std::initializer_list<TStrVec>{{}, {"localhost"}, {"127.0.0.1"}}) {
            Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
                .In={
                    .Cfg=MakeConfig(true),
                    .Host=h,
                    .SetCookie=ToCookie({{.Name="yandexuid",},}),
                },
                .Out={
                    .SetCookie=ToCookie({{.Name="yandexuid",},}),
                    .Stats=FixStatsNoGdpr({
                        {"cpol-xxx-yyy-off_summ", 1},
                        {"cpol-xxx-yyy-total_summ", 1},
                    }, 1),
                },
            }));
        }
    }

    Y_UNIT_TEST(TestPolicy) {
        for (auto h : {"yandex.ru", "blogs.yandex.ru", "yandex.ru."}) {
            Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
                .In={
                    .Cfg=MakeConfig(true),
                    .Host={h},
                    .SetCookie=MakeCookies(),
                },
                .Out={
                    .SetCookie=ToCookie({
                        {.Name="yandexuid_ok", .Path="/", .Domain=".yandex.ru"},
                        {.Name="yandexuid_ok2", .Path="/", .Domain="yandex.ru"},
                        {.Name="yandexuid_nopath", .Path="/", .Domain=".yandex.ru"},
                        {.Name="yandexuid_locpath", .Path="/", .Domain=".yandex.ru"},
                        {.Name="yandexuid_locpathdomain", .Path="/", .Domain=".yandex.ru"},
                        {.Name="yandexuid_wrongdomain", .Path="/", .Domain=".yandex.ru"},
                        {.Name="yandexuid_baddomain", .Path="/", .Domain=".yandex.ru"},
                        {.Name="yandexuid_nodomain", .Path="/", .Domain=".yandex.ru"},
                        {.Name="other",},
                    }),
                    .RespLog=" [cpol u:xxx-yyy fail+fix"
                             " yandexuid_nopath=Fix:1!NotGlobalScope,"
                             " yandexuid_locpath=Fix:1!NotGlobalScope,"
                             " yandexuid_locpathdomain=Fix:1!NotGlobalScope,"
                             " yandexuid_wrongdomain=Fix:1!WrongDomain,"
                             " yandexuid_baddomain=Fix:1!BadDomain,"
                             " yandexuid_nodomain=Fix:1!NotGlobalScope,]",
                    .Stats=FixStatsNoGdpr({
                        {"cpol-xxx-modified-resp_summ", 1},
                        {"cpol-xxx-yyy-cookie-fail_summ", 6},
                        {"cpol-xxx-yyy-cookie-fix_summ", 6},
                        {"cpol-xxx-yyy-cookie-pass_summ", 2},
                        {"cpol-xxx-yyy-cookie-skip_summ", 1},
                        {"cpol-xxx-yyy-cookie-total_summ", 9},
                        {"cpol-xxx-yyy-fail_summ", 1},
                        {"cpol-xxx-yyy-fix_summ", 1},
                        {"cpol-xxx-yyy-total_summ", 1},
                    }, MakeCookies().size()),
                },
            }));
        }
    }

    Y_UNIT_TEST(TestPolicyDryRun) {
        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeConfig(false),
                .Host={"blogs.yandex.ru"},
                .SetCookie=MakeCookies(),
            },
            .Out={
                .SetCookie=MakeCookies(),
                .RespLog=" [cpol u:xxx-yyy fail+dryRun"
                         " yandexuid_nopath=Fix:1!NotGlobalScope,"
                         " yandexuid_locpath=Fix:1!NotGlobalScope,"
                         " yandexuid_locpathdomain=Fix:1!NotGlobalScope,"
                         " yandexuid_wrongdomain=Fix:1!WrongDomain,"
                         " yandexuid_baddomain=Fix:1!BadDomain,"
                         " yandexuid_nodomain=Fix:1!NotGlobalScope,]",
                .Stats=FixStatsNoGdpr({
                    {"cpol-xxx-yyy-cookie-fail_summ", 6},
                    {"cpol-xxx-yyy-cookie-fix-dryRun_summ", 6},
                    {"cpol-xxx-yyy-cookie-pass_summ", 2},
                    {"cpol-xxx-yyy-cookie-skip_summ", 1},
                    {"cpol-xxx-yyy-cookie-total_summ", 9},
                    {"cpol-xxx-yyy-fail_summ", 1},
                    {"cpol-xxx-yyy-fix-dryRun_summ", 1},
                    {"cpol-xxx-yyy-total_summ", 1},
                }, MakeCookies().size()),
            },
        }));

        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeConfig(false),
                .Host={"blogs.yandex.ru", "oops.unexpected.me"},
                .SetCookie=MakeCookies(),
            },
            .Out={
                .SetCookie=MakeCookies(),
                .RespLog=" [cpol u:xxx-yyy fail+dryRun"
                         " yandexuid_nopath=Fix:1!NotGlobalScope,"
                         " yandexuid_locpath=Fix:1!NotGlobalScope,"
                         " yandexuid_locpathdomain=Fix:1!NotGlobalScope,"
                         " yandexuid_wrongdomain=Fix:1!WrongDomain,"
                         " yandexuid_baddomain=Fix:1!BadDomain,"
                         " yandexuid_nodomain=Fix:1!NotGlobalScope,]",
                .Stats=FixStatsNoGdpr({
                    {"cpol-xxx-yyy-cookie-fail_summ", 6},
                    {"cpol-xxx-yyy-cookie-fix-dryRun_summ", 6},
                    {"cpol-xxx-yyy-cookie-pass_summ", 2},
                    {"cpol-xxx-yyy-cookie-skip_summ", 1},
                    {"cpol-xxx-yyy-cookie-total_summ", 9},
                    {"cpol-xxx-yyy-fail_summ", 1},
                    {"cpol-xxx-yyy-fix-dryRun_summ", 1},
                    {"cpol-xxx-yyy-total_summ", 1},
                }, MakeCookies().size()),
            },
        }));
    }
}
