#include <balancer/modules/cookie_policy/ut_common/ut_common.h>

#include <balancer/kernel/cookie/gdpr/lifetime.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NModCookiePolicy;

Y_UNIT_TEST_SUITE(TPolicyIsGdprCookieTest) {
    namespace {
        auto MakeConfig(bool fix) {
            return MakeCombinedPolicy(Sprintf(R"=(instance={
                uuid="xxx";
                is_gdpr_cookie={
                    yyy={
                        %s
                    };
                };
            })=", (fix ? "mode='fix';" : "")));
        }
    }

    Y_UNIT_TEST(TestSkipBadHost) {
        TInstant now = TInstant::Seconds(1600092462);

        for (auto&& h : std::initializer_list<TStrVec>{{}, {"localhost"}, {"127.0.0.1"}}) {
            Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
                .In={
                    .Cfg=MakeConfig(true),
                    .Now=now,
                    .Host=h
                },
                .Out={
                    .Stats=FixStatsNoGdpr({
                        {"cpol-xxx-yyy-off_summ", 1},
                        {"cpol-xxx-yyy-total_summ", 1},
                    }, 0),
                },
            }));
        }
    }

    Y_UNIT_TEST(TestSkipNoUpdate) {
        TInstant now = TInstant::Seconds(1600092462);

        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeConfig(true),
                .Now=now,
                .Host={"yandex.ru"},
            },
            .Out={
                .Stats=FixStatsNoGdpr({
                    {"cpol-xxx-yyy-off_summ", 1},
                    {"cpol-xxx-yyy-total_summ", 1},
                }, 0),
            },
        }));

        for (auto&& c : std::initializer_list<TStrVec>{
            {"is_gdpr=0"},
            {"is_gdpr=1"},
        }) {
            Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
                .In={
                    .Cfg=MakeConfig(true),
                    .Now=now,
                    .Host={"yandex.ru"},
                    .Cookie=c,
                },
                .Out={
                    .Cookie=c,
                    .Stats=FixStatsNoGdpr({
                        {"cpol-xxx-IsGdpr-cookie_summ", 1},
                        {"cpol-xxx-yyy-off_summ", 1},
                        {"cpol-xxx-yyy-total_summ", 1},
                    }, 0),
                },
            }));
        }

        for (auto&& c : std::initializer_list<TStrVec>{
            {"is_gdpr=0; is_gdpr=0"},
            {"is_gdpr=0; is_gdpr=1"}, // a conflict
        }) {
            Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
                .In={
                    .Cfg=MakeConfig(true),
                    .Now=now,
                    .Host={"yandex.ru"},
                    .Cookie=c,
                },
                .Out={
                    .Cookie=c,
                    .Stats=FixStatsNoGdpr({
                        {"cpol-xxx-IsGdpr-cookie_summ", 1},
                        {"cpol-xxx-IsGdpr-duplicate_summ", 1},
                        {"cpol-xxx-yyy-off_summ", 1},
                        {"cpol-xxx-yyy-total_summ", 1},
                    }, 0),
                },
            }));
        }

        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeConfig(true),
                .Now=now,
                .Host={"yandex.ru"},
                .Cookie={"is_gdpr=xxx"},
            },
            .Out={
                .Cookie={"is_gdpr=xxx"},
                .Stats=FixStatsNoGdpr({
                    {"cpol-xxx-IsGdpr-cookie_summ", 1},
                    {"cpol-xxx-IsGdpr-badValue_summ", 1},
                    {"cpol-xxx-yyy-off_summ", 1},
                    {"cpol-xxx-yyy-total_summ", 1},
                }, 0),
            },
        }));

        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeConfig(true),
                .Now=now,
                .Host={"yandex.ru"},
                .XYandexEURequest={"1"},
                .Cookie={"is_gdpr=1"},
            },
            .Out={
                .GeoCache={.XYandexEURequest=1},
                .Gdpr={
                    .Value={.IsGdpr=1},
                    .SafeValue={.IsGdpr=1},
                },
                .Cookie={"is_gdpr=1"},
                .ReqLog=" XYandexEURequest=1",
                .Stats=FixStats({
                    {"cpol-xxx-IsGdpr-cookie_summ", 1},
                    {"cpol-xxx-gdpr-geo-1_summ", 1},
                    {"cpol-xxx-gdpr-mod-0_summ", 1},
                    {"cpol-xxx-gdpr-src-XYandexEURequest_summ", 1},
                    {"cpol-xxx-gdpr-safe-geo-1_summ", 1},
                    {"cpol-xxx-gdpr-safe-mod-0_summ", 1},
                    {"cpol-xxx-gdpr-safe-src-XYandexEURequest_summ", 1},
                    {"cpol-xxx-XYandexEURequest_summ", 1},
                    {"cpol-xxx-yyy-off_summ", 1},
                    {"cpol-xxx-yyy-total_summ", 1},
                }, 0),
            },
        }));

        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeConfig(true),
                .Now=now,
                .Host={"yandex.ru"},
                .XYandexEURequest={"1"},
                .Cookie={"is_gdpr=1; is_gdpr=1"},
            },
            .Out={
                .GeoCache={.XYandexEURequest=1},
                .Gdpr={
                    .Value={.IsGdpr=1},
                    .SafeValue={.IsGdpr=1},
                },
                .Cookie={"is_gdpr=1; is_gdpr=1"},
                .ReqLog=" XYandexEURequest=1",
                .Stats=FixStats({
                    {"cpol-xxx-IsGdpr-cookie_summ", 1},
                    {"cpol-xxx-IsGdpr-duplicate_summ", 1},
                    {"cpol-xxx-gdpr-geo-1_summ", 1},
                    {"cpol-xxx-gdpr-mod-0_summ", 1},
                    {"cpol-xxx-gdpr-src-XYandexEURequest_summ", 1},
                    {"cpol-xxx-gdpr-safe-geo-1_summ", 1},
                    {"cpol-xxx-gdpr-safe-mod-0_summ", 1},
                    {"cpol-xxx-gdpr-safe-src-XYandexEURequest_summ", 1},
                    {"cpol-xxx-XYandexEURequest_summ", 1},
                    {"cpol-xxx-yyy-off_summ", 1},
                    {"cpol-xxx-yyy-total_summ", 1},
                }, 0),
            },
        }));

        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeConfig(true),
                .Now=now,
                .Host={"yandex.ru"},
                .XYandexEURequest={"0"},
                .Cookie={"is_gdpr=0"},
            },
            .Out={
                .GeoCache={.XYandexEURequest=0},
                .Gdpr={
                    .Value={.IsGdpr=0},
                    .SafeValue={.IsGdpr=0},
                },
                .Cookie={"is_gdpr=0"},
                .ReqLog=" XYandexEURequest=0",
                .Stats=FixStats({
                    {"cpol-xxx-IsGdpr-cookie_summ", 1},
                    {"cpol-xxx-gdpr-geo-0_summ", 1},
                    {"cpol-xxx-gdpr-mod-0_summ", 1},
                    {"cpol-xxx-gdpr-src-XYandexEURequest_summ", 1},
                    {"cpol-xxx-gdpr-safe-geo-0_summ", 1},
                    {"cpol-xxx-gdpr-safe-mod-0_summ", 1},
                    {"cpol-xxx-gdpr-safe-src-XYandexEURequest_summ", 1},
                    {"cpol-xxx-XYandexEURequest_summ", 1},
                    {"cpol-xxx-yyy-off_summ", 1},
                    {"cpol-xxx-yyy-total_summ", 1},
                }, 0),
            },
        }));

        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeConfig(true),
                .Now=now,
                .Host={"yandex.ru"},
                .XYandexEURequest={"0"},
                .Cookie={"is_gdpr=0; is_gdpr=0"},
            },
            .Out={
                .GeoCache={.XYandexEURequest=0},
                .Gdpr={
                    .Value={.IsGdpr=0},
                    .SafeValue={.IsGdpr=0},
                },
                .Cookie={"is_gdpr=0; is_gdpr=0"},
                .ReqLog=" XYandexEURequest=0",
                .Stats=FixStats({
                    {"cpol-xxx-IsGdpr-cookie_summ", 1},
                    {"cpol-xxx-IsGdpr-duplicate_summ", 1},
                    {"cpol-xxx-gdpr-geo-0_summ", 1},
                    {"cpol-xxx-gdpr-mod-0_summ", 1},
                    {"cpol-xxx-gdpr-src-XYandexEURequest_summ", 1},
                    {"cpol-xxx-gdpr-safe-geo-0_summ", 1},
                    {"cpol-xxx-gdpr-safe-mod-0_summ", 1},
                    {"cpol-xxx-gdpr-safe-src-XYandexEURequest_summ", 1},
                    {"cpol-xxx-XYandexEURequest_summ", 1},
                    {"cpol-xxx-yyy-off_summ", 1},
                    {"cpol-xxx-yyy-total_summ", 1},
                }, 0),
            },
        }));
    }

    Y_UNIT_TEST(TestCreate) {
        TInstant now = TInstant::Seconds(1600092462);
        for (auto [ua, ss] : std::initializer_list<std::pair<TStrVec, ESameSite>>{
            {{}, ESameSite::Undefined},
            {{ToString(FirefoxUA)}, ESameSite::Undefined},
            {{ToString(ChromimUA)}, ESameSite::None},
        }) {
            Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
                .In={
                    .Cfg=MakeConfig(true),
                    .Now=now,
                    .Host={"yandex.ru"},
                    .UserAgent=ua,
                    .XYandexEURequest={"1"},
                },
                .Out={
                    .GeoCache={.XYandexEURequest=1},
                    .Gdpr={
                        .Value={.IsGdpr=1},
                        .SafeValue={.IsGdpr=1},
                    },
                    .Cookie={"is_gdpr=1"},
                    .SetCookie=ToCookie({IsGdprSetCookie(1, now, ".yandex.ru", ss)}),
                    .ReqLog=" XYandexEURequest=1"
                            " [cpol-req u:xxx-yyy fail+fix is_gdpr=Add:1!Create,]",
                    .RespLog=" [cpol u:xxx-yyy fail+fix is_gdpr=Add:1!Create,]",
                    .Stats=FixStats({
                        {"cpol-xxx-gdpr-geo-1_summ", 1},
                        {"cpol-xxx-gdpr-mod-0_summ", 1},
                        {"cpol-xxx-gdpr-src-XYandexEURequest_summ", 1},
                        {"cpol-xxx-gdpr-safe-geo-1_summ", 1},
                        {"cpol-xxx-gdpr-safe-mod-0_summ", 1},
                        {"cpol-xxx-gdpr-safe-src-XYandexEURequest_summ", 1},
                        {"cpol-xxx-modified-req_summ", 1},
                        {"cpol-xxx-modified-resp_summ", 1},
                        {"cpol-xxx-XYandexEURequest_summ", 1},
                        {"cpol-xxx-yyy-cookie-add_summ", 1},
                        {"cpol-xxx-yyy-fail_summ", 1},
                        {"cpol-xxx-yyy-fix_summ", 1},
                        {"cpol-xxx-yyy-req-cookie-add_summ", 1},
                        {"cpol-xxx-yyy-total_summ", 1},
                    }, 0),
                },
            }));
        }
    }

    Y_UNIT_TEST(TestNoCreateOverride) {
        TInstant now = TInstant::Seconds(1600092462);

        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeConfig(false),
                .ModeCtl="{policy_modes={yyy={mode='fix'}}}",
                .Now=now,
                .Host={"yandex.ru"},
            },
            .Out={
                .Stats=FixStatsNoGdpr({
                    {"cpol-xxx-yyy-off_summ", 1},
                    {"cpol-xxx-yyy-total_summ", 1},
                }, 0),
            },
        }));
    }

    Y_UNIT_TEST(TestCreateDryRun) {
        TInstant now = TInstant::Seconds(1600092462);
        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeConfig(false),
                .Now=now,
                .Host={"yandex.ru"},
                .XYandexEURequest={"1"},
            },
            .Out={
                .GeoCache={.XYandexEURequest=1},
                .Gdpr={
                    .Value={.IsGdpr=1},
                    .SafeValue={.IsGdpr=1},
                },
                .ReqLog=" XYandexEURequest=1"
                        " [cpol-req u:xxx-yyy fail+dryRun is_gdpr=Add:1!Create,]",
                .RespLog=" [cpol u:xxx-yyy fail+dryRun is_gdpr=Add:1!Create,]",
                .Stats=FixStats({
                    {"cpol-xxx-gdpr-geo-1_summ", 1},
                    {"cpol-xxx-gdpr-mod-0_summ", 1},
                    {"cpol-xxx-gdpr-src-XYandexEURequest_summ", 1},
                    {"cpol-xxx-gdpr-safe-geo-1_summ", 1},
                    {"cpol-xxx-gdpr-safe-mod-0_summ", 1},
                    {"cpol-xxx-gdpr-safe-src-XYandexEURequest_summ", 1},
                    {"cpol-xxx-XYandexEURequest_summ", 1},
                    {"cpol-xxx-yyy-cookie-add-dryRun_summ", 1},
                    {"cpol-xxx-yyy-fail_summ", 1},
                    {"cpol-xxx-yyy-fix-dryRun_summ", 1},
                    {"cpol-xxx-yyy-req-cookie-add-dryRun_summ", 1},
                    {"cpol-xxx-yyy-total_summ", 1},
                }, 0),
            },
        }));
    }

    Y_UNIT_TEST(TestUpdate) {
        TInstant now = TInstant::Seconds(1600092462);
        for (auto [ua, ss] : std::initializer_list<std::pair<TStrVec, ESameSite>>{
            {{}, ESameSite::Undefined},
            {{ToString(FirefoxUA)}, ESameSite::Undefined},
            {{ToString(ChromimUA)}, ESameSite::None},
        }) {
            Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
                .In={
                    .Cfg=MakeConfig(true),
                    .Now=now,
                    .Host={"yandex.ru"},
                    .UserAgent=ua,
                    .XYandexEURequest={"0"},
                    .Cookie={"is_gdpr=1"},
                },
                .Out={
                    .GeoCache={.XYandexEURequest=0,},
                    .Cookie={"is_gdpr=0"},
                    .SetCookie=ToCookie({IsGdprSetCookie(0, now, ".yandex.ru", ss)}),
                    .ReqLog=" XYandexEURequest=0"
                            " [cpol-req u:xxx-yyy fail+fix is_gdpr=Fix:1!Update,]",
                    .RespLog=" [cpol u:xxx-yyy fail+fix is_gdpr=Add:1!Update,]",
                    .Stats=FixStats({
                        {"cpol-xxx-IsGdpr-cookie_summ", 1},
                        {"cpol-xxx-gdpr-geo-0_summ", 1},
                        {"cpol-xxx-gdpr-mod-0_summ", 1},
                        {"cpol-xxx-gdpr-src-XYandexEURequest_summ", 1},
                        {"cpol-xxx-gdpr-safe-geo-0_summ", 1},
                        {"cpol-xxx-gdpr-safe-mod-0_summ", 1},
                        {"cpol-xxx-gdpr-safe-src-XYandexEURequest_summ", 1},
                        {"cpol-xxx-modified-req_summ", 1},
                        {"cpol-xxx-modified-resp_summ", 1},
                        {"cpol-xxx-XYandexEURequest_summ", 1},
                        {"cpol-xxx-yyy-cookie-add_summ", 1},
                        {"cpol-xxx-yyy-fail_summ", 1},
                        {"cpol-xxx-yyy-fix_summ", 1},
                        {"cpol-xxx-yyy-req-cookie-fix_summ", 1},
                        {"cpol-xxx-yyy-total_summ", 1},
                    }, 0),
                },
            }));

            Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
                .In={
                    .Cfg=MakeConfig(true),
                    .Now=now,
                    .Host={"yandex.ru"},
                    .UserAgent=ua,
                    .XYandexEURequest={"1"},
                    .Cookie={"is_gdpr=0"},
                },
                .Out={
                    .GeoCache={.XYandexEURequest=1},
                    .Gdpr={
                        .Value={.IsGdpr=1},
                        .SafeValue={.IsGdpr=1},
                    },
                    .Cookie={"is_gdpr=1"},
                    .SetCookie=ToCookie({IsGdprSetCookie(1, now, ".yandex.ru", ss)}),
                    .ReqLog=" XYandexEURequest=1"
                            " [cpol-req u:xxx-yyy fail+fix is_gdpr=Fix:1!Update,]",
                    .RespLog=" [cpol u:xxx-yyy fail+fix is_gdpr=Add:1!Update,]",
                    .Stats=FixStats({
                        {"cpol-xxx-IsGdpr-cookie_summ", 1},
                        {"cpol-xxx-gdpr-geo-1_summ", 1},
                        {"cpol-xxx-gdpr-mod-0_summ", 1},
                        {"cpol-xxx-gdpr-src-XYandexEURequest_summ", 1},
                        {"cpol-xxx-gdpr-safe-geo-1_summ", 1},
                        {"cpol-xxx-gdpr-safe-mod-0_summ", 1},
                        {"cpol-xxx-gdpr-safe-src-XYandexEURequest_summ", 1},
                        {"cpol-xxx-modified-req_summ", 1},
                        {"cpol-xxx-modified-resp_summ", 1},
                        {"cpol-xxx-XYandexEURequest_summ", 1},
                        {"cpol-xxx-yyy-cookie-add_summ", 1},
                        {"cpol-xxx-yyy-fail_summ", 1},
                        {"cpol-xxx-yyy-fix_summ", 1},
                        {"cpol-xxx-yyy-req-cookie-fix_summ", 1},
                        {"cpol-xxx-yyy-total_summ", 1},
                    }, 0),
                },
            }));
        }
    }

    Y_UNIT_TEST(TestUpdateDryRun) {
        TInstant now = TInstant::Seconds(1600092462);
        for (auto [ua, ss] : std::initializer_list<std::pair<TStrVec, ESameSite>>{
            {{}, ESameSite::Undefined},
            {{ToString(FirefoxUA)}, ESameSite::Undefined},
            {{ToString(ChromimUA)}, ESameSite::None},
        }) {
            Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
                .In={
                    .Cfg=MakeConfig(false),
                    .Now=now,
                    .Host={"yandex.ru"},
                    .UserAgent=ua,
                    .XYandexEURequest={"0"},
                    .Cookie={"is_gdpr=1"},
                },
                .Out={
                    .GeoCache={.XYandexEURequest=0},
                    .Cookie={"is_gdpr=1"},
                    .ReqLog=" XYandexEURequest=0"
                            " [cpol-req u:xxx-yyy fail+dryRun is_gdpr=Fix:1!Update,]",
                    .RespLog=" [cpol u:xxx-yyy fail+dryRun is_gdpr=Add:1!Update,]",
                    .Stats=FixStats({
                        {"cpol-xxx-gdpr-geo-0_summ", 1},
                        {"cpol-xxx-gdpr-mod-0_summ", 1},
                        {"cpol-xxx-gdpr-src-XYandexEURequest_summ", 1},
                        {"cpol-xxx-gdpr-safe-geo-0_summ", 1},
                        {"cpol-xxx-gdpr-safe-mod-0_summ", 1},
                        {"cpol-xxx-gdpr-safe-src-XYandexEURequest_summ", 1},
                        {"cpol-xxx-XYandexEURequest_summ", 1},
                        {"cpol-xxx-IsGdpr-cookie_summ", 1},
                        {"cpol-xxx-yyy-cookie-add-dryRun_summ", 1},
                        {"cpol-xxx-yyy-fail_summ", 1},
                        {"cpol-xxx-yyy-fix-dryRun_summ", 1},
                        {"cpol-xxx-yyy-req-cookie-fix-dryRun_summ", 1},
                        {"cpol-xxx-yyy-total_summ", 1},
                    }, 0),
                },
            }));

            Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
                .In={
                    .Cfg=MakeConfig(false),
                    .Now=now,
                    .Host={"yandex.ru"},
                    .UserAgent=ua,
                    .XYandexEURequest={"1"},
                    .Cookie={"is_gdpr=0"},
                },
                .Out={
                    .GeoCache={.XYandexEURequest=1},
                    .Gdpr={
                        .Value={.IsGdpr=1},
                        .SafeValue={.IsGdpr=1},
                    },
                    .Cookie={"is_gdpr=0"},
                    .ReqLog=" XYandexEURequest=1"
                            " [cpol-req u:xxx-yyy fail+dryRun is_gdpr=Fix:1!Update,]",
                    .RespLog=" [cpol u:xxx-yyy fail+dryRun is_gdpr=Add:1!Update,]",
                    .Stats=FixStats({
                        {"cpol-xxx-gdpr-geo-1_summ", 1},
                        {"cpol-xxx-gdpr-mod-0_summ", 1},
                        {"cpol-xxx-gdpr-src-XYandexEURequest_summ", 1},
                        {"cpol-xxx-gdpr-safe-geo-1_summ", 1},
                        {"cpol-xxx-gdpr-safe-mod-0_summ", 1},
                        {"cpol-xxx-gdpr-safe-src-XYandexEURequest_summ", 1},
                        {"cpol-xxx-XYandexEURequest_summ", 1},
                        {"cpol-xxx-IsGdpr-cookie_summ", 1},
                        {"cpol-xxx-yyy-cookie-add-dryRun_summ", 1},
                        {"cpol-xxx-yyy-fail_summ", 1},
                        {"cpol-xxx-yyy-fix-dryRun_summ", 1},
                        {"cpol-xxx-yyy-req-cookie-fix-dryRun_summ", 1},
                        {"cpol-xxx-yyy-total_summ", 1},
                    }, 0),
                },
            }));
        }
    }

    Y_UNIT_TEST(TestFixBadValue) {
        TInstant now = TInstant::Seconds(1600092462);
        for (auto [ua, ss] : std::initializer_list<std::pair<TStrVec, ESameSite>>{
            {{}, ESameSite::Undefined},
            {{ToString(FirefoxUA)}, ESameSite::Undefined},
            {{ToString(ChromimUA)}, ESameSite::None},
        }) {
            Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
                .In={
                    .Cfg=MakeConfig(true),
                    .Now=now,
                    .Host={"yandex.ru"},
                    .UserAgent=ua,
                    .XYandexEURequest={"0"},
                    .Cookie={"is_gdpr=xxx"},
                },
                .Out={
                    .GeoCache={.XYandexEURequest=0},
                    .Cookie={"is_gdpr=0"},
                    .SetCookie=ToCookie({IsGdprSetCookie(0, now, ".yandex.ru", ss)}),
                    .ReqLog=" XYandexEURequest=0"
                            " [cpol-req u:xxx-yyy fail+fix is_gdpr=Drop:1!BadValue, is_gdpr=Add:1!Create,]",
                    .RespLog=" [cpol u:xxx-yyy fail+fix is_gdpr=Add:1!BadValue,Update,]",
                    .Stats=FixStats({
                        {"cpol-xxx-IsGdpr-badValue_summ", 1},
                        {"cpol-xxx-IsGdpr-cookie_summ", 1},
                        {"cpol-xxx-XYandexEURequest_summ", 1},
                        {"cpol-xxx-gdpr-geo-0_summ", 1},
                        {"cpol-xxx-gdpr-mod-0_summ", 1},
                        {"cpol-xxx-gdpr-src-XYandexEURequest_summ", 1},
                        {"cpol-xxx-gdpr-safe-geo-0_summ", 1},
                        {"cpol-xxx-gdpr-safe-mod-0_summ", 1},
                        {"cpol-xxx-gdpr-safe-src-XYandexEURequest_summ", 1},
                        {"cpol-xxx-modified-req_summ", 1},
                        {"cpol-xxx-modified-resp_summ", 1},
                        {"cpol-xxx-yyy-cookie-add_summ", 1},
                        {"cpol-xxx-yyy-fail_summ", 1},
                        {"cpol-xxx-yyy-fix_summ", 1},
                        {"cpol-xxx-yyy-req-cookie-add_summ", 1},
                        {"cpol-xxx-yyy-req-cookie-drop_summ", 1},
                        {"cpol-xxx-yyy-total_summ", 1},
                    }, 0),
                },
            }));
        }
    }

    Y_UNIT_TEST(TestFixBadValueDryRun) {
        TInstant now = TInstant::Seconds(1600092462);
        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeConfig(false),
                .Now=now,
                .Host={"yandex.ru"},
                .XYandexEURequest={"0"},
                .Cookie={"is_gdpr=xxx"},
            },
            .Out={
                .GeoCache={.XYandexEURequest=0},
                .Cookie={"is_gdpr=xxx"},
                .ReqLog=" XYandexEURequest=0"
                        " [cpol-req u:xxx-yyy fail+dryRun is_gdpr=Drop:1!BadValue, is_gdpr=Add:1!Create,]",
                .RespLog=" [cpol u:xxx-yyy fail+dryRun is_gdpr=Add:1!BadValue,Update,]",
                .Stats=FixStats({
                    {"cpol-xxx-IsGdpr-badValue_summ", 1},
                    {"cpol-xxx-IsGdpr-cookie_summ", 1},
                    {"cpol-xxx-XYandexEURequest_summ", 1},
                    {"cpol-xxx-gdpr-geo-0_summ", 1},
                    {"cpol-xxx-gdpr-mod-0_summ", 1},
                    {"cpol-xxx-gdpr-src-XYandexEURequest_summ", 1},
                    {"cpol-xxx-gdpr-safe-geo-0_summ", 1},
                    {"cpol-xxx-gdpr-safe-mod-0_summ", 1},
                    {"cpol-xxx-gdpr-safe-src-XYandexEURequest_summ", 1},
                    {"cpol-xxx-yyy-cookie-add-dryRun_summ", 1},
                    {"cpol-xxx-yyy-fail_summ", 1},
                    {"cpol-xxx-yyy-fix-dryRun_summ", 1},
                    {"cpol-xxx-yyy-req-cookie-add-dryRun_summ", 1},
                    {"cpol-xxx-yyy-req-cookie-drop-dryRun_summ", 1},
                    {"cpol-xxx-yyy-total_summ", 1},
                }, 0),
            },
        }));
    }

    Y_UNIT_TEST(TestConflict) {
        // Ignoring conflicts. There should be none anyway.
        TInstant now = TInstant::Seconds(1600092462);

        for (auto [ua, ss] : std::initializer_list<std::pair<TStrVec, ESameSite>>{
            {{}, ESameSite::Undefined},
            {{ToString(FirefoxUA)}, ESameSite::Undefined},
            {{ToString(ChromimUA)}, ESameSite::None},
        }) {
            Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
                .In={
                    .Cfg=MakeConfig(true),
                    .Now=now,
                    .Path="/zen",
                    .Host={"zen.yandex.ru"},
                    .UserAgent=ua,
                    .XYandexEURequest={"0"},
                    .Cookie={"is_gdpr=1; is_gdpr=0"},
                },
                .Out={
                    .GeoCache={.XYandexEURequest=0},
                    .Cookie={"is_gdpr=1; is_gdpr=0"},
                    .ReqLog=" XYandexEURequest=0",
                    .Stats=FixStats({
                        {"cpol-xxx-IsGdpr-cookie_summ", 1},
                        {"cpol-xxx-IsGdpr-duplicate_summ", 1},
                        {"cpol-xxx-XYandexEURequest_summ", 1},
                        {"cpol-xxx-gdpr-geo-0_summ", 1},
                        {"cpol-xxx-gdpr-mod-0_summ", 1},
                        {"cpol-xxx-gdpr-src-XYandexEURequest_summ", 1},
                        {"cpol-xxx-gdpr-safe-geo-0_summ", 1},
                        {"cpol-xxx-gdpr-safe-mod-0_summ", 1},
                        {"cpol-xxx-gdpr-safe-src-XYandexEURequest_summ", 1},
                        {"cpol-xxx-yyy-off_summ", 1},
                        {"cpol-xxx-yyy-total_summ", 1},
                    }, 0),
                },
            }));

            Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
                .In={
                    .Cfg=MakeConfig(true),
                    .Now=now,
                    .Path="/zen",
                    .Host={"zen.yandex.ru"},
                    .UserAgent=ua,
                    .XYandexEURequest={"1"},
                    .Cookie={"is_gdpr=1; is_gdpr=0"},
                },
                .Out={
                    .GeoCache={.XYandexEURequest=1},
                    .Gdpr={
                        .Value={.IsGdpr=1},
                        .SafeValue={.IsGdpr=1},
                    },
                    .Cookie={"is_gdpr=1"},
                    .SetCookie=ToCookie({
                        IsGdprSetCookie(1, now, ".yandex.ru", ss),
                    }),
                    .ReqLog=" XYandexEURequest=1"
                            " [cpol-req u:xxx-yyy fail+fix is_gdpr=Drop:2!Conflict, is_gdpr=Add:1!Create,]",
                    .RespLog=" [cpol u:xxx-yyy fail+fix is_gdpr=Add:1!Conflict,Update,]",
                    .Stats=FixStats({
                        {"cpol-xxx-IsGdpr-cookie_summ", 1},
                        {"cpol-xxx-IsGdpr-duplicate_summ", 1},
                        {"cpol-xxx-XYandexEURequest_summ", 1},
                        {"cpol-xxx-gdpr-geo-1_summ", 1},
                        {"cpol-xxx-gdpr-mod-0_summ", 1},
                        {"cpol-xxx-gdpr-src-XYandexEURequest_summ", 1},
                        {"cpol-xxx-gdpr-safe-geo-1_summ", 1},
                        {"cpol-xxx-gdpr-safe-mod-0_summ", 1},
                        {"cpol-xxx-gdpr-safe-src-XYandexEURequest_summ", 1},
                        {"cpol-xxx-modified-req_summ", 1},
                        {"cpol-xxx-modified-resp_summ", 1},
                        {"cpol-xxx-yyy-cookie-add_summ", 1},
                        {"cpol-xxx-yyy-fail_summ", 1},
                        {"cpol-xxx-yyy-fix_summ", 1},
                        {"cpol-xxx-yyy-req-cookie-add_summ", 1},
                        {"cpol-xxx-yyy-req-cookie-drop_summ", 2},
                        {"cpol-xxx-yyy-total_summ", 1},
                    }, 0),
                },
            }));
        }
    }
}
