#include <balancer/modules/cookie_policy/ut_common/ut_common.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NModCookiePolicy;

Y_UNIT_TEST_SUITE(TPolicyProtectedCookieTest) {
    namespace {
        auto MakeConfig(bool fix) {
            return MakeCombinedPolicy(Sprintf(R"=(instance={
                uuid="xxx";
                protected_cookie={
                    yyy={
                        name_re="Session_id";
                        %s
                    };
                };
            })=", (fix ? "mode='fix'" : "")));
        }

        auto MakeDomainConfig() {
            return MakeCombinedPolicy(R"=(instance={
                uuid="xxx";
                protected_cookie={
                    yyy={
                        name_re="Session_id";
                        mode="fix";
                        domain_filter={
                            include_domains_re="(.*[.])?yandex[.]TLD";
                            exclude_domains_re="(.*[.])?passport[.]yandex[.]TLD";
                        };
                    };
                };
            })=");
        }
    }

    Y_UNIT_TEST(TestPolicy) {
        const auto now = TInstant::Seconds(1596470989);
        const auto cookie = ToCookie({
            {.Name="Session_id"},
            {.Name="Session_id", .MaxAge=0},
            {.Name="Session_id", .Expires=TExpires::Min()},
            {.Name="SomeOther"},
        });

        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeConfig(true),
                .Now=now,
                .SetCookie=cookie,
            },
            .Out={
                .SetCookie=ToCookie({
                    {.Name="SomeOther"},
                }),
                .RespLog=" [cpol u:xxx-yyy fail+fix"
                         " Session_id=Drop:1!Forbidden, Session_id=Drop:1!Forbidden, Session_id=Drop:1!Forbidden,]",
                .Stats=FixStatsNoGdpr({
                    {"cpol-xxx-modified-resp_summ", 1},
                    {"cpol-xxx-yyy-cookie-fail_summ", 3},
                    {"cpol-xxx-yyy-cookie-drop_summ", 3},
                    {"cpol-xxx-yyy-cookie-skip_summ", 1},
                    {"cpol-xxx-yyy-cookie-total_summ", 4},
                    {"cpol-xxx-yyy-fail_summ", 1},
                    {"cpol-xxx-yyy-fix_summ", 1},
                    {"cpol-xxx-yyy-total_summ", 1},
                }, cookie.size()),
            },
        }));
    }

    Y_UNIT_TEST(TestPolicyDryRun) {
        const auto now = TInstant::Seconds(1596470989);
        const auto cookie = ToCookie({
            {.Name="Session_id"},
            {.Name="Session_id", .MaxAge=0},
            {.Name="Session_id", .Expires=TExpires::Min()},
            {.Name="SomeOther"},
        });

        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeConfig(false),
                .Now=now,
                .SetCookie=cookie,
            },
            .Out={
                .SetCookie=cookie,
                .RespLog=" [cpol u:xxx-yyy fail+dryRun"
                         " Session_id=Drop:1!Forbidden, Session_id=Drop:1!Forbidden, Session_id=Drop:1!Forbidden,]",
                .Stats=FixStatsNoGdpr({
                    {"cpol-xxx-yyy-cookie-fail_summ", 3},
                    {"cpol-xxx-yyy-cookie-drop-dryRun_summ", 3},
                    {"cpol-xxx-yyy-cookie-skip_summ", 1},
                    {"cpol-xxx-yyy-cookie-total_summ", 4},
                    {"cpol-xxx-yyy-fail_summ", 1},
                    {"cpol-xxx-yyy-fix-dryRun_summ", 1},
                    {"cpol-xxx-yyy-total_summ", 1},
                }, cookie.size()),
            },
        }));
    }


    Y_UNIT_TEST(TestDomainFilteredSkip) {
        const auto now = TInstant::Seconds(1596470989);
        const auto cookie = ToCookie({
            {.Name="Session_id"},
            {.Name="Session_id", .MaxAge=0},
            {.Name="Session_id", .Expires=TExpires::Min()},
            {.Name="SomeOther"},
        });

        TCPTestResult out = {
            .SetCookie=cookie,
            .Stats=FixStatsNoGdpr({
                {"cpol-xxx-yyy-off_summ", 1},
                {"cpol-xxx-yyy-total_summ", 1},
            }, 4),
        };

        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeDomainConfig(),
                .Now=now,
                .Host={"passport.yandex.ru"},
                .SetCookie=cookie,
            },
            .Out=out,
        }));

        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeDomainConfig(),
                .Now=now,
                .Host={"beru.ru"},
                .SetCookie=cookie,
            },
            .Out=out,
        }));

        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeDomainConfig(),
                .Now=now,
                .SetCookie=cookie,
            },
            .Out=out,
        }));
    }

    Y_UNIT_TEST(TestDomainFilteredMatch) {
        const auto now = TInstant::Seconds(1596470989);
        const auto cookie = ToCookie({
            {.Name="Session_id"},
            {.Name="Session_id", .MaxAge=0},
            {.Name="Session_id", .Expires=TExpires::Min()},
            {.Name="SomeOther"},
        });

        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeDomainConfig(),
                .Now=now,
                .Host={"yandex.com.tr"},
                .SetCookie=cookie,
            },
            .Out={
                .SetCookie=ToCookie({
                    {.Name="SomeOther"},
                }),
                .RespLog=" [cpol u:xxx-yyy fail+fix"
                         " Session_id=Drop:1!Forbidden, Session_id=Drop:1!Forbidden, Session_id=Drop:1!Forbidden,]",
                .Stats=FixStatsNoGdpr({
                {"cpol-xxx-modified-resp_summ", 1},
                    {"cpol-xxx-yyy-cookie-fail_summ", 3},
                    {"cpol-xxx-yyy-cookie-drop_summ", 3},
                    {"cpol-xxx-yyy-cookie-skip_summ", 1},
                    {"cpol-xxx-yyy-cookie-total_summ", 4},
                    {"cpol-xxx-yyy-fail_summ", 1},
                    {"cpol-xxx-yyy-fix_summ", 1},
                    {"cpol-xxx-yyy-total_summ", 1},
                }, cookie.size()),
            },
        }));
    }
}
