#include <balancer/modules/cookie_policy/ut_common/ut_common.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/string/printf.h>

using namespace NModCookiePolicy;

Y_UNIT_TEST_SUITE(TPolicySecureTrueTest) {
    namespace {
        auto MakeSimpleConfig(bool fix) {
            return MakeCombinedPolicy(Sprintf(R"=(instance={
                uuid="xxx";
                secure_true={
                    yyy={
                        name_re="yandexuid.*";
                        %s
                    };
                };
            })=", (fix ? "mode='fix'" : "")));
        }

        auto MakeDomainConfig() {
            return MakeCombinedPolicy(R"=(instance={
                uuid="xxx";
                secure_true={
                    yyy={
                        name_re="yandexuid";
                        mode="fix";
                        domain_filter={
                            include_domains_re="yandex[.]TLD";
                            exclude_domains_re="yandex[.]ru";
                        };
                    };
                };
            })=");
        }

        auto MakeCookie() {
            return ToCookie({
                {.Name="yandexuid_ok", .Secure=true,},
                {.Name="yandexuid_nosec",},
                {.Name="yandexuid_del1", .MaxAge=0,},
                {.Name="yandexuid_del2", .Expires=TExpires::Min(),},
                {.Name="other",},
            });
        }
    }

    Y_UNIT_TEST(TestNotSecure) {
        const auto now = TInstant::Seconds(1596470989);

        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeSimpleConfig(true),
                .Now=now,
                .SetCookie=MakeCookie(),
                .IsSecure=false,
            },
            .Out={
                .SetCookie=MakeCookie(),
                .Stats=FixStatsNoGdpr({
                    {"cpol-xxx-yyy-off_summ", 1},
                    {"cpol-xxx-yyy-total_summ", 1},
                }, MakeCookie().size()),
            },
        }));

    }

    Y_UNIT_TEST(TestSimple) {
        const auto now = TInstant::Seconds(1596470989);

        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeSimpleConfig(true),
                .Now=now,
                .SetCookie=MakeCookie(),
            },
            .Out={
                .SetCookie=ToCookie({
                    {.Name="yandexuid_ok", .Secure=true,},
                    {.Name="yandexuid_nosec", .Secure=true,},
                    {.Name="yandexuid_del1", .MaxAge=0, .Secure=true,},
                    {.Name="yandexuid_del2", .Expires=TExpires::Min(), .Secure=true,},
                    {.Name="other",},
                }),
                .RespLog=" [cpol u:xxx-yyy fail+fix"
                         " yandexuid_nosec=Fix:1!NotSecure, yandexuid_del1=Fix:1!NotSecure, yandexuid_del2=Fix:1!NotSecure,]",
                .Stats=FixStatsNoGdpr({
                    {"cpol-xxx-modified-resp_summ", 1},
                    {"cpol-xxx-yyy-cookie-fail_summ", 3},
                    {"cpol-xxx-yyy-cookie-fix_summ", 3},
                    {"cpol-xxx-yyy-cookie-skip_summ", 1},
                    {"cpol-xxx-yyy-cookie-pass_summ", 1},
                    {"cpol-xxx-yyy-cookie-total_summ", 5},
                    {"cpol-xxx-yyy-fail_summ", 1},
                    {"cpol-xxx-yyy-fix_summ", 1},
                    {"cpol-xxx-yyy-total_summ", 1},
                }, MakeCookie().size()),
            },
        }));
    }

    Y_UNIT_TEST(TestSimpleDryRun) {
        const auto now = TInstant::Seconds(1596470989);

        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeSimpleConfig(false),
                .Now=now,
                .SetCookie=MakeCookie(),
            },
            .Out={
                .SetCookie=MakeCookie(),
                .RespLog=" [cpol u:xxx-yyy fail+dryRun"
                         " yandexuid_nosec=Fix:1!NotSecure, yandexuid_del1=Fix:1!NotSecure, yandexuid_del2=Fix:1!NotSecure,]",
                .Stats=FixStatsNoGdpr({
                    {"cpol-xxx-yyy-cookie-fail_summ", 3},
                    {"cpol-xxx-yyy-cookie-fix-dryRun_summ", 3},
                    {"cpol-xxx-yyy-cookie-skip_summ", 1},
                    {"cpol-xxx-yyy-cookie-pass_summ", 1},
                    {"cpol-xxx-yyy-cookie-total_summ", 5},
                    {"cpol-xxx-yyy-fail_summ", 1},
                    {"cpol-xxx-yyy-fix-dryRun_summ", 1},
                    {"cpol-xxx-yyy-total_summ", 1},
                }, MakeCookie().size()),
            },
        }));
    }

    Y_UNIT_TEST(TestDomainFilteredSkip) {
        const auto now = TInstant::Seconds(1596470989);
        TCPTestResult out = {
            .SetCookie=ToCookie({{.Name="yandexuid",},}),
            .Stats=FixStatsNoGdpr({
                {"cpol-xxx-yyy-off_summ", 1},
                {"cpol-xxx-yyy-total_summ", 1},
            }, 1),
        };

        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeDomainConfig(),
                .Now=now,
                .Host={"yandex.ru"},
                .SetCookie=ToCookie({{.Name="yandexuid",},}),
            },
            .Out=out,
        }));

        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeDomainConfig(),
                .Now=now,
                .Host={"beru.ru"},
                .SetCookie=ToCookie({{.Name="yandexuid",},}),
            },
            .Out=out,
        }));

        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeDomainConfig(),
                .Now=now,
                .SetCookie=ToCookie({{.Name="yandexuid",},}),
            },
            .Out=out,
        }));
    }

    Y_UNIT_TEST(TestDomainFilteredMatch) {
        const auto now = TInstant::Seconds(1596470989);
        Y_COOKIE_POLICY_TEST_POLICY((TPolicyTestCase{
            .In={
                .Cfg=MakeDomainConfig(),
                .Now=now,
                .Host={"yandex.com.tr"},
                .SetCookie=ToCookie({{.Name="yandexuid",},}),
            },
            .Out={
                .SetCookie=ToCookie({{.Name="yandexuid", .Secure=true,},}),
                .RespLog=" [cpol u:xxx-yyy fail+fix yandexuid=Fix:1!NotSecure,]",
                .Stats=FixStatsNoGdpr({
                    {"cpol-xxx-modified-resp_summ", 1},
                    {"cpol-xxx-yyy-cookie-fail_summ", 1},
                    {"cpol-xxx-yyy-cookie-fix_summ", 1},
                    {"cpol-xxx-yyy-cookie-total_summ", 1},
                    {"cpol-xxx-yyy-fail_summ", 1},
                    {"cpol-xxx-yyy-fix_summ", 1},
                    {"cpol-xxx-yyy-total_summ", 1},
                }, 1),
            },
        }));
    }
}
