# -*- coding: utf-8 -*-
import time

from configs import ExpStaticConfig, ExpStaticNestedConfig
from balancer.test.util.predef.handler.server.http import SimpleConfig
from balancer.test.util.predef import http
from balancer.test.util import asserts


def _check_exp(ctx, items):
    for ip, res in items:
        ctx.create_http_connection(source_address=(ip, 0)).perform_request(http.request.get())
        asserts.header_values(ctx.backend.state.get_request(), 'Y-ExpStatic-Test', [res])


def test_experiment(ctx):
    """
    Проверяем что localhost попал в эксепримент
    127.0.0.1 вместо localhost потому, что в testenv падают тесты из-за путаницы в ipv4/ipv6 ресолва
    """
    rate_file = ctx.manager.fs.create_file('rate_file')
    ctx.manager.fs.rewrite(rate_file, "2000,0.3")
    ctx.start_backend(SimpleConfig())
    ctx.start_balancer(ExpStaticConfig(
        exp_id="2000",
        cont_id="1000",
        slots_count=40,
        rate_file=rate_file
    ))
    _check_exp(ctx, [
        ("127.0.0.1", "no"),
        ("127.0.0.2", "1000,0,24"),
        ("127.0.0.3", "1000,0,95"),
        ("127.0.0.4", "no"),
        ("127.0.0.5", "1000,0,42"),
        ("127.0.0.6", "no"),
        ("127.0.0.7", "2000,0,16"),
        ("127.0.0.8", "no"),
        ("127.0.0.9", "no"),
        ("127.0.0.10", "2000,0,34"),
    ])


def test_experiment_rate_salt(ctx):
    """
    BALANCER-2161
    Проверяем что изменение рейта приводит к изменению соли.
    """
    rate_file = ctx.manager.fs.create_file('rate_file')
    ctx.manager.fs.rewrite(rate_file, "2000,0.4")
    ctx.start_backend(SimpleConfig())
    ctx.start_balancer(ExpStaticConfig(
        exp_id="2000",
        cont_id="1000",
        slots_count=40,
        rate_file=rate_file
    ))
    _check_exp(ctx, [
        ("127.0.0.1", "2000,0,41"),
        ("127.0.0.2", "1000,0,65"),
        ("127.0.0.3", "2000,0,81"),
        ("127.0.0.4", "no"),
        ("127.0.0.5", "2000,0,14"),
        ("127.0.0.6", "2000,0,46"),
        ("127.0.0.7", "2000,0,79"),
        ("127.0.0.8", "2000,0,3"),
        ("127.0.0.9", "no"),
        ("127.0.0.10", "2000,0,17"),
    ])


def test_experiment_with_salt(ctx):
    """
    Проверяем что хэш для localhost изменяется при добавлении соли
    """
    rate_file = ctx.manager.fs.create_file('rate_file')
    ctx.manager.fs.rewrite(rate_file, "2000,0.4")
    ctx.start_backend(SimpleConfig())
    ctx.start_balancer(ExpStaticConfig(
        exp_id="2000",
        cont_id="1000",
        slots_count=40,
        rate_file=rate_file,
        salt="ZZZ"
    ))
    _check_exp(ctx, [
        ("127.0.0.1", "2000,0,14"),
        ("127.0.0.2", "2000,0,99"),
        ("127.0.0.3", "1000,0,30"),
        ("127.0.0.4", "2000,0,82"),
        ("127.0.0.5", "2000,0,8"),
        ("127.0.0.6", "no"),
        ("127.0.0.7", "1000,0,1"),
        ("127.0.0.8", "1000,0,51"),
        ("127.0.0.9", "2000,0,92"),
        ("127.0.0.10", "2000,0,7"),
    ])


def test_incorrect_value(ctx):
    """
    Проверяем что при некорретном значении эксперимент не включается
    """
    rate_file = ctx.manager.fs.create_file('rate_file')
    ctx.manager.fs.rewrite(rate_file, "2000,abc")
    ctx.start_backend(SimpleConfig())
    ctx.start_balancer(ExpStaticConfig(
        exp_id="2000",
        cont_id="1000",
        slots_count=100,
        rate_file=rate_file
    ))
    _check_exp(ctx, [
        ("127.0.0.1", "no"),
        ("127.0.0.2", "no"),
        ("127.0.0.3", "no"),
        ("127.0.0.4", "no"),
        ("127.0.0.5", "no"),
        ("127.0.0.6", "no"),
        ("127.0.0.7", "no"),
        ("127.0.0.8", "no"),
        ("127.0.0.9", "no"),
        ("127.0.0.10", "no"),
    ])


def test_not_in_range(ctx):
    """
    Проверяем что при не попадании в диапазон эксперимент не включается
    """
    rate_file = ctx.manager.fs.create_file('rate_file')
    ctx.manager.fs.rewrite(rate_file, "2000,1.1")
    ctx.start_backend(SimpleConfig())
    ctx.start_balancer(ExpStaticConfig(
        exp_id="2000",
        cont_id="1000",
        slots_count=100,
        rate_file=rate_file
    ))
    _check_exp(ctx, [
        ("127.0.0.1", "no"),
        ("127.0.0.2", "no"),
        ("127.0.0.3", "no"),
        ("127.0.0.4", "no"),
        ("127.0.0.5", "no"),
        ("127.0.0.6", "no"),
        ("127.0.0.7", "no"),
        ("127.0.0.8", "no"),
        ("127.0.0.9", "no"),
        ("127.0.0.10", "no"),
    ])


def test_file_switch(ctx):
    """
    Проверяем что файл отключает эксперименты
    """
    ctx.start_backend(SimpleConfig())
    ctx.start_balancer(ExpStaticConfig(
        exp_id="2000",
        cont_id="1000",
        slots_count=10
    ))
    _check_exp(ctx, [
        ("127.0.0.1", "no"),
        ("127.0.0.2", "no"),
        ("127.0.0.3", "no"),
        ("127.0.0.4", "no"),
        ("127.0.0.5", "no"),
        ("127.0.0.6", "no"),
        ("127.0.0.7", "no"),
        ("127.0.0.8", "no"),
        ("127.0.0.9", "no"),
        ("127.0.0.10", "no"),
    ])


def test_file_switch_runtime(ctx):
    """
    Проверяем что файл отключает эксперименты после изменения файла
    rate обновляется только при получении запроса
    """
    rate_file = ctx.manager.fs.create_file('rate_file')
    ctx.manager.fs.rewrite(rate_file, "2000,0.5")

    ctx.start_backend(SimpleConfig())
    ctx.start_balancer(ExpStaticConfig(
        exp_id="2000",
        cont_id="1000",
        slots_count=10,
        rate_file=rate_file,
        salt="Z"
    ))

    _check_exp(ctx, [
        ("127.0.0.1", "1000,0,49"),
        ("127.0.0.2", "2000,0,59"),
        ("127.0.0.3", "2000,0,40"),
        ("127.0.0.4", "2000,0,59"),
        ("127.0.0.5", "2000,0,40"),
        ("127.0.0.6", "2000,0,18"),
        ("127.0.0.7", "1000,0,2"),
        ("127.0.0.8", "2000,0,43"),
        ("127.0.0.9", "1000,0,46"),
        ("127.0.0.10", "2000,0,12"),
        ("127.0.0.11", "1000,0,86"),
        ("127.0.0.12", "2000,0,98"),
        ("127.0.0.13", "2000,0,32"),
        ("127.0.0.14", "1000,0,83"),
        ("127.0.0.15", "1000,0,82"),
        ("127.0.0.16", "1000,0,96"),
        ("127.0.0.17", "2000,0,13"),
        ("127.0.0.18", "1000,0,98"),
        ("127.0.0.19", "2000,0,68"),
        ("127.0.0.20", "2000,0,26"),
        ("127.0.0.21", "2000,0,75"),
        ("127.0.0.22", "1000,0,81"),
        ("127.0.0.23", "2000,0,34"),
        ("127.0.0.24", "1000,0,88"),
        ("127.0.0.25", "1000,0,11"),
        ("127.0.0.26", "2000,0,73"),
        ("127.0.0.27", "1000,0,97"),
        ("127.0.0.28", "1000,0,44"),
        ("127.0.0.29", "2000,0,40"),
        ("127.0.0.30", "2000,0,70")
    ])

    ctx.manager.fs.rewrite(rate_file, "2000,0.0")
    time.sleep(1.5)

    _check_exp(ctx, [
        ("127.0.0.1", "no"),
        ("127.0.0.2", "no"),
        ("127.0.0.3", "no"),
        ("127.0.0.4", "no"),
        ("127.0.0.5", "no"),
        ("127.0.0.6", "no"),
        ("127.0.0.7", "no"),
        ("127.0.0.8", "no"),
        ("127.0.0.9", "no"),
        ("127.0.0.10", "no"),
        ("127.0.0.11", "no"),
        ("127.0.0.12", "no"),
        ("127.0.0.13", "no"),
        ("127.0.0.14", "no"),
        ("127.0.0.15", "no"),
        ("127.0.0.16", "no"),
        ("127.0.0.17", "no"),
        ("127.0.0.18", "no"),
        ("127.0.0.19", "no"),
        ("127.0.0.20", "no"),
        ("127.0.0.21", "no"),
        ("127.0.0.22", "no"),
        ("127.0.0.23", "no"),
        ("127.0.0.24", "no"),
        ("127.0.0.25", "no"),
        ("127.0.0.26", "no"),
        ("127.0.0.27", "no"),
        ("127.0.0.28", "no"),
        ("127.0.0.29", "no"),
        ("127.0.0.30", "no"),
    ])

    ctx.manager.fs.rewrite(rate_file, "2000,0.5")
    time.sleep(1.5)

    _check_exp(ctx, [
        ("127.0.0.1", "1000,0,49"),
        ("127.0.0.2", "2000,0,59"),
        ("127.0.0.3", "2000,0,40"),
        ("127.0.0.4", "2000,0,59"),
        ("127.0.0.5", "2000,0,40"),
        ("127.0.0.6", "2000,0,18"),
        ("127.0.0.7", "1000,0,2"),
        ("127.0.0.8", "2000,0,43"),
        ("127.0.0.9", "1000,0,46"),
        ("127.0.0.10", "2000,0,12"),
        ("127.0.0.11", "1000,0,86"),
        ("127.0.0.12", "2000,0,98"),
        ("127.0.0.13", "2000,0,32"),
        ("127.0.0.14", "1000,0,83"),
        ("127.0.0.15", "1000,0,82"),
        ("127.0.0.16", "1000,0,96"),
        ("127.0.0.17", "2000,0,13"),
        ("127.0.0.18", "1000,0,98"),
        ("127.0.0.19", "2000,0,68"),
        ("127.0.0.20", "2000,0,26"),
        ("127.0.0.21", "2000,0,75"),
        ("127.0.0.22", "1000,0,81"),
        ("127.0.0.23", "2000,0,34"),
        ("127.0.0.24", "1000,0,88"),
        ("127.0.0.25", "1000,0,11"),
        ("127.0.0.26", "2000,0,73"),
        ("127.0.0.27", "1000,0,97"),
        ("127.0.0.28", "1000,0,44"),
        ("127.0.0.29", "2000,0,40"),
        ("127.0.0.30", "2000,0,70")
    ])

    ctx.manager.fs.rewrite(rate_file, "2000,0.0")
    time.sleep(1.5)

    _check_exp(ctx, [
        ("127.0.0.1", "no"),
        ("127.0.0.2", "no"),
        ("127.0.0.3", "no"),
        ("127.0.0.4", "no"),
        ("127.0.0.5", "no"),
        ("127.0.0.6", "no"),
        ("127.0.0.7", "no"),
        ("127.0.0.8", "no"),
        ("127.0.0.9", "no"),
        ("127.0.0.10", "no"),
        ("127.0.0.11", "no"),
        ("127.0.0.12", "no"),
        ("127.0.0.13", "no"),
        ("127.0.0.14", "no"),
        ("127.0.0.15", "no"),
        ("127.0.0.16", "no"),
        ("127.0.0.17", "no"),
        ("127.0.0.18", "no"),
        ("127.0.0.19", "no"),
        ("127.0.0.20", "no"),
        ("127.0.0.21", "no"),
        ("127.0.0.22", "no"),
        ("127.0.0.23", "no"),
        ("127.0.0.24", "no"),
        ("127.0.0.25", "no"),
        ("127.0.0.26", "no"),
        ("127.0.0.27", "no"),
        ("127.0.0.28", "no"),
        ("127.0.0.29", "no"),
        ("127.0.0.30", "no"),
    ])

    ctx.manager.fs.rewrite(rate_file, "2000,0.5,SALT")
    time.sleep(1.5)
    _check_exp(ctx, [
        ("127.0.0.1", "1000,0,86"),
        ("127.0.0.2", "2000,0,70"),
        ("127.0.0.3", "1000,0,96"),
        ("127.0.0.4", "2000,0,86"),
        ("127.0.0.5", "2000,0,18"),
        ("127.0.0.6", "2000,0,51"),
        ("127.0.0.7", "2000,0,77"),
        ("127.0.0.8", "2000,0,1"),
        ("127.0.0.9", "1000,0,81"),
        ("127.0.0.10", "2000,0,99"),
        ("127.0.0.11", "1000,0,99"),
        ("127.0.0.12", "2000,0,11"),
        ("127.0.0.13", "2000,0,11"),
        ("127.0.0.14", "1000,0,18"),
        ("127.0.0.15", "2000,0,18"),
        ("127.0.0.16", "1000,0,60"),
        ("127.0.0.17", "2000,0,96"),
        ("127.0.0.18", "1000,0,67"),
        ("127.0.0.19", "2000,0,83"),
        ("127.0.0.20", "2000,0,22"),
        ("127.0.0.21", "1000,0,22"),
        ("127.0.0.22", "2000,0,95"),
        ("127.0.0.23", "2000,0,17"),
        ("127.0.0.24", "1000,0,96"),
        ("127.0.0.25", "2000,0,28"),
        ("127.0.0.26", "2000,0,21"),
        ("127.0.0.27", "1000,0,41"),
        ("127.0.0.28", "1000,0,1"),
        ("127.0.0.29", "2000,0,36"),
        ("127.0.0.30", "1000,0,21"),
    ])


def test_file_switch_runtime_shutdown(ctx):
    """
    Проверяем, что при занулении веса (те удалении строки) эксперимент выключается
    """
    rate_file = ctx.manager.fs.create_file('rate_file')
    ctx.manager.fs.rewrite(rate_file, "2000,0.5")

    ctx.start_backend(SimpleConfig())
    ctx.start_balancer(ExpStaticConfig(
        exp_id="2000",
        cont_id="1000",
        slots_count=10,
        rate_file=rate_file,
        salt="Z"
    ))

    _check_exp(ctx, [
        ("127.0.0.1", "1000,0,49"),
        ("127.0.0.2", "2000,0,59"),
        ("127.0.0.3", "2000,0,40"),
        ("127.0.0.4", "2000,0,59"),
        ("127.0.0.5", "2000,0,40"),
        ("127.0.0.6", "2000,0,18"),
        ("127.0.0.7", "1000,0,2"),
        ("127.0.0.8", "2000,0,43"),
        ("127.0.0.9", "1000,0,46"),
        ("127.0.0.10", "2000,0,12"),
    ])

    ctx.manager.fs.rewrite(rate_file, "")
    time.sleep(1.5)

    _check_exp(ctx, [
        ("127.0.0.1", "no"),
        ("127.0.0.2", "no"),
        ("127.0.0.3", "no"),
        ("127.0.0.4", "no"),
        ("127.0.0.5", "no"),
        ("127.0.0.6", "no"),
        ("127.0.0.7", "no"),
        ("127.0.0.8", "no"),
        ("127.0.0.9", "no"),
        ("127.0.0.10", "no"),
    ])


def test_nested_experiment_control(ctx):
    """
    Проверяем что localhost попал в эксперимент и вложенную контрольную группу
    """
    rate_file = ctx.manager.fs.create_file('rate_file')
    ctx.manager.fs.rewrite(rate_file, "2000,0.3\n4000,0.3")
    ctx.start_backend(SimpleConfig())
    ctx.start_balancer(ExpStaticNestedConfig(
        exp_id="2000",
        cont_id="1000",
        exp_id_nested="4000",
        cont_id_nested="3000",
        slots_count=10,
        rate_file=rate_file,
        salt="Z",
    ))
    ctx.create_http_connection(host="127.0.0.1").perform_request(http.request.get())

    req = ctx.backend.state.get_request()
    asserts.header_value(req, 'Y-ExpStatic-Test', '3000,0,25;1000,0,25')
